/*
 *  $Id: echo_client.c,v 1.9 2001/03/16 17:49:44 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * echo_client.c  - main program module
 *
 */

#include "sctp.h"

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>         /* for atoi() under Linux */

#define ECHO_PORT                             7
#define MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES    10
#define MAXIMUM_NUMBER_OF_ASSOCIATIONS        5
#define MAXIMUM_NUMBER_OF_IN_STREAMS         17
#define MAXIMUM_NUMBER_OF_OUT_STREAMS        17
#define SCTP_GENERIC_PAYLOAD_PROTOCOL_ID      0
#define min(x,y)            (x)<(y)?(x):(y)

struct ulp_data {
    int maximumStreamID;
    unsigned int chunkCount;
};

static struct ulp_data ulpData[MAXIMUM_NUMBER_OF_ASSOCIATIONS];
static unsigned char localAddressList[MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES][SCTP_MAX_IP_LEN];
static unsigned char destinationAddress[SCTP_MAX_IP_LEN];

static unsigned short noOfLocalAddresses      = 0;
static unsigned short numberOfInitialPackets  = 1;
static unsigned short chunkLength             = 512;
static unsigned char  tosByte                 = 0x10;  /* IPTOS_LOWDELAY */

static int verbose  = 0;
static int vverbose = 0;

void getArgs(int argc, char **argv)
{
    int c;
    extern char *optarg;
    extern int optind;

    while ((c = getopt(argc, argv, "d:l:n:t:s:vV")) != -1)
    {
        switch (c) {
        case 'd':
            if (strlen(optarg) < SCTP_MAX_IP_LEN) {
                strcpy(destinationAddress, optarg);
            }
            break;
        case 'l':
            chunkLength = min(atoi(optarg), SCTP_MAXIMUM_DATA_LENGTH);
            break;
        case 'n':
            numberOfInitialPackets = atoi(optarg);
            break;
        case 's':
            if ((noOfLocalAddresses < MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES) &&
                (strlen(optarg) < SCTP_MAX_IP_LEN  )) {
                strcpy(localAddressList[noOfLocalAddresses], optarg);
                noOfLocalAddresses++;
            }
            break;  
        case 't':
            tosByte = (unsigned char) atoi(optarg);
            break;
        case 'v':
            verbose = 1;
            break;
        case 'V':
            verbose = 1;
            vverbose = 1;
            break;
        }
    }
}

void dataArriveNotif(unsigned int assocID, unsigned int streamID, unsigned int len,
                     unsigned int protoID, unsigned int unordered, void* ulpDataPtr)
{
    unsigned char chunk[SCTP_MAXIMUM_DATA_LENGTH];
    int length;
 
    if (vverbose) {  
      fprintf(stdout, "%-8u: Data arrived (%u bytes on stream %u, %s)\n",
                      assocID, len, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered");
      fflush(stdout);
    }
    /* read it */
    length = sizeof(chunk);
    sctp_receive(assocID, streamID, chunk, &length);
    /* and send it */
    sctp_send(assocID,
              min(streamID, ((struct ulp_data *) ulpDataPtr)->maximumStreamID),
              chunk, length,
              protoID,
              SCTP_USE_PRIMARY, SCTP_NO_CONTEXT, SCTP_INFINITE_LIFETIME, unordered, SCTP_BUNDLING_DISABLED);
}

void sendFailureNotif(unsigned int assocID,
                      unsigned char *unsent_data, unsigned int dataLength, unsigned int *context, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Send failure\n", assocID);
    fflush(stdout);
  }
}

void networkStatusChangeNotif(unsigned int assocID, short destAddrIndex, unsigned short newState, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    SCTP_PathStatus pathStatus;
    unsigned short pathID;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Network status change: path %u is now %s\n", 
        assocID, destAddrIndex, ((newState == SCTP_PATH_OK) ? "ACTIVE" : "INACTIVE"));
        fflush(stdout);
    }
    
    /* if the primary path has become inactive */
    if ((newState == SCTP_PATH_UNREACHABLE) &&
        (destAddrIndex == sctp_getPrimary(assocID))) {
        
        /* select a new one */ /* should we have a sctp_get_primary()? */
        sctp_getAssocStatus(assocID, &assocStatus);
        for (pathID=0; pathID < assocStatus.numberOfAddresses; pathID++){
            sctp_getPathStatus(assocID, pathID, &pathStatus);
            if (pathStatus.state == SCTP_PATH_OK)
                break;
        }
        
        /* and use it */
        if (pathID < assocStatus.numberOfAddresses) {
            sctp_setPrimary(assocID, pathID);
        }
    }
}

void* communicationUpNotif(unsigned int assocID, unsigned short status,
                           int noOfDestinations,
                           unsigned short noOfInStreams, unsigned short noOfOutStreams, void* dummy)
{	
    unsigned int index, packetNumber;
    unsigned char chunk[SCTP_MAXIMUM_DATA_LENGTH];
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Communication up (%u paths)\n", assocID, noOfDestinations);
        fflush(stdout);
    }
  
    /* look for a free ULP data */
    for (index=0; index < MAXIMUM_NUMBER_OF_ASSOCIATIONS; index++) {
        if (ulpData[index].maximumStreamID == -1)
            break;
    }
   
    /* if found */
    if (index < MAXIMUM_NUMBER_OF_ASSOCIATIONS) {
        /* use it */
        ulpData[index].maximumStreamID = noOfOutStreams - 1;   
        /* send the initial packets */
        memset(chunk, 0, sizeof(chunk));
        for(packetNumber=1; packetNumber <= numberOfInitialPackets; packetNumber++) {
            sctp_send(assocID,
                      0,
                      chunk, chunkLength,
                      SCTP_GENERIC_PAYLOAD_PROTOCOL_ID,
                      SCTP_USE_PRIMARY, SCTP_NO_CONTEXT, SCTP_INFINITE_LIFETIME,
                      SCTP_ORDERED_DELIVERY, SCTP_BUNDLING_DISABLED);
        }
        return &ulpData[index];       
    } else {
        /* abort assoc due to lack of resources */
        sctp_abort(assocID);
        return NULL;
    }
}

void communicationLostNotif(unsigned int assocID, unsigned short status, void* ulpDataPtr)
{	
    unsigned char buffer[SCTP_MAXIMUM_DATA_LENGTH];
    unsigned int bufferLength;
    unsigned short streamID, streamSN;
    unsigned int protoID;
    
    if (verbose) {
        fprintf(stdout, "%-8x: Communication lost (status %u)\n", assocID, status);
        fflush(stdout);
    }
  
    /* retrieve data */
    bufferLength = sizeof(buffer);
    while (sctp_receiveUnsent(assocID, buffer, &bufferLength,           
                               &streamID, &streamSN, &protoID) > 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
    
    bufferLength = sizeof(buffer);
    while (sctp_receiveUnacked(assocID, buffer, &bufferLength,
			       &streamID, &streamSN, &protoID) > 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
                      
    /* free ULP data */
    ((struct ulp_data *) ulpDataPtr)->maximumStreamID = -1;
    
    /* delete the association */
    sctp_deleteAssociation(assocID);
}

void communicationErrorNotif(unsigned int assocID, unsigned short status, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Communication error (status %u)\n", assocID, status);
    fflush(stdout);
  }
}

void restartNotif(unsigned int assocID, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Restart\n", assocID);
        fflush(stdout);
    }
    sctp_getAssocStatus(assocID, &assocStatus);
    /* update ULP data */
    ((struct ulp_data *) ulpDataPtr)->maximumStreamID = assocStatus.outStreams - 1;
}

void shutdownCompleteNotif(unsigned int assocID, void* ulpDataPtr)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Shutdown complete\n", assocID);
    fflush(stdout);
  }
  /* free ULP data */
  ((struct ulp_data *) ulpDataPtr)->maximumStreamID = -1;
  sctp_deleteAssociation(assocID);

}

int main(int argc, char **argv)
{
    unsigned short sctpInstance;
    SCTP_ulpCallbacks echoUlp;
    SCTP_InstanceParameters instanceParameters;
    unsigned int index;
    
    /* initialize ULP data */
    for (index=0; index < MAXIMUM_NUMBER_OF_ASSOCIATIONS; index++) {
        ulpData[index].maximumStreamID = -1;
    }

    /* initialize the echo_ulp variable */
    echoUlp.dataArriveNotif          = &dataArriveNotif;
    echoUlp.sendFailureNotif         = &sendFailureNotif;
    echoUlp.networkStatusChangeNotif = &networkStatusChangeNotif;
    echoUlp.communicationUpNotif     = &communicationUpNotif;
    echoUlp.communicationLostNotif   = &communicationLostNotif;
    echoUlp.communicationErrorNotif  = &communicationErrorNotif;
    echoUlp.restartNotif             = &restartNotif;
    echoUlp.shutdownCompleteNotif    = &shutdownCompleteNotif;

    /* handle all command line options */
    getArgs(argc, argv);

    /* set up the "server" */
    sctpInstance = sctp_registerInstance(ECHO_PORT,
					  MAXIMUM_NUMBER_OF_IN_STREAMS,
					  MAXIMUM_NUMBER_OF_OUT_STREAMS,
					  noOfLocalAddresses, localAddressList,
					  echoUlp);
    /* set the TOS field */                  
    sctp_getAssocDefaults(sctpInstance, &instanceParameters);
    instanceParameters.ipTos=tosByte;
    sctp_setAssocDefaults(sctpInstance, &instanceParameters);
    
    /* start the client */
    sctp_associate(sctpInstance, 1, destinationAddress, ECHO_PORT, &ulpData[0]);
    /* run the event handler forever */
    while (sctp_getEvents() >= 0) {
        sctp_eventLoop();
    };

    /* this will never be reached */
    return 0;
}




