/*
 *  $Id: SCTP-control.c,v 1.12 2001/03/16 13:44:26 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This module implements SCTP-control.
 *          SCTP-control controls the setup and shutdown of an association.
 *          For this purpose it receives primitives from the ULP (via message-distribution)
 *          and the peer (via (de-)bundling). In response to this input-signals, SCTP-Control
 *          sends control-primitives to the ULP and the peer. It also stores the state of
 *          an association.
 *
 * function prefixes: scu_ for  primitives originating from the ULP
 *                    scr_ for primitives originating from the peer
 *                    sci_ for SCTP-internal calls
 *
 * Remarks: Host and network byte order (HBO and NBO):
 *          In this module all data are data are HBO, except the IP-Addresses and message strings.
 *          Thus everything entered into a message string must be converted to NBO and
 *          everthing extracted from a message string must be converted to HBO (except the
 *          IP-addresses).
 *
 */


/* TODO : check that in SHUTDOWN states we reset the retransmission counters, any time we
            get a valid datagram from our peer. This must also be done for SACKs, DATA, etc.
            we get from peer !
 */


#include "SCTP-control.h"

#include "distribution.h"
#include "bundling.h"
#include "adaptation.h"
#include "pathmanagement.h"
#include "reltransfer.h"
#include "recvctrl.h"
#include "chunkHandler.h"
#include "flowcontrol.h"
#include "streamengine.h"

#ifdef HAVE_STRINGS_H
    #include <strings.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include <sctp.h>






/** @name SCTP State Machine Controller 
 
  \hline
   Used function prefixes:
   \begin{itemize}
    \item scu_ for  primitives originating from the ULP
    \item scr_ for primitives originating from the peer
    \item sci_ for SCTP-internal calls
    \end{itemize}
 */

//@{

/** macro to return the minimum of two values */
#define min(x,y)            (x)<(y)?(x):(y)

/******************** Typedef *********************************************************************/

/**
  SCTP-control structure. Stores also the current state of the state-machine.
 */
typedef struct SCTP_CONTROLDATA
{
    //@{
    /** the state of this state machine */
    guint32 association_state;
    /** stores timer-ID of init/cookie-timer, used to stop this timer */
    TimerID initTimer;
    ///
    unsigned int initTimerDuration;
    ///  stores the association id (==tag) of this association
    unsigned int associationID;
    /** Counter for init and cookie retransmissions */
    short initRetransCounter;
    /** pointer to the init chunk data structure (for retransmissions) */
    SCTP_init *initChunk;
    /** pointer to the cookie chunk data structure (for retransmissions) */
    SCTP_cookie_echo *cookieChunk;
    /** my tie tag for cross initialization and other sick cases */
    guint32 local_tie_tag;
    /** peer's tie tag for cross initialization and other sick cases */
    guint32 peer_tie_tag;
    /// we store these here, too. Maybe better be stored with StreamEngine ?
    unsigned short NumberOfOutStreams;
    /// we store these here, too. Maybe better be stored with StreamEngine ?
    unsigned short NumberOfInStreams;
    /** value for maximum retransmissions per association */
    int assocMaxRetransmissions;
    /** value for maximum initial retransmissions per association */
    int assocMaxInitRetransmissions;
    /** value for the current cookie lifetime */
    int cookieLifeTime;
    /** the sctp instance */
    void * instance;
    //@}
} SCTP_controlData;

/* -------------------- Declarations -------------------------------------------------------------*/

/// pointer to the current controller structure. Only set when association exists.
static SCTP_controlData *localData;


/* ------------------ Function Implementations ---------------------------------------------------*/

/*------------------- Internal Functions ---------------------------------------------------------*/

/*------------------- Functions called by adaption layer -----------------------------------------*/

/**
 * Defines the callback function that is called when an (INIT, COOKIE, SHUTDOWN etc.) timer expires.
 * @param timerID               ID of timer
 * @param associationIDvoid     pointer to param1, here to an Association ID value, it may be used
 *                              to identify the association, to which the timer function belongs
 * @param unused                pointer to param2 - timers have two params, by default. Not needed here.
 */
static void sci_timer_expired(TimerID timerID, void *associationIDvoid, void *unused)
{
    unsigned int state;
    boolean initFailed;
    ChunkID shutdownCID;
    ChunkID shutdown_complete_CID;
    guint primary;


    if (mdi_setAssociationData(*((unsigned int *) associationIDvoid))) {
        /* error log: expired timer refers to a non existent association. */
        error_logi(ERROR_MAJOR,
                   "init timer expired but association does not exist",
                   (unsigned int *) associationIDvoid);
        return;
    }


    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log: association exist, but has no SCTP-control ? */
        error_log(ERROR_MAJOR, "Association without SCTP-control");
        return;
    }

    state = localData->association_state;
    initFailed = FALSE;

    primary = pm_readPrimaryPath();

    switch (state) {
    case COOKIE_WAIT:

        event_log(EXTERNAL_EVENT, "init timer expired in state COOKIE_WAIT");

        if (localData->initRetransCounter <= localData->assocMaxInitRetransmissions) {
            /* increase retransmissission-counter, resend init and restart init-timer */
            localData->initRetransCounter++;
            bu_put_Ctrl_Chunk((SCTP_simple_chunk *) localData->initChunk);
            bu_sendAllChunks(NULL);
            /* restart init timer after timer backoff */
            localData->initTimerDuration = min(localData->initTimerDuration * 2, RTO_MAX);
            event_logi(INTERNAL_EVENT_0, "init timer backedoff %d msecs",
                       localData->initTimerDuration);
            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);
        } else {
            /* log error to log-file */
            event_log(EXTERNAL_EVENT,
                      "init retransmission counter exeeded threshold in state COOKIE_WAIT");
            /* report error to ULP tbd: status */
            mdi_communicationLostNotif(SCTP_COMM_LOST_EXCEEDED_RETRANSMISSIONS);
            /* free memory for initChunk */
            free(localData->initChunk);
            localData->initChunk = NULL;
            /* delete this association */
            initFailed = TRUE;
        }
        break;

    case COOKIE_ECHOED:

        event_log(EXTERNAL_EVENT, "cookie timer expired in state COOKIE_ECHOED");

        if (localData->initRetransCounter <= localData->assocMaxInitRetransmissions) {
            /* increase retransmissission-counter, resend init and restart init-timer */
            localData->initRetransCounter++;
            bu_put_Ctrl_Chunk((SCTP_simple_chunk *) localData->cookieChunk);
            bu_sendAllChunks(NULL);
            /* restart cookie timer after timer backoff */
            localData->initTimerDuration = min(localData->initTimerDuration * 2, RTO_MAX);
            event_logi(INTERNAL_EVENT_0, "cookie timer backedoff %d msecs",
                       localData->initTimerDuration);
            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);
        } else {
            /* log error to log-file */
            event_log(EXTERNAL_EVENT,
                      "init retransmission counter exeeded threshold; state: COOKIE_ECHOED");
            /* report error to ULP tbd: status */
            mdi_communicationLostNotif(SCTP_COMM_LOST_EXCEEDED_RETRANSMISSIONS);
            /* free memory for cookieChunk */
            free(localData->cookieChunk);
            localData->cookieChunk = NULL;
            /* delete this association */
            initFailed = TRUE;
        }
        break;

    case SHUTDOWNSENT:

        /* some of the variable names are missleading, because they where only used
           for init, but are reused for shutdown after the shutdown timer was introduced
           in the draft. */

        if (localData->initRetransCounter <= localData->assocMaxRetransmissions) {
            /* increase retransmissission-counter */
            localData->initRetransCounter++;

            /* make and send shutdown again, with updated TSN (section 9.2)     */
            shutdownCID = ch_makeShutdown(rxc_read_cummulativeTSNacked());
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownCID));
            bu_sendAllChunks(&primary);
            ch_deleteChunk(shutdownCID);

            /* restart shutdown timer after timer backoff */
            localData->initTimerDuration = min(localData->initTimerDuration * 2, RTO_MAX);
            event_logi(INTERNAL_EVENT_0, "shutdown timer backed off %d msecs",
                       localData->initTimerDuration);
            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);
        } else {
            mdi_communicationLostNotif(SCTP_COMM_LOST_EXCEEDED_RETRANSMISSIONS);
            /* shut down failed, delete current association. */
            initFailed = TRUE;
        }
        break;

    case SHUTDOWNACKSENT:

        /* some of the variable names are missleading, because they where only used
           for init, but are reused for shutdown */

        if (localData->initRetransCounter <= localData->assocMaxRetransmissions) {
            /* increase retransmissission-counter */
            localData->initRetransCounter++;

            /* make and send shutdown_complete again */
            shutdown_complete_CID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NONE);
            bu_put_Ctrl_Chunk(ch_chunkString(shutdown_complete_CID));
            bu_sendAllChunks(&primary);
            ch_deleteChunk(shutdown_complete_CID);

            /* restart shutdown timer after timer backoff */
            localData->initTimerDuration = min(localData->initTimerDuration * 2, RTO_MAX);
            event_logi(INTERNAL_EVENT_0, "shutdown timer backed off %d msecs",
                       localData->initTimerDuration);
            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);
        } else {
            mdi_communicationLostNotif(SCTP_COMM_LOST_EXCEEDED_RETRANSMISSIONS);
            /* shut down failed, delete current association. */
            initFailed = TRUE;
        }
        break;

    default:
        /* error log */
        error_logi(ERROR_MAJOR, "unexpected event: timer expired in state %02d", state);
        break;
    }

    localData = NULL;
    if (initFailed)
        mdi_deleteCurrentAssociation();
    else
        mdi_clearAssociationData(*((unsigned int *) associationIDvoid));
}


/*------------------- Functions called by the ULP via message-distribution -----------------------*/

/**
 * This function is called to initiate the setup an association.
 *
 * The local tag and the initial TSN are randomly generated.
 * Together with the parameters of the function, they are used to create the init-message.
 * This data are also stored in a newly created association-record.
 *
 * @param noOfOutStreams        number of send streams.
 * @param noOfInStreams         number of receive streams.
 */
void scu_associate(unsigned short noOfOutStreams, unsigned short noOfInStreams)
{
    guint32 state;
    guint16 nlAddresses;
    union sockunion lAddresses[MAX_NUM_ADDRESSES];
    ChunkID initCID;

    /* ULP has called sctp_associate at distribution.
       Distribution has allready allocated the association data and partially initialized */

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        error_log(ERROR_MAJOR, "read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case CLOSED:
        event_log(EXTERNAL_EVENT, "event: scu_assocatiate in state CLOSED");
        /* create init chunk and write data to it -- take AssocID as tag !!! */
        initCID = ch_makeInit(mdi_readAssociationID(),
                              mdi_getDefaultMyRwnd(),
                              noOfOutStreams, noOfInStreams, mdi_generateStartTSN());

        /* store the number of streams */
        localData->NumberOfOutStreams = noOfOutStreams;
        localData->NumberOfInStreams = noOfInStreams;

        /* enter enter local addresses to message */
        mdi_readLocalAddresses(lAddresses, &nlAddresses);

#ifdef HAVE_IPV6
        ch_enterSupportedAddressTypes(initCID, TRUE, TRUE, FALSE);
#else
        ch_enterSupportedAddressTypes(initCID, TRUE, FALSE, FALSE);
#endif


        if (nlAddresses > 0)
            ch_enterIPaddresses(initCID, lAddresses, nlAddresses);

        localData->initChunk = (SCTP_init *) ch_chunkString(initCID);
        ch_forgetChunk(initCID);
        /* send init chunk */
        bu_put_Ctrl_Chunk((SCTP_simple_chunk *) localData->initChunk);
        bu_sendAllChunks(NULL);

        localData->cookieChunk = NULL;
        localData->local_tie_tag = 0;
        localData->peer_tie_tag = 0;

        /* start init timer */
        localData->initTimerDuration = pm_readRTO(pm_readPrimaryPath());
        localData->initTimer = sctp_startTimer(localData->initTimerDuration,
                                               &sci_timer_expired,
                                               (void *) &localData->associationID, NULL);

        state = COOKIE_WAIT;
        break;
    default:
        error_logi(EXTERNAL_EVENT_X, "Erroneous Event : scu_associate called in state %u", state);
        break;
    }

    localData->association_state = state;
    localData = NULL;
}



/**
 * function initiates the shutdown of this association.
 */
void scu_shutdown()
{
    guint32 state;
    ChunkID shutdownCID;
    boolean readyForShutdown;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case ESTABLISHED:
        event_log(EXTERNAL_EVENT, "event: scu_shutdown in state ESTABLISHED");

        /* disable heartbeat */
        pm_disableAllHB();

        /* stop reliable transfer and read its state */
        readyForShutdown = (rtx_readNumberOfUnackedChunks() == 0) &&
            (fc_readNumberOfQueuedChunks() == 0);

        if (readyForShutdown) {
            /* make and send shutdown */
            shutdownCID = ch_makeShutdown(rxc_read_cummulativeTSNacked());
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownCID));
            bu_sendAllChunks(NULL);
            ch_deleteChunk(shutdownCID);

            /* start shutdown timer */
            localData->initTimerDuration = pm_readRTO(pm_readPrimaryPath());
            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);

            localData->initRetransCounter = 0;

            /* receive control must acknoweledge every datachunk at once after the shutdown
               was sent. */
            rxc_send_sack_everytime();

            state = SHUTDOWNSENT;
        } else {
            /* shutdown in progress info to reliable transfer, this stopps data transmission */
            rtx_shutdown();
            /* wait for sci_allChunksAcked from reliable transfer */
            state = SHUTDOWNPENDING;
        }
        localData->association_state = state;
        localData = NULL;

        break;
    case CLOSED:
    case COOKIE_WAIT:
    case COOKIE_ECHOED:        /* Siemens convention: ULP can not send datachunks
                                   until it has received the communication up. */
        event_logi(EXTERNAL_EVENT, "event: scu_shutdown in state %02d --> aborting", state);
        scu_abort();
        break;
    case SHUTDOWNSENT:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNPENDING:
    case SHUTDOWNACKSENT:
        /* ignore, keep on waiting for completion of the running shutdown */
        event_logi(EXTERNAL_EVENT, "event: scu_shutdown in state %", state);
        localData = NULL;
        break;
    default:
        /* error logging */
        event_log(EXTERNAL_EVENT_X, "unexpected event: scu_shutdown");
        localData = NULL;
        break;
    }
}



/**
 * this function aborts this association.
 */
void scu_abort()
{
    guint32 state;
    ChunkID abortCID;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case CLOSED:
        event_log(EXTERNAL_EVENT, "event: scu_abort in state CLOSED");
        /* delete all data of this association */
        mdi_deleteCurrentAssociation();

        break;

    case COOKIE_WAIT:
    case COOKIE_ECHOED:
    case SHUTDOWNSENT:
    case SHUTDOWNACKSENT:
        event_logi(EXTERNAL_EVENT, "event: scu_abort in state %2d --> send abort", state);

        /* make and send abort message */
        abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NONE);

        bu_put_Ctrl_Chunk(ch_chunkString(abortCID));
        bu_sendAllChunks(NULL);
				bu_unlock_sender(NULL);
        /* free abort chunk */
        ch_deleteChunk(abortCID);
        /* stop init timer */
        sctp_stopTimer(localData->initTimer);
        /* delete all data of this association */
        mdi_communicationLostNotif(SCTP_COMM_LOST_ABORTED);
        mdi_deleteCurrentAssociation();

        break;
    case ESTABLISHED:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:

        event_logi(EXTERNAL_EVENT, "event: scu_abort in state %02d --> send abort", state);

        /* make and send abort message */
        abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NONE);

        bu_put_Ctrl_Chunk(ch_chunkString(abortCID));
        bu_sendAllChunks(NULL);
				bu_unlock_sender(NULL);
        /* free abort chunk */
        ch_deleteChunk(abortCID);
        /* delete all data of this association */
        mdi_communicationLostNotif(SCTP_COMM_LOST_ABORTED);
        mdi_deleteCurrentAssociation();

        break;
    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scu_abort in state %02d: unexpected event", state);
        break;
    }
}


/*------------------- Functions called by the (de-)bundling for received control chunks ----------*/

/**
 * scr_init is called by bundling when a init message is received from the peer.
 * an InitAck may be returned, alongside with a cookie chunk variable parameter.
 * The following data are created and included in the init acknowledgement:
 * a COOKIE parameter.
 * @param init  pointer to the received init-chunk (including optional parameters)
 */
gboolean scr_init(SCTP_init * init)
{
    /*  this function does not expect any data allocated for the new association,
       but if there are, implementation will act according to section 5.2.1 (simultaneous
       initialization) and section 5.2.2 (duplicate initialization)
     */

    unsigned int state;
    guint16 nlAddresses;
    union sockunion lAddresses[MAX_NUM_ADDRESSES];
    guint16 nrAddresses;
    union sockunion rAddresses[MAX_NUM_ADDRESSES];
    ChunkID initCID;
    ChunkID initCID_local;
    ChunkID initAckCID;
    ChunkID abortCID;
    ChunkID shutdownAckCID;
    unsigned short inbound_streams;
    gboolean removed_association = FALSE;
    guint primary;

    event_log(EXTERNAL_EVENT, "event: scr_init received");

    initCID = ch_makeChunk((SCTP_simple_chunk *) init);

    if (ch_chunkType(initCID) != CHUNK_INIT) {
        /* error logging */
        ch_forgetChunk(initCID);
        error_log(ERROR_MAJOR, "scr_init: wrong chunk type");
        return removed_association;
    }

    if (ch_noOutStreams(initCID) == 0 || ch_noInStreams(initCID) == 0) {
        event_log(EXTERNAL_EVENT, "event: received init with zero number of streams");

        /* make and send abort message */
        abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NONE);
        bu_put_Ctrl_Chunk(ch_chunkString(abortCID));
        bu_sendAllChunks(NULL);
        /* free abort chunk */
        ch_deleteChunk(abortCID);
        /* delete all data of this association */
        if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) != NULL) {
            mdi_communicationLostNotif(SCTP_COMM_LOST_ZERO_STREAMS);
            mdi_deleteCurrentAssociation();
            removed_association = TRUE;
        }
        return removed_association;
    }

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* DO_5_1_B_INIT : Normal case, no association exists yet */
        /* save a-sides init-tag from init-chunk to be used as a verification tag of the sctp-
           message carrying the initAck (required since no association is created). */
        mdi_writeLastInitiateTag(ch_initiateTag(initCID));

        /* Limit the number of sendstreams a-side requests to the max. number of input streams
           this z-side is willing to accept.
         */
        inbound_streams = min(ch_noOutStreams(initCID), mdi_readLocalInStreams());
        /* fire back an InitAck with a Cookie */
        initAckCID = ch_makeInitAck(mdi_generateLocalTag(),
                                    mdi_getDefaultMyRwnd(),
                                    ch_noInStreams(initCID),
                                    inbound_streams, mdi_generateStartTSN());

        /* enter variable length params initAck */
        mdi_readLocalAddresses(lAddresses, &nlAddresses);
        /* enter local addresses into initAck */
        if (nlAddresses > 0)
            ch_enterIPaddresses(initAckCID, lAddresses, nlAddresses);

        /* retreive a-side source addresses from message */
        nrAddresses = ch_IPaddresses(initCID, rAddresses);

        /* TODO: IPv6 treatment */
        /* append cookie to InitAck Chunk */
        ch_enterCookieVLP(initAckCID,
                          ch_initFixed(initCID), ch_initFixed(initAckCID),
                          ch_cookieLifeTime(initCID), 0, /* tie tags are both zero */
                          0, lAddresses, nlAddresses, rAddresses, nrAddresses);

        /* send initAck */
        bu_put_Ctrl_Chunk(ch_chunkString(initAckCID));
        bu_sendAllChunks(NULL);
        ch_deleteChunk(initAckCID);
        event_log(INTERNAL_EVENT_1, "event: initAck sent");
    } else {
        /* save a-sides init-tag from init-chunk to be used as a verification tag of the sctp-
           message carrying the initAck (required since peer may have changed the verification
           tag).
           mdi_writeLastInitiateTag(ch_initiateTag(initCID)); */

        state = localData->association_state;
        event_logi(EXTERNAL_EVENT, "scr_init: received INIT chunk in state %02u", state);
        primary = pm_readPrimaryPath();

        switch (state) {
            /* see section 5.2.1 */
        case COOKIE_WAIT:
            if ((localData->local_tie_tag != 0)
                || (localData->peer_tie_tag != 0))
                error_logii(ERROR_MAJOR,
                            "Tie tags NOT zero in COOKIE_WAIT, but %u and %u",
                            localData->local_tie_tag, localData->peer_tie_tag);

        case COOKIE_ECHOED:
            if ((state == COOKIE_ECHOED)
                && ((localData->local_tie_tag == 0)
                    || (localData->peer_tie_tag == 0)))
                error_logii(ERROR_MAJOR,
                            "Tie tags zero in COOKIE_ECHOED, local: %u, peer: %u",
                            localData->local_tie_tag, localData->peer_tie_tag);

            /* save remote  tag ?
               mdi_writeLastInitiateTag(ch_initiateTag(initCID)); */
            inbound_streams = min(ch_noOutStreams(initCID), mdi_readLocalInStreams( ));

            /* Set length of chunk to HBO !! */
            initCID_local = ch_makeChunk((SCTP_simple_chunk *) localData->initChunk);
            /* section 5.2.1 : take original parameters from first INIT chunk */
            initAckCID = ch_makeInitAck(ch_initiateTag(initCID_local),
                                        ch_receiverWindow(initCID_local), ch_noInStreams(initCID), /* peers inbound are MY outbound */
                                        inbound_streams, ch_initialTSN(initCID_local));

            /* reset length field again to NBO...and remove reference */
            ch_chunkString(initCID_local);
            ch_forgetChunk(initCID_local);

            /* the initAck (and consequently the Cookie) will contain my assocID as my local
               tag, and the peers tag from the init we got here */
            mdi_readLocalAddresses(lAddresses, &nlAddresses);
            /* enter local addresses into initAck */
            if (nlAddresses > 0)
                ch_enterIPaddresses(initAckCID, lAddresses, nlAddresses);
            /* retreive a-side source addresses from message */
            nrAddresses = ch_IPaddresses(initCID, rAddresses);
            /* TODO: IPv6 treatment */
            ch_enterCookieVLP(initAckCID,
                              ch_initFixed(initCID),
                              ch_initFixed(initAckCID), ch_cookieLifeTime(initCID), localData->local_tie_tag, /* tie tags may be zero OR populated here */
                              localData->peer_tie_tag, lAddresses, nlAddresses,
                              rAddresses, nrAddresses);

            /* send initAck */
            bu_put_Ctrl_Chunk(ch_chunkString(initAckCID));
            bu_sendAllChunks(NULL);
            ch_deleteChunk(initAckCID);
            event_logi(INTERNAL_EVENT_1, "event: initAck sent in state %u", state);
            break;

            /* see section 5.2.2 */
        case ESTABLISHED:
        case SHUTDOWNPENDING:
        case SHUTDOWNRECEIVED:
        case SHUTDOWNSENT:
            if ((localData->local_tie_tag == 0)
                || (localData->peer_tie_tag == 0))
                error_logiii(ERROR_MAJOR,
                             "Tie tags zero in state %u, local: %u, peer: %u",
                             state, localData->local_tie_tag, localData->peer_tie_tag);

            inbound_streams = min(ch_noOutStreams(initCID), mdi_readLocalInStreams());

            initAckCID = ch_makeInitAck(mdi_generateLocalTag(),
                                        rxc_get_local_receiver_window(),
                                        se_numOfSendStreams(), se_numOfRecvStreams(),
                                        /* TODO : check whether we take NEW TSN or leave an old one */
                                        mdi_generateStartTSN());

            mdi_readLocalAddresses(lAddresses, &nlAddresses);
            /* enter local addresses into initAck */
            if (nlAddresses > 0)
                ch_enterIPaddresses(initAckCID, lAddresses, nlAddresses);
            /* retreive remote source addresses from message */
            nrAddresses = ch_IPaddresses(initCID, rAddresses);
            /* TODO: IPv6 treatment */
            ch_enterCookieVLP(initAckCID,
                              ch_initFixed(initCID),
                              ch_initFixed(initAckCID), ch_cookieLifeTime(initCID), localData->local_tie_tag, /* this should be different from that in Init_Ack now */
                              localData->peer_tie_tag, lAddresses, nlAddresses,
                              rAddresses, nrAddresses);

            /* send initAck */
            bu_put_Ctrl_Chunk(ch_chunkString(initAckCID));
            bu_sendAllChunks(NULL);
            ch_deleteChunk(initAckCID);
            event_logi(INTERNAL_EVENT_1, "event: initAck sent in state %u", state);
            break;
        case SHUTDOWNACKSENT:
            /* We are supposed to discard the Init, and retransmit SHUTDOWN_ACK (9.2) */
            shutdownAckCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_ACK, FLAG_NONE);
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownAckCID));
            bu_sendAllChunks(NULL);
            ch_deleteChunk(shutdownAckCID);
            break;
        default:
            error_logi(ERROR_MAJOR, "Unexpected State %02u - Program Error ???", state);
            break;
        }
    }

    /* was only treated with ch_makeChunk -- it is enough to "FORGET" it */
    ch_forgetChunk(initCID);
    return removed_association;
}


/**
 * scr_initAck is called by bundling when a init acknowledgement was received from the peer.
 * The following data are retrieved from the init-data and saved for this association:
 * \begin{itemize}
 * \item remote tag from the initiate tag field
 * \item receiver window credit of the peer
 * \item number of send streams of the peer, must be lower or equal the number of receive streams
 *   this host has announced with the init-chunk
 * \item number of receive streams the peer allows the receiver of this initAck to use
 * \end{itemize}
 * The initAck must contain a cookie which is returned to the peer with the cookie acknowledgement.
 * @param initAck  pointer to received initAck-chunk including optional parameters without chunk header
 */
gboolean scr_initAck(SCTP_init * initAck)
{
    guint32 state;
    union sockunion *destAddress;
    union sockunion dAddresses[MAX_NUM_ADDRESSES];
    int ndAddresses;
    unsigned short inbound_streams;
    unsigned short outbound_streams;
    ChunkID cookieCID;
    ChunkID initCID;
    ChunkID initAckCID;
    gboolean removed_association = FALSE;

    initAckCID = ch_makeChunk((SCTP_simple_chunk *) initAck);

    if (ch_chunkType(initAckCID) != CHUNK_INIT_ACK) {
        /* error logging */
        ch_forgetChunk(initAckCID);
        error_log(ERROR_MAJOR, "scr_initAck: wrong chunk type");
        return removed_association;
    }

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        ch_forgetChunk(initAckCID);
        error_log(ERROR_MAJOR, "scr_initAck: read SCTP-control failed");
        return removed_association;
    }

    state = localData->association_state;

    switch (state) {
    case COOKIE_WAIT:

        event_log(EXTERNAL_EVENT, "event: initAck in state COOKIE_WAIT");

        /* Set length of chunk to HBO !! */
        initCID = ch_makeChunk((SCTP_simple_chunk *) localData->initChunk);

        if (ch_noOutStreams(initAckCID) == 0 || ch_noInStreams(initAckCID) == 0) {
            /* delete all data of this association */
            mdi_communicationLostNotif(0);
            mdi_deleteCurrentAssociation();
            removed_association = TRUE;
        }

        /* retrieve addresses from initAck */
        ndAddresses = ch_IPaddresses(initAckCID, dAddresses);

        /* tbde: read IPv6, if present send abort with error cause unresolvable address. */

        if (ndAddresses > 0) {
            /* save addresses if initAck contained more than zero, otherwise the address
               of the associate primitive will be used for this association. */
            mdi_writeDestinationAddresses(dAddresses, ndAddresses);
        } else {
            /* otherwise the source address of the IP-message carrying the cookie-chunk
               will be used for this association. This may be equal to the one the init
               was sent to, but can also be different if the peer decides to setup the
               association on another address. */
            destAddress = mdi_readLastFromAddress();
            mdi_writeDestinationAddresses(destAddress, 1);
            free(destAddress);
        }

        /* initialize rest of association with data received from peer */

        inbound_streams = min(ch_noOutStreams(initAckCID), localData->NumberOfInStreams);
        outbound_streams = min(ch_noInStreams(initAckCID), localData->NumberOfOutStreams);

        mdi_initAssociation(ch_receiverWindow(initAckCID), /* remotes side initial rwnd */
                            inbound_streams, /* # of remote output/local input streams */
                            outbound_streams, /* # of remote input/local output streams */
                            ch_initialTSN(initAckCID), /* remote initial TSN */
                            ch_initiateTag(initAckCID), /* remote init tag */
                            ch_initialTSN(initCID)); /* local initial TSN for sending */

       event_logii(VERBOSE, "scr_InitAck(): called mdi_initAssociation(in-streams=%u, out-streams=%u)",
                    inbound_streams,outbound_streams);


        /* reset length field again to NBO... */
        ch_chunkString(initCID),
            /* free initChunk memory */
            ch_forgetChunk(initCID);

        cookieCID = ch_makeCookie(ch_cookieParam(initAckCID));

        if (cookieCID < 0) {
            event_log(EXTERNAL_EVENT, "received a initAck without cookie");

            /* stop shutdown timer */
            sctp_stopTimer(localData->initTimer);
            mdi_communicationLostNotif(0);
            /* delete this association */
            mdi_deleteCurrentAssociation();
            removed_association = TRUE;
            state = CLOSED;
            return removed_association;
        }

        localData->cookieChunk = (SCTP_cookie_echo *) ch_chunkString(cookieCID);
        /* populate tie tags -> section 5.2.1/5.2.2 */
        localData->local_tie_tag = mdi_readAssociationID();
        localData->peer_tie_tag = ch_initiateTag(initAckCID);


        localData->NumberOfOutStreams = outbound_streams;
        localData->NumberOfInStreams =  inbound_streams;


        ch_forgetChunk(cookieCID);
        ch_forgetChunk(initAckCID);

        /* send cookie */
        bu_put_Ctrl_Chunk((SCTP_simple_chunk *) localData->cookieChunk);
        bu_sendAllChunks(NULL);
        event_log(INTERNAL_EVENT_1, "event: sent cookie echo");

        state = COOKIE_ECHOED;
        /* stop init timer */
        sctp_stopTimer(localData->initTimer);
        /* start cookie timer */
        localData->initTimer = sctp_startTimer(localData->initTimerDuration,
                                               &sci_timer_expired,
                                               (void *) &localData->associationID, NULL);
        break;

    case COOKIE_ECHOED:
        /* Duplicated initAck, ignore */
        event_log(EXTERNAL_EVENT, "event: duplicatied scr_initAck in state COOKIE_ECHOED");
        break;
    case CLOSED:
    case ESTABLISHED:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNSENT:
        /* In this states the initAck is unexpected event. */
        event_logi(EXTERNAL_EVENT, "discarding event: scr_initAck in state %02d", state);
        break;
    default:
        /* error logging: unknown event */
        event_logi(EXTERNAL_EVENT, "scr_initAck: unknown state %02d", state);
        break;
    }

    localData->association_state = state;
    localData = NULL;
    return removed_association;
}


/**
  scr_cookie_echo is called by bundling when a cookie echo chunk was received from  the peer.
  The following data is retrieved from the cookie and saved for this association:
    \begin{itemize}
    \item  from the init chunk:
        \begin{itemize}
        \item peers tag
        \item peers receiver window credit
        \item peers initial TSN
        \item peers network address list if multihoming is used
        \end{itemize}
    \item local tag generated before the initAck was sent
    \item my initial TSN generated before the initAck was sent
    \item number of send streams I use, must be lower or equal to peers number of receive streams from init chunk
    \item number of receive streams I use (can be lower than number of send streams the peer requested in
     the init chunk
    \end{itemiz}
   @param  cookie_echo pointer to the received cookie echo chunk
 */
void scr_cookie_echo(SCTP_cookie_echo * cookie_echo)
{

/* CHECKME :
    Address handling in the ReInitialization Case -> do we need to
    update to new addresses when cookie echo carries other/new addresses
    of our peer
 */
    union sockunion *destAddress;
    union sockunion dAddresses[MAX_NUM_ADDRESSES];
    int ndAddresses;
    guint32 state, new_state = 0xFFFFFFFF;
    unsigned int cookieLifetime;
    ChunkID cookieCID;
    ChunkID cookieAckCID;
    ChunkID initCID;
    ChunkID initAckCID;
    ChunkID shutdownAckCID;
    ChunkID errorCID;
    guint32 cookie_local_tag, cookie_remote_tag;
    guint32 cookie_local_tietag, cookie_remote_tietag;
    guint32 local_tag, remote_tag;
    short primaryDestinationAddress;
    short noOfDestinationAddresses;
    unsigned short noSuccess;

    cookieCID = ch_makeChunk((SCTP_simple_chunk *) cookie_echo);

    if (ch_chunkType(cookieCID) != CHUNK_COOKIE_ECHO) {
        /* error logging */
        ch_forgetChunk(cookieCID);
        error_log(ERROR_MAJOR, "scr_cookie_echo: wrong chunk type");
        return;
    }
    /* section 5.2.4. 1) and 2.) */
    if (ch_goodCookie(cookieCID)) {
        ch_forgetChunk(cookieCID);
        event_log(EXTERNAL_EVENT, "event: invalidCookie received");
        return;
    }
    initCID = ch_cookieInitFixed(cookieCID);
    initAckCID = ch_cookieInitAckFixed(cookieCID);

    cookie_remote_tag = ch_initiateTag(initCID);
    cookie_local_tag = ch_initiateTag(initAckCID);

    local_tag = mdi_readAssociationID();
    remote_tag = mdi_readTagRemote();

    /* section 5.2.4. 3.) */
    if ((cookieLifetime = ch_staleCookie(cookieCID)) > 0) {
        event_logi(EXTERNAL_EVENT, "event: staleCookie received, lifetime = %d", cookieLifetime);

        if ((cookie_local_tag != local_tag)
            || (cookie_remote_tag != remote_tag)) {

            mdi_writeLastInitiateTag(cookie_remote_tag);
            /* make and send stale cookie error */
            errorCID = ch_makeSimpleChunk(CHUNK_ERROR, FLAG_NONE);
            ch_enterStaleCookieError(errorCID, (unsigned int) (1.2 * cookieLifetime));
            bu_put_Ctrl_Chunk(ch_chunkString(errorCID));
            bu_sendAllChunks(NULL);
            ch_forgetChunk(cookieCID);
            ch_deleteChunk(initCID);
            ch_deleteChunk(initAckCID);
            ch_deleteChunk(errorCID);
            return;
        }                       /* ELSE : Case 5.2.4.E. Valid Cookie, unpack into a TCB */
    }


    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        destAddress = mdi_readLastFromAddress();
        noOfDestinationAddresses = 1;
        primaryDestinationAddress = 0;

        /* why is INSTANCE_NAME here zero ? */
        noSuccess = mdi_newAssociation(0, mdi_readLastDestPort(), mdi_readLastFromPort(), cookie_local_tag, /* this is MY tag */
                                       primaryDestinationAddress,
                                       noOfDestinationAddresses, destAddress);

        free(destAddress);

        if (noSuccess) {
            /* new association could not be entered in the list of associations */
            error_log(ERROR_MAJOR, "scr_cookie_echo: Creation of association failed");
            ch_deleteChunk(initCID);
            ch_deleteChunk(initAckCID);
            ch_forgetChunk(cookieCID);
            return;
        }
    }

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        error_log(ERROR_MAJOR, "scr_cookie-echo: program error: SCTP-control failed");
        ch_deleteChunk(initCID);
        ch_deleteChunk(initAckCID);
        ch_forgetChunk(cookieCID);
        return;
    }

    state = localData->association_state;

    event_logiii(VERBOSE,
                 "State : %u, cookie_remote_tag : %x , cookie_local_tag : %x ",
                 state, cookie_remote_tag, cookie_local_tag);
    event_logii(VERBOSE, "remote_tag ; %x , local_tag : %x ", remote_tag, local_tag);

    switch (state) {
    case CLOSED:
  /*----------------- Normal association setup -----------------------------------------*/
        event_log(EXTERNAL_EVENT, "event: scr_cookie_echo in state CLOSED");
        /* retrieve destination addresses from cookie */
        ndAddresses = ch_cookieIPDestAddresses(cookieCID, dAddresses);
        if (ndAddresses > 0) {
            /* save addresses if initAck contained more then zero, otherwise the source address
               of the IP-message carrying the cookie-chunk will be used for this association. */
            event_logi(VERBOSE, "Storing %d destination addresses as paths", ndAddresses);
            mdi_writeDestinationAddresses(dAddresses, ndAddresses);
        }

        /* initialize new association from cookie data */
        /* FIXME : check number of remote out streams....might be too high a value */
        mdi_initAssociation(ch_receiverWindow(initCID),
                            ch_noOutStreams(initAckCID),
                            ch_noInStreams(initAckCID),
                            ch_initialTSN(initCID), cookie_remote_tag, ch_initialTSN(initAckCID));


        localData->NumberOfOutStreams = ch_noOutStreams(initAckCID);
        localData->NumberOfInStreams = ch_noInStreams(initAckCID);
        event_logii(VERBOSE, "Set Outbound Stream to %u, Inbound Streams to %u",
            localData->NumberOfOutStreams, localData->NumberOfInStreams);


        /* make cookie acknowledgement */
        cookieAckCID = ch_makeSimpleChunk(CHUNK_COOKIE_ACK, FLAG_NONE);

        /* notification to ULP */
        mdi_communicationUpNotif(SCTP_COMM_UP_RECEIVED_VALID_COOKIE);

        /* send cookie acknowledgement */
        bu_put_Ctrl_Chunk(ch_chunkString(cookieAckCID));
        bu_sendAllChunks(NULL);
        ch_deleteChunk(cookieAckCID);

        new_state = ESTABLISHED;
        break;

        /* For the rest of these (pathological) cases, refer to section 5.2.4 as to what to do */

    case COOKIE_WAIT:
    case COOKIE_ECHOED:
    case ESTABLISHED:
    case SHUTDOWNPENDING:
    case SHUTDOWNSENT:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNACKSENT:
        cookie_local_tietag = ch_CookieLocalTieTag(cookieCID);
        cookie_remote_tietag = ch_CookiePeerTieTag(cookieCID);

        event_logii(VERBOSE,
                    "cookie_remote_tietag ; %u , cookie_local_tietag : %u ",
                    cookie_remote_tietag, cookie_local_tietag);
        /* cookie_local_tag, cookie_remote_tag are set */
        /* local_tag, remote_tag are also set from the TCB */

        if (cookie_local_tag == local_tag) {        /* cases B or D */
            if (cookie_remote_tag == remote_tag) {  /* case D */
                /*  the endpoint should always enter the ESTABLISHED state, if it has not
                    already done so. It should stop any init or cookie timers that may be
                    running and send a COOKIE ACK */
                event_log(VERBOSE, "Dupl. CookieEcho, case 5.2.4.D)");
                /* stop COOKIE timers */
                sctp_stopTimer(localData->initTimer);
                /* go to ESTABLISHED state */
                new_state = ESTABLISHED;
                /* make cookie acknowledgement */
                cookieAckCID = ch_makeSimpleChunk(CHUNK_COOKIE_ACK, FLAG_NONE);
                /* send cookie acknowledgement */
                bu_put_Ctrl_Chunk(ch_chunkString(cookieAckCID));
                bu_sendAllChunks(NULL);
                ch_deleteChunk(cookieAckCID);
            } else {                                /* case B */
                /*  The endpoint should stay in or enter
                    the ESTABLISHED state but it MUST update its peer's Verification
                    Tag from the State Cookie, stop any init or cookie timers that may
                    running and send a COOKIE ACK. */
                event_log(VERBOSE, "Dupl. CookieEcho, case 5.2.4.B)");
                mdi_rewriteTagRemote(cookie_remote_tag);
                /* stop COOKIE timers */
                sctp_stopTimer(localData->initTimer);
                /* go to ESTABLISHED state */
                new_state = ESTABLISHED;
                /* make cookie acknowledgement */
                cookieAckCID = ch_makeSimpleChunk(CHUNK_COOKIE_ACK, FLAG_NONE);
                /* send cookie acknowledgement */
                bu_put_Ctrl_Chunk(ch_chunkString(cookieAckCID));
                bu_sendAllChunks(NULL);
                ch_deleteChunk(cookieAckCID);
            }
        } else {                                    /* cases A or C */
            if ((cookie_remote_tag == remote_tag) &&
                (cookie_local_tietag == 0) &&
                (cookie_remote_tietag == 0)) {  /* is case C */
                    /* section 5.2.4. action C : silently discard cookie */
                    event_log(VERBOSE, "Dupl. CookieEcho, case 5.2.4.C) --> Silently discard !");
                    ch_forgetChunk(cookieCID);
                    ch_deleteChunk(initCID);
                    ch_deleteChunk(initAckCID);
                    /* localData = 0 ???? */
                    return;         /* process data as usual ? */
            }  else if ((cookie_remote_tag != remote_tag) &&
                        (cookie_local_tietag == local_tag) &&
                        (cookie_remote_tietag == remote_tag)) {     /* case A */
                /* section 5.2.4. action A : Possible Peer Restart  */
                if (state != SHUTDOWNACKSENT) {
                    event_logi(VERBOSE, "Peer Restart, case 5.2.4.A, state == %u", state);
                    /* Pass RESTART_NOTIFICATION to ULP */
                    mdi_restartNotif();
                    /* what happens to SCTP data chunks is implementation specific */
                    /* reset congestion control parameters */
                    fc_restart(ch_receiverWindow(initCID));
                    rxc_restart_receivecontrol(ch_initialTSN(initCID));
                    /* go to ESTABLISHED state */
                    new_state = ESTABLISHED;
                    /* make cookie acknowledgement */
                    cookieAckCID = ch_makeSimpleChunk(CHUNK_COOKIE_ACK, FLAG_NONE);
                    /* send cookie acknowledgement */
                    bu_put_Ctrl_Chunk(ch_chunkString(cookieAckCID));
                    bu_sendAllChunks(NULL);
                    ch_deleteChunk(cookieAckCID);
                } else {
                    event_log(VERBOSE, "Peer Restart case, state == SHUTDOWN_ACK_SENT");
                    /* resend SHUTDOWN_ACK */
                    shutdownAckCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_ACK, FLAG_NONE);
                    /* add ERROR_CHUNK with Error Cause : "Cookie Received while shutting down" */
                    bu_put_Ctrl_Chunk(ch_chunkString(shutdownAckCID));
                    errorCID = ch_makeError(ECC_COOKIE_RECEIVED_DURING_SHUTDWN);
                    bu_put_Ctrl_Chunk(ch_chunkString(errorCID));
                    /* send cookie acknowledgement */
                    bu_sendAllChunks(NULL);
                    ch_deleteChunk(shutdownAckCID);
                    ch_deleteChunk(errorCID);
                }

            } else { /* silently discard */
                event_log(VERBOSE, "Dupl. CookieEcho, silently discarding CookieEcho");
                ch_forgetChunk(cookieCID);
                ch_deleteChunk(initCID);
                ch_deleteChunk(initAckCID);
                /* localData = 0 ???? */
                return;             /* process data as usual ? */
            }
        }
        break;
    default:
        /* error logging: unknown event */
        error_logi(EXTERNAL_EVENT_X, "scr_cookie_echo : unknown state %02u", state);
        break;
    }

    ch_deleteChunk(initCID);
    ch_deleteChunk(initAckCID);
    ch_forgetChunk(cookieCID);

    if (new_state != 0xFFFFFFFF)
        localData->association_state = new_state;
    localData = NULL;
}



/**
  scr_cookieAck is called by bundling when a cookieAck chunk was received from  the peer.
  The only purpose is to inform the active side that peer has received the cookie chunk.
  The association is in established state after this function is called.
  Communication up is signalled to the upper layer in this case.
  @param cookieAck pointer to the received cookie ack chunk
*/
void scr_cookieAck(SCTP_simple_chunk * cookieAck)
{
    guint32 state;
    ChunkID cookieAckCID;

    cookieAckCID = ch_makeChunk(cookieAck);

    if (ch_chunkType(cookieAckCID) != CHUNK_COOKIE_ACK) {
        /* error logging */
        error_log(ERROR_MAJOR, "scr_cookieAck: wrong chunk type");
        return;
    }
    ch_forgetChunk(cookieAckCID);


    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        error_log(ERROR_MAJOR, "scr_cookieAck: read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case COOKIE_ECHOED:

        event_logi(EXTERNAL_EVENT_X, "event: scr_cookieAck in state %02d", state);
        /* stop init timer */
        sctp_stopTimer(localData->initTimer);
        /* free  cookieChunk */
        free(localData->initChunk);
        free(localData->cookieChunk);
        localData->initChunk = NULL;
        localData->cookieChunk = NULL;
        mdi_communicationUpNotif(SCTP_COMM_UP_RECEIVED_COOKIE_ACK);

        state = ESTABLISHED;
        break;

    case ESTABLISHED:
        /* Duplicated cookie, ignore */
        break;
    case CLOSED:
    case COOKIE_WAIT:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNSENT:
        /* In this states the cookie is unexpected event.
           Do error logging  */
        event_logi(EXTERNAL_EVENT_X, "unexpected event: scr_cookieAck in state %02d", state);
        break;
    default:
        /* error logging: unknown event */
        break;
    }

    localData->association_state = state;
    localData = NULL;
}


/**
  scr_shutdown is called by bundling when a shutdown chunk was received from the peer.
  This function initiates a graceful shutdown of the association.
  @param  shutdown_chunk pointer to the received shutdown chunk
*/
gboolean scr_shutdown(SCTP_simple_chunk * shutdown_chunk)
{
    guint32 state, new_state;
    boolean readyForShutdown;
    unsigned int lastFromPath;
    gboolean removed_association = FALSE;
    ChunkID abortCID;
    ChunkID shutdownAckCID;
    ChunkID shutdownCID;

    shutdownCID = ch_makeChunk(shutdown_chunk);

    if (ch_chunkType(shutdownCID) != CHUNK_SHUTDOWN) {
        /* error logging */
        error_log(ERROR_MAJOR, "scr_cookieAck: wrong chunk type");
        ch_forgetChunk(shutdownCID);
        return removed_association;
    }

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "scr_shutdown: read SCTP-control failed");
        ch_forgetChunk(shutdownCID);
        return removed_association;
    }

    state = localData->association_state;
    new_state = state;

    lastFromPath = mdi_readLastFromPath();

    switch (state) {
    case CLOSED:
        event_log(EXTERNAL_EVENT, "event: scr_shutdown in state CLOSED, send ABORT ! ");
        abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NO_TCB);
        bu_put_Ctrl_Chunk(ch_chunkString(abortCID));
        bu_sendAllChunks(&lastFromPath);
		bu_unlock_sender(&lastFromPath);
        ch_deleteChunk(abortCID);
        ch_forgetChunk(shutdownCID);
        mdi_communicationLostNotif(SCTP_COMM_LOST_NO_TCB);
        /* delete all data of this association */
        mdi_deleteCurrentAssociation();
        removed_association = TRUE;
        break;

    case COOKIE_WAIT:
    case COOKIE_ECHOED:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNACKSENT:
        event_logi(EXTERNAL_EVENT, "event: scr_shutdown in state %2u -> discarding !", state);
        /* stop init timer */
        ch_forgetChunk(shutdownCID);
        break;

    case ESTABLISHED:
        event_log(EXTERNAL_EVENT, "event: scr_shutdown in state ESTABLISHED");
        new_state = SHUTDOWNRECEIVED;
        readyForShutdown = (rtx_readNumberOfUnackedChunks() == 0) &&
            (fc_readNumberOfQueuedChunks() == 0);

        if (readyForShutdown) {
            /* retransmissions are not necessary */
            /* send shutdownAck */
            event_log(VERBOSE, "We are ready for SHUTDOWN, sending SHUTDOWN_ACK !");
            shutdownAckCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_ACK, FLAG_NONE);
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownAckCID));
            bu_sendAllChunks(&lastFromPath);
            ch_deleteChunk(shutdownAckCID);
            ch_forgetChunk(shutdownCID);
            if (localData->initTimer != 0)
                sctp_stopTimer(localData->initTimer);

            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);
            new_state = SHUTDOWNACKSENT;
        } else {
            /* retrieve cummunalative TSN acked from shutdown chunk */
            rtx_shutdown();
            rtx_rcv_shutdown_ctsna(ch_cummulativeTSNacked(shutdownCID));
            /* retransmissions are necessary */
            /* call relible transfer and wait for sci_allChunksAcked */
        }
        break;

    case SHUTDOWNSENT:
        /* check wether reliable transfer is ready for shutdown */
        readyForShutdown = (rtx_readNumberOfUnackedChunks() == 0) &&
            (fc_readNumberOfQueuedChunks() == 0);

        if (readyForShutdown) {
            /* retransmissions are not necessary */
            /* send shutdownAck */
            shutdownAckCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_ACK, FLAG_NONE);
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownAckCID));
            bu_sendAllChunks(&lastFromPath);
            ch_deleteChunk(shutdownAckCID);
            ch_forgetChunk(shutdownCID);
            if (localData->initTimer != 0)
                sctp_stopTimer(localData->initTimer);

            localData->initTimer =
                sctp_startTimer(localData->initTimerDuration, &sci_timer_expired,
                                (void *) &localData->associationID, NULL);

            new_state = SHUTDOWNACKSENT;
        } else {
            error_log(ERROR_MAJOR, "Error in Program Logic !!!");
            error_log(ERROR_MAJOR,
                      "SHUTDOWN_SENT state may not be entered, if queues are not empty !!!!");

        }
        break;

    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scr_shutdown in state %02d: unexpected event", state);
        break;
    }
    localData->association_state = new_state;
    localData = NULL;
    return removed_association;

}



/**
  scr_shutdownAck is called by bundling when a shutdownAck chunk was received from the peer.
  Depending on the current state of the association, COMMUNICATION LOST is signaled to the
  Upper Layer Protocol, and the association marked for removal.
*/
gboolean scr_shutdownAck()
{
    guint32 state, new_state;
    unsigned int lastFromPath;
    ChunkID shdcCID;
    gboolean removed_association = FALSE;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "scr_shutdownAck: read SCTP-control failed");
        return removed_association;
    }

    lastFromPath = mdi_readLastFromPath();
    state = localData->association_state;
    new_state = state;

    switch (state) {
    case CLOSED:
    case COOKIE_WAIT:
    case COOKIE_ECHOED:
        /* see also section 8.5.E.) treat this like OOTB packet */
        event_logi(EXTERNAL_EVENT,
                   "event: scr_shutdownAck in state %u, send SHUTDOWN_COMPLETE ! ", state);
        shdcCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NO_TCB);
        bu_put_Ctrl_Chunk(ch_chunkString(shdcCID));
        bu_sendAllChunks(&lastFromPath);
		bu_unlock_sender(&lastFromPath);
        ch_deleteChunk(shdcCID);
        mdi_communicationLostNotif(SCTP_COMM_LOST_NO_TCB);
        removed_association = TRUE;
        /* delete all data of this association */
        mdi_deleteCurrentAssociation();
        break;
    case ESTABLISHED:
        error_log(EXTERNAL_EVENT,
                  "scr_shutdownAck in state ESTABLISHED, peer not standard conform ! ");
        break;
    case SHUTDOWNPENDING:
        error_log(EXTERNAL_EVENT,
                  "scr_shutdownAck in state SHUTDOWNPENDING, peer not standard conform ! ");
        break;
    case SHUTDOWNRECEIVED:
        error_log(EXTERNAL_EVENT,
                  "scr_shutdownAck in state SHUTDOWNRECEIVED, peer not standard conform ! ");
        break;

    case SHUTDOWNSENT:
    case SHUTDOWNACKSENT:
        shdcCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NONE);
        bu_put_Ctrl_Chunk(ch_chunkString(shdcCID));

        bu_sendAllChunks(&lastFromPath);
        ch_deleteChunk(shdcCID);
        if (localData->initTimer != 0)
            sctp_stopTimer(localData->initTimer);
        else
            error_log(ERROR_MAJOR, "Timer not running - Error in Program Logic");

        bu_unlock_sender(&lastFromPath);

        mdi_shutdownCompleteNotif();
        /* delete all data of this association */
        removed_association = TRUE;
        mdi_deleteCurrentAssociation();
        new_state = CLOSED;
        break;

    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scr_shutdownAck in state %02d: unexpected event", state);
        break;
    }
    localData->association_state = new_state;
    localData = NULL;
    return removed_association;

}

/**
  scr_shutdownComplete is called by bundling when a SHUTDOWN COMPLETE chunk was received from the peer.
  COMMUNICATION LOST is signaled to the ULP, timers stopped, and the association is marked for removal.
*/
gboolean scr_shutdownComplete()
{
    guint32 state, new_state;
    unsigned int lastFromPath;
    gboolean removed_association = FALSE;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "scr_shutdownComplete: read SCTP-control failed");
        return removed_association;
    }

    lastFromPath = mdi_readLastFromPath();

    state = localData->association_state;
    new_state = state;

    switch (state) {
    case CLOSED:
    case COOKIE_WAIT:
    case COOKIE_ECHOED:
    case ESTABLISHED:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNSENT:
        error_logi(EXTERNAL_EVENT, "scr_shutdownComplete in state %u -> discarding ! ", state);
        break;

    case SHUTDOWNACKSENT:
        if (localData->initTimer != 0)
            sctp_stopTimer(localData->initTimer);
        else
            event_log(INTERNAL_EVENT_0,
                      "scr_shutdownComplete : Timer not running - problem in Program Logic ??!!");
        mdi_shutdownCompleteNotif();
        /* delete all data of this association */
        mdi_deleteCurrentAssociation();
        removed_association = TRUE;
        new_state = CLOSED;
        break;

    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scr_shutdownComplete in state %02d: unexpected event", state);
        break;
    }
    localData->association_state = new_state;
    localData = NULL;
    return removed_association;
}

/**
  scr_abort is called by bundling when an ABORT chunk was received from  the peer.
  COMMUNICATION LOST is signalled to the ULP, timers are stopped, and the association
  is marked for removal.
 */
gboolean scr_abort()
{
    guint32 state;
    gboolean removed_association = FALSE;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "scr_abort: read SCTP-control failed");
        return removed_association;
    }

    state = localData->association_state;

    switch (state) {
    case CLOSED:
        event_log(EXTERNAL_EVENT, "event: scr_abort in state CLOSED -> discard chunk");
        /* discard chunk */
        break;
    case COOKIE_WAIT:
    case COOKIE_ECHOED:
    case SHUTDOWNSENT:
        event_logi(EXTERNAL_EVENT, "event: scr_abort in state %2d", state);

        /* stop init timer */
        sctp_stopTimer(localData->initTimer);
        /* delete all data of this association */
        mdi_communicationLostNotif(SCTP_COMM_LOST_ABORTED);
        removed_association = TRUE;
        mdi_deleteCurrentAssociation();

        break;
    case ESTABLISHED:
    case SHUTDOWNPENDING:
    case SHUTDOWNRECEIVED:
    case SHUTDOWNACKSENT:
        event_logi(EXTERNAL_EVENT, "event: scr_abort in state %02d", state);
        /* delete all data of this association */
        mdi_communicationLostNotif(SCTP_COMM_LOST_ABORTED);
        removed_association = TRUE;
        mdi_deleteCurrentAssociation();

        break;
    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scr_abort in state %02d: unexpected event", state);
        break;
    }
    return removed_association;
}


/**
   scr_staleCookie is called by bundling when a 'stale cookie' error chunk was received.
   @param error_chunk pointer to the received error chunk
*/
void scr_staleCookie(SCTP_simple_chunk * error_chunk)
{
    guint32 state;
    ChunkID errorCID;
    ChunkID initCID;

    errorCID = ch_makeChunk((SCTP_simple_chunk *) error_chunk);

    if (ch_chunkType(errorCID) != CHUNK_ERROR) {
        /* error logging */
        ch_forgetChunk(errorCID);
        error_log(ERROR_MAJOR, "scr_staleCookie: wrong chunk type");
        return;
    }

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "scr_staleCookie: read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case COOKIE_ECHOED:

        /* make chunkHandler init chunk from stored init chunk string */
        initCID = ch_makeChunk((SCTP_simple_chunk *) localData->initChunk);

        /* read staleness from error chunk and enter it into the cookie preserv. */
        ch_enterCookiePreservative(initCID, ch_stalenessOfCookieError(errorCID));

        /* resend init */
        bu_put_Ctrl_Chunk(ch_chunkString(initCID));
        bu_sendAllChunks(NULL);
        ch_forgetChunk(initCID);

        state = COOKIE_WAIT;
        break;

    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "scr_staleCookie in state %02d: unexpected event", state);
        break;
    }
    localData->association_state = state;
    localData = NULL;
}

/**
   sci_getState is called by distribution to get the state of the current SCTP-control instance.
   This function also logs the state with log-level VVERBOSE.
   @return state value (0=CLOSED, 3=ESTABLISHED)
*/
guint32 sci_getState()
{
    guint32 state;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_getState: read SCTP-control failed");
        return CLOSED;
    }

    state = localData->association_state;

    switch (state) {
    case CLOSED:
        event_log(VVERBOSE, "Current state : CLOSED");
        break;
    case COOKIE_WAIT:
        event_log(VVERBOSE, "Current state :COOKIE_WAIT ");
        break;
    case COOKIE_ECHOED:
        event_log(VVERBOSE, "Current state : COOKIE_ECHOED");
        break;
    case ESTABLISHED:
        event_log(VVERBOSE, "Current state : ESTABLISHED");
        break;
    case SHUTDOWNPENDING:
        event_log(VVERBOSE, "Current state : SHUTDOWNPENDING");
        break;
    case SHUTDOWNRECEIVED:
        event_log(VVERBOSE, "Current state : SHUTDOWNRECEIVED");
        break;
    case SHUTDOWNSENT:
        event_log(VVERBOSE, "Current state : SHUTDOWNSENT");
        break;
    case SHUTDOWNACKSENT:
        event_log(VVERBOSE, "Current state : SHUTDOWNACKSENT");
        break;
    default:
        event_log(VVERBOSE, "Unknown state : return closed");
        return CLOSED;
        break;
    }

    return state;
    localData = NULL;
}


/*------------------- Functions called by reliable transfer --------------------------------------*/

/**
  Called by reliable transfer if all (sent !) chunks in its retransmission queue have been acked.
  This function is used to move from state SHUTDOWNPENDING to  SHUTDOWNSENT (after having sent a
  shutdown chunk) or to move from  SHUTDOWNRECEIVED to SHUTDOWNACKSENT (after having sent a
  shutdown-ack chunk)
*/
void sci_allChunksAcked()
{
    guint32 state;
    ChunkID shutdownCID;
    ChunkID shutdownAckCID;

    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_allChunksAcked: read SCTP-control failed");
        return;
    }

    state = localData->association_state;

    switch (state) {
    case SHUTDOWNPENDING:

        event_log(EXTERNAL_EVENT, "event: sci_allChunksAcked in state SHUTDOWNPENDING");

        /* make and send shutdown */
        shutdownCID = ch_makeShutdown(rxc_read_cummulativeTSNacked());
        bu_put_Ctrl_Chunk(ch_chunkString(shutdownCID));
        bu_sendAllChunks(NULL);
        ch_deleteChunk(shutdownCID);

        /* start shutdown timer */
        localData->initTimerDuration = pm_readRTO(pm_readPrimaryPath());
        localData->initTimer = sctp_startTimer(localData->initTimerDuration,
                                               &sci_timer_expired,
                                               (void *) &localData->associationID, NULL);

        localData->initRetransCounter = 0;

        /* receive control must acknowledge every datachunk at once after the shutdown
           was sent. */
        rxc_send_sack_everytime();

        state = SHUTDOWNSENT;

        break;

    case SHUTDOWNRECEIVED:

        event_log(EXTERNAL_EVENT, "event: sci_allChunksAcked in state SHUTDOWNRECEIVED");

        /* send shutdownAck */
        shutdownAckCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_ACK, FLAG_NONE);
        bu_put_Ctrl_Chunk(ch_chunkString(shutdownAckCID));
        bu_sendAllChunks(NULL);
        ch_deleteChunk(shutdownAckCID);

        state = SHUTDOWNACKSENT;
        break;

    default:
        /* error logging */
        event_logi(EXTERNAL_EVENT_X, "unexpected event: sci_allChunksAcked in state %d", state);
        break;
    }

    localData->association_state = state;
    localData = NULL;
}



/*------------------- Functions called message by distribution to create and delete --------------*/

/**
    newSCTP_control allocates data for a new SCTP-Control instance
 */
void *sci_newSCTP_control(void* sctpInstance)
{
    localData = (SCTP_controlData *) malloc(sizeof(SCTP_controlData));

    localData->association_state = CLOSED;
    localData->initTimer = 0;
    localData->initTimerDuration = RTO_INITIAL;
    localData->initRetransCounter = 0;
    localData->initChunk = NULL;
    localData->cookieChunk = NULL;
    localData->associationID = mdi_readAssociationID();
    localData->NumberOfOutStreams = mdi_readLocalOutStreams();
    localData->NumberOfInStreams = mdi_readLocalInStreams();

    localData->assocMaxRetransmissions = mdi_getDefaultAssocMaxRetransmits(sctpInstance);
    localData->assocMaxInitRetransmissions = mdi_getDefaultMaxInitRetransmits(sctpInstance);
    localData->cookieLifeTime = mdi_getDefaultValidCookieLife(sctpInstance);
    localData->instance = sctpInstance;

    event_log(INTERNAL_EVENT_0, "event: created SCTP-control Instance");

    return (void *) localData;
}



/**
  deleteSCTP_control frees memory allocated for a SCTP-Control instance
*/
void sci_deleteSCTP_control(void *sctpControlData)
{
    SCTP_controlData *sctpCD;

    event_log(INTERNAL_EVENT_0, "deleting SCTP-control");
    sctpCD = (SCTP_controlData *) sctpControlData;
    if (sctpCD->initChunk != NULL)
        free(sctpCD->initChunk);
    if (sctpCD->cookieChunk != NULL)
        free(sctpCD->cookieChunk);
    free(sctpControlData);
}

/**
 * get current parameter value for assocMaxRetransmissions
 * @return current value, -1 on error
 */
int sci_getMaxAssocRetransmissions(void)
{
    int max;
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_getMaxAssocRetransmissions(): read SCTP-control failed");
        return -1;
    }
    max =   localData->assocMaxRetransmissions;
    localData = NULL;
    return max;
}

/**
 * get current parameter value for assocMaxInitRetransmissions
 * @return current value, -1 on error
 */
int sci_getMaxInitRetransmissions(void)
{
    int max;
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_getMaxInitRetransmissions(): read SCTP-control failed");
        return -1;
    }
    max =   localData->assocMaxInitRetransmissions;
    localData = NULL;
    return max;
}

/**
 * get current parameter value for cookieLifeTime
 * @return current value, -1 on error
 */
int sci_getCookieLifeTime(void)
{
    int max;
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_getCookieLifeTime(): read SCTP-control failed");
        return -1;
    }
    max =   localData->cookieLifeTime;
    localData = NULL;
    return max;
}

/**
 * set new parameter value for assocMaxRetransmissions
 * @param new_max  new parameter value for assocMaxRetransmissions
 * @return 0 for success, -1 on error
 */
int sci_setMaxAssocRetransmissions(int new_max)
{
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_setMaxAssocRetransmissions(): read SCTP-control failed");
        return -1;
    }
    localData->assocMaxRetransmissions = new_max;
    localData = NULL;
    return 0;
}

/**
 * set new parameter value for assocMaxRetransmissions
 * @param new_max  new parameter value for assocMaxRetransmissions
 * @return 0 for success, -1 on error
 */
int sci_setMaxInitRetransmissions(int new_max)
{
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_setMaxInitRetransmissions(): read SCTP-control failed");
        return -1;
    }
    localData->assocMaxInitRetransmissions = new_max;
    localData = NULL;
    return 0;
}

/**
 * set new parameter value for cookieLifeTime
 * @param new_max  new parameter value for cookieLifeTime
 * @return 0 for success, -1 on error
 */
int sci_setCookieLifeTime(int new_max)
{
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_setCookieLifeTime(): read SCTP-control failed");
        return -1;
    }
    localData->cookieLifeTime= new_max;
    localData = NULL;
    return 0;
}


gboolean sci_shutdown_procedure_started()
{

    guint32 state;
    if ((localData = (SCTP_controlData *) mdi_readSCTP_control()) == NULL) {
        /* error log */
        error_log(ERROR_MAJOR, "sci_readState : read SCTP-control failed");
        return FALSE;
    }
    state = localData->association_state;

    if (state == SHUTDOWNPENDING || state == SHUTDOWNRECEIVED
        || state == SHUTDOWNSENT || state == SHUTDOWNACKSENT) return TRUE;
    else
        return FALSE;
}

//@}
