/*
 *  $Id: auxiliary.c,v 1.1.1.2 2001/02/15 14:19:38 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module offers functions needed for validation and
 * other purposes.
 *
 */

#include "auxiliary.h"
#include "globals.h"

#include <stdlib.h>
#include <stdio.h>

#define BASE 65521L             /* largest prime smaller than 65536 */
#define NMAX 5552

/* NMAX is the largest n such that 255n(n+1)/2 + (n+1)(BASE-1) <= 2^32-1 */

#define DO1(buf,i)  {s1 += buf[i]; s2 += s1;}
#define DO2(buf,i)  DO1(buf,i); DO1(buf,i+1);
#define DO4(buf,i)  DO2(buf,i); DO2(buf,i+2);
#define DO8(buf,i)  DO4(buf,i); DO4(buf,i+4);
#define DO16(buf)   DO8(buf,0); DO8(buf,8);




unsigned int adler32(unsigned int adler, const unsigned char *buf, unsigned int len);



unsigned char* key_operation(int operation_code)
{
    static unsigned char *secret_key = NULL;
    unsigned int count = 0, tmp;

    if (operation_code == KEY_READ) return secret_key;
    else if (operation_code == KEY_INIT) {
        if (secret_key != NULL) {
            error_log(ERROR_MAJOR, "tried to init secret key, but key already created !");
            return secret_key;
        }
        secret_key = malloc(SECRET_KEYSIZE);
        while (count < SECRET_KEYSIZE){
            /* if you care for security, you need to use a cryptographically secure PRNG */
            tmp = random();
            memcpy(&secret_key[count], &tmp, sizeof(unsigned int));
            count += 4;
        }
    } else {
        error_log(ERROR_MAJOR, "unknown key operation code !");
        return NULL;
    }
    return secret_key;
}


int validate_insert_checksum(unsigned char *buffer, int length)
{
    SCTP_message *message;
    unsigned int a32;
    /* save crc value from PDU */
    if (length > NMAX)
        return -1;
    message = (SCTP_message *) buffer;
    message->common_header.checksum = htonl(0L);

    /* now compute the thingie */
    /* FIXME : sanity checks for size etc. */
    a32 = adler32(1L, buffer, length);

    /* and insert it into the message */
    message->common_header.checksum = htonl(a32);

    event_logi(VERBOSE, "DEBUG Validation : Inserting adler32 == %x", a32);

    return 1;
}


int validate_size(unsigned char *header_start, int length)
{
    if ((length % 4) != 0L)
        return 0;
    return 1;
}



int validate_crc(unsigned char *header_start, int length)
{
    SCTP_message *message;
    unsigned int old_crc32;
    unsigned int a32;

    if (length > NMAX)
        return -1;
    /* save crc value from PDU */
    message = (SCTP_message *) header_start;
    old_crc32 = ntohl(message->common_header.checksum);

    event_logi(VVERBOSE, "DEBUG Validation : old_crc32 == %x", old_crc32);

    message->common_header.checksum = htonl(0L);

    /* now compute the thingie */
    a32 = adler32(1L, header_start, length);

    event_logi(VVERBOSE, "DEBUG Validation : adler32 == %x", a32);
    if (a32 == old_crc32)
        return 1;
    return 0;
}

int validate_datagram(unsigned char *buffer, int length)
{
    /* sanity check for size (min,max, multiple of 32 bits) */
    if (!validate_size(buffer, length))
        return 0;
    if (!validate_crc(buffer, length))
        return 0;
    /* FIXME :  validation is not yet complete */
    return 1;
}

/** 
 * adler32.c -- compute the Adler-32 checksum of a data stream
 * Copyright (C) 1995-1996 Mark Adler
 * For conditions of distribution and use, see copyright notice in zlib.h
 * available, e.g. from  http://www.cdrom.com/pub/infozip/zlib/
 */
unsigned int adler32(unsigned int adler, const unsigned char *buf, unsigned int len)
{
    unsigned int s1 = adler & 0xffff;
    unsigned int s2 = (adler >> 16) & 0xffff;
    int k;

    if (buf == NULL)
        return 1L;

    while (len > 0) {
        k = len < NMAX ? len : NMAX;
        len -= k;
        while (k >= 16) {
            DO16(buf);
            buf += 16;
            k -= 16;
        }
        if (k != 0)
            do {
                s1 += *buf++;
                s2 += s1;
            }
            while (--k);
        s1 %= BASE;
        s2 %= BASE;
    }
    return (s2 << 16) | s1;
}
