/*
 * $Id: bundling.h,v 1.4 2001/03/12 18:59:58 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 *
 * Purpose: This module bundels chunks chunks to be send into UDP-datagramms. Chunks are accepted
 *          with the putChunk function until sendChunk is called, which causes the transmission of
 *          all chunks accumulated so far.
 *          In the receive direction, this module gets UDP-dataparts from SCTP message-distribution,
 *          which are disassambled (debundled) into chunks. Depending on the chunk-type, the chunk
 *          are distributed to SCTP-control, RX-control, pathmanagement or reliable transfer.
 */

#ifndef BUNDLING_H
#define BUNDLING_H


#include "globals.h"            /* for chunk struct definition */
#include "SCTP-control.h"
#include "pathmanagement.h"

void bu_init_bundling(void);
gint bu_put_Ctrl_Chunk(SCTP_simple_chunk * chunk);
gint bu_put_Data_Chunk(SCTP_simple_chunk * chunk);


/*
 * bu_new: Creates a new bundling instance and returns a pointer to its data. 
 */
gpointer bu_new(void);

/* 
 * bu_delete: Deletes a bundling instance
 * 
 * Params: Pointer/handle which was returned by bu_new()
 */
void bu_delete(gpointer instancePtr);


/*
 * rbu_scanDatagram: looks for chunk_type in a newly received datagram
 *
 * All chunks within the datagram are lookes at, until one is found
 * that equals the parameter chunk_type.
 * @return true is chunk_type exists in SCTP datagram, false if it is not in there
 */
gboolean rbu_scanDatagram(guchar * datagram, guint len, gushort chunk_type);


/*
 * rbu_findChunk: looks for chunk_type in a newly received datagram
 *
 * All chunks within the datagram are looked at, until one is found
 * that equals the parameter chunk_type.
 * @return pointer to first chunk of chunk_type in SCTP datagram, else NULL
 */
guchar* rbu_findChunk(guchar * datagram, guint len, gushort chunk_type);

/*
 * rbu_scanDatagramForError : looks for Error chunk_type in a newly received datagram
 * that contains a special error cause code
 *
 * All chunks within the datagram are lookes at, until one is found
 * that equals the parameter chunk_type.
 * @return true is chunk_type exists in SCTP datagram, false if it is not in there
 */
gboolean rbu_scanDatagramForError(guchar * datagram, guint len, gushort error_cause);


/*
 * rbu_rcvDatagram: Hands a lower layer datagram over to bundling (for de-bundling)
 * 
 * All chunks within the datagram are dispatched and sent to the appropriate
 * module, i.e.: control chunks are sent to sctp_control/pathmanagement, 
 * SACK chunks to reliable_transfer, and data_chunks to RX_control.
 * Those modules must get a pointer to the start of a chunk and 
 * information about its size (without padding).
 */
gint rbu_rcvDatagram(guint address_index, guchar * datagram, guint len);


void bu_lock_sender();
void bu_unlock_sender(guint* ad_idx);

/**
 * bu_put_SACK_Chunk: Called by recvcontrol, when a SACK must be piggy-backed
 *
 * @param chunk pointer to chunk, that is to be put in the bundling buffer
 * @return error value, 0 on success, -1 on error
 */
gint bu_put_SACK_Chunk(SCTP_sack_chunk * chunk);


/*
 * bu_sendAllChunks: Trigger to send all chunks previously entered with putChunk.
 *
 * Return value: error value
 * Chunks sent are deleted afterwards.
 */
gint bu_sendAllChunks(guint * ad_idx);

void bu_request_sack(void);

#endif
