/*
 *  $Id: distribution.c,v 1.18 2001/03/13 14:14:45 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This modules implements the interface defined distribution.h and sctp.h
 *          and holds a private list of associations.
 *          As an SCTP-instance has usually more than one associations, the main purpose
 *          of this module is the  distribution of signals from the ULP and from the 
 *          peer (via the socket and unix-interface) to the addressed association.
 *          Signals from the UNIX-interface are always forwarded to 
 *          the bundling module of the addressed association.
 *          The corresponding function receive_msg is not defined here but in distribution.c,
 *          because it is called via a function-pointer that is registered at the
 *          UNIX-interface when SCTP is initialized.
 *          Signals from the ULP are forwarded to SCTP-Control, Pathmanagement or
 *          Streamengine of the addressed association depending on the chunk type.
 *          All signals from the ULP contain the association ID, which is used to
 *          identify the association.
 *
 * Remarks: Host and network byte order (HBO and NBO):
 *          In this module all data are data are HBO, except the IP-Addresses and message strings.
 *          Thus everything entered into a message string must be converted to NBO and
 *          everthing extracted from a message string must be converted to HBO (except the
 *          IP-addresses).
 *
 * function prefixes: sctp_    for interface-defs imported from sctp.h
 *                    mdi_     for interface-defs imported from distribution.h
 */

#include  <sctp.h>              /* ULP-interface definitions */
#include  "distribution.h"      /* SCTP-internal interfaces to message distribution */
#include  "adaptation.h"        /* interfaces to adaptation layer */
#include  "bundling.h"          /* interfaces to bundling */
#include  "SCTP-control.h"      /* interfaces to SCTP-control */
#include  "auxiliary.h"
#include  "streamengine.h"      /* interfaces to streamengine */
#include  "flowcontrol.h"       /* interfaces to flowcontrol */
#include  "recvctrl.h"          /* interfaces to receive-controller */
#include  "chunkHandler.h"

#include  <sys/types.h>
#include  <stdlib.h>            /* for malloc */
#include  <stdio.h>             /* for malloc */
#include  <errno.h>
#include  <arpa/inet.h>         /* for inet_ntoa() under both SOLARIS/LINUX */

/*------------------------ Default Definitions --------------------------------------------------*/


/*------------------------Structure Definitions --------------------------------------------------*/

/**
 * This struct stores data of SCTP-instances.
 * Each SCTP-instances is related to one port and to
 * one SCTP adaption-layer. This may change soon !
 */
typedef struct SCTPINSTANCE
{
    //@{
    /** The name of this SCTP-instance, used as key. */
    unsigned short sctpInstanceName;
    /** The local port of this instance, or zero for don't cares.
        Once assigned this should not be changed !   */
    unsigned short localPort;
    guint16 noOfLocalAddresses;
    union sockunion *localAddressList;
    unsigned char* localAddressStrings;
    /** set of callback functions that were registered by the ULP */
    SCTP_ulpCallbacks ULPcallbackFunctions;
    /** maximum number of incoming streams that this instance will take */
    unsigned short noOfInStreams;
    /** maximum number of outgoingng streams that this instance will take */
    unsigned short noOfOutStreams;
    /** here follow default parameters for instance initialization */
    unsigned int default_rtoInitial;
    unsigned int default_validCookieLife;
    unsigned int default_assocMaxRetransmits;
    unsigned int default_pathMaxRetransmits;
    unsigned int default_maxInitRetransmits;
    unsigned int default_myRwnd;
    unsigned int default_delay;
    unsigned char default_ipTos;
    unsigned int default_rtoMin;
    unsigned int default_maxSendQueue;
    unsigned int default_maxRecvQueue;

   //@}
}
SCTP_instance;


/**
 * This struct contains all data of an association. As far as other modules must know elements
 * of this struct, read functions are provided. No other module has write access to this structure.
 */
typedef struct ASSOCIATION
{
    //@{

    /** The local tag of this association.
      It is also used as ID of this association,
      it is used as a key to find a association in the list.  */
    unsigned int tagLocal;
    /** The tag of remote side of this association */
    unsigned int tagRemote;
    /** Pointer to the SCTP-instance this association
        belongs to. It is equal to the wellknown port
        number of the ULP that uses this instance. */
    SCTP_instance*  sctpInstance;
    /** the local port number of this association. */
    unsigned short localPort;
    /** the remote port number of this association. */
    unsigned short remotePort;
    /** number of destination networks (paths) */
    unsigned int noOfNetworks;
    /** array of destination addresses */
    union sockunion *destinationAddresses;

    /** pointer to flowcontrol structure */
    void *flowControl;
    /** pointer to reliable-transfer structure */
    void *reliableTransfer;
    /** pointer to receive-control structure */
    void *rx_control;
    /** pointer to stream structure */
    void *streamengine;
    /** pointer to pathmanagement structure */
    void *pathMan;
    /** pointer to bundling structure */
    void *bundling;
    /** pointer to SCTP-control */
    void *sctp_control;
    /** marks an association for deletion */
    boolean deleted;
    /** transparent pointer to some upper layer data */
    void * ulp_dataptr;
    /** IP TOS value per association */
    unsigned char ipTos;
    //@}
} Association;


/******************** Declarations ****************************************************************/

/*
    Keyed list of SCTP-instances with the instanceName as key
*/
/**
 * Keyed list of associations with the association-ID as key
 */
static GList* AssociationList = NULL;

/**
 * Whenever an external event (ULP-call, socket-event or timer-event) this variable must
 * contain the addressed sctp instance.
 * This pointer must be reset to null after the event  has been handled.
 */
static SCTP_instance *sctpInstance;

/**
 * Keyed list of SCTP instances with the instance name as key
 */
static GList* InstanceList = NULL;
static unsigned int ipv4_users = 0;
#ifdef HAVE_IPV6
    static unsigned int ipv6_users = 0;
#endif
static gboolean first_instance = TRUE;
/**
 * Whenever an external event (ULP-call, socket-event or timer-event) this variable must
 * contain the addressed association.
 * Read functions for 'global data' read data from the association pointed to by this pointer.
 * This pointer must be reset to null after the event  has been handled.
 */
static Association *currentAssociation;
static Association tmpAssoc;
static union sockunion tmpAddress;


/* Asssociation ID's are derived by a random number generator because they are also
   used as local tag. If firstSCTP_instance is true, a seed is generated by
   use of (current time). After the first SCTP-instance was created, firstSCTP_instance
   is set to false.
*/
static boolean clearAssociation;
static unsigned short lastSCTP_instanceName = 1;

/**
   initAck is sent to this address
   In this case, SCTP-control reads this address on reception of the cookie echo
   (which consequently also does not contain an addresslist) to initialize the new association.
 */
static union sockunion *lastFromAddress;

static short lastFromPath;
static unsigned short lastFromPort;
static unsigned short lastDestPort;
static unsigned int lastInitiateTag;

/**
  Descriptor of socket used by all associations and SCTP-instances.
 */
static gint sctp_socket;

#ifdef HAVE_IPV6
static gint ipv6_sctp_socket;
#endif

/* port management array */
static unsigned char portsSeized[0x10000];
static unsigned int lastPortSeized;
static unsigned int numberOfSeizedPorts;

static long rstate[2];


/* ---------------------- Internal Function Prototypes ------------------------------------------- */

static void mdi_deleteAssociation(Association * assoc);

/* ------------------------- Function Implementations --------------------------------------------- */

/*------------------- Internal Functions ---------------------------------------------------------*/




/*------------------- Internal LIST Functions ----------------------------------------------------*/

gint CompareInstancePorts(gconstpointer a, gconstpointer b)
{
    if ((((SCTP_instance*)a)->localPort) < ((SCTP_instance*)b)->localPort) return -1;
    else if ((((SCTP_instance*)a)->localPort) == ((SCTP_instance*)b)->localPort) return 0;
    else return 1;
}

gint CompareInstanceNames(gconstpointer a, gconstpointer b)
{
    if ((((SCTP_instance*)a)->sctpInstanceName) < ((SCTP_instance*)b)->sctpInstanceName) return -1;
    else if ((((SCTP_instance*)a)->sctpInstanceName) == ((SCTP_instance*)b)->sctpInstanceName) return 0;
    else return 1;
}


/**
 *  compareAssociationIDs compares the association ID's of two associations and returns 0
 *  if they are equal. This is a call back function called by List Functions whenever two
 *  association need to be compared.
 *  @param a  pointer to association struct 1
 *  @param b  pointer to association struct 2
 *  @return    0 if a->tagLocal equals b->tagLocal, 1 if bigger, -1 if smaller
 */
gint compareAssociationIDs(gconstpointer a, gconstpointer b)
{
    /* delete association that have already been marked for deletion */
    /* a1 is the one in list, a2 is the one to be matched */
    if (((Association*)a)->deleted) {
        return 0;
    }
    /* two associations are equal if there local tags (in this implementation also used as
       association ID) are equal. */
    if (((Association*)a)->tagLocal == ((Association*)b)->tagLocal)
        return 0;
    else if (((Association*)a)->tagLocal < ((Association*)b)->tagLocal)
        return -1;
    else
        return 1;
}



/**
 *  equalAssociations compares two associations and returns 0 if they are equal. In contrast to
 *  function compareAssociationIDs, equal here means the two associations belong to the same
 *  SCTP-instance and have at least one destinationaddress in common.
 *  This is a call back function called by GList-functions whenever two association need to be compared.
 *  @param i1  association data 1
 *  @param i2  association data 2
 *  @return 0 if il1 and il2 are equal according to above definition, 1 else
 */
gint equalAssociations(gconstpointer a, gconstpointer b)
{
    int i,j;
    /* two associations are equal if their remote and local ports are equal and at least
       one of there remote addresses are equal. This is like in TCP, where a connection
       is identified by the transport address, i.e. the IP-address and port of the peer. */
    if ( (((Association *)a)->remotePort == ((Association *)b)->remotePort) &&
         (((Association *)a)->localPort == ((Association *)b)->localPort) ){
        for (i = 0; i < ((Association *)a)->noOfNetworks; i++)
            for (j = 0; j < ((Association *)b)->noOfNetworks; j++) {
                if (adl_equal_address
                    (&(((Association *)a)->destinationAddresses[i]),
                     &(((Association *)b)->destinationAddresses[j])) == TRUE)
                    return 0;
            }
        return 1;
    }
    return 1;
}



/**
 * enterAssociation adds an association to the list.
 * @param assoc pointer to association data
 */
static void enterAssociation(Association *assoc)
{
    event_logi(INTERNAL_EVENT_0, "entering association %08x into list", assoc->tagLocal);

    AssociationList = g_list_insert_sorted(AssociationList,assoc, &compareAssociationIDs);
}



/**
 * retrieveAssociation retrieves a association from the list using the local tag as key.
 * Returns NULL also if the association is marked "deleted" !
 * @param assocID  association ID
 * @return  pointer to the retrieved association, or NULL
 */
Association *retrieveAssociation(unsigned int assocID)
{
    Association *assoc;
    Association *assocFindP;
    GList* result = NULL;

    event_logi(INTERNAL_EVENT_0, "retrieving association %08x from list", assocID);

    tmpAssoc.tagLocal = assocID;
    tmpAssoc.deleted = FALSE;
    assocFindP = &tmpAssoc;
    assoc = NULL;

    result = g_list_find_custom(AssociationList, assocFindP, &compareAssociationIDs);
    if (result != NULL) {

        assoc = (Association *)result->data;

        if (assoc->deleted) {
            assoc = NULL;
        }
    } else {
        event_logi(INTERNAL_EVENT_0, "association %08x not in list", assocID);
        assoc = NULL;
    }
    return assoc;
}



/**
 *  retrieveAssociationULP retrieves a association from the list using the local tag as key.
 *  This function is required, because for ULP calls at SCTP, it is not known if the association
 *  pointer is set or not. For instance, sctp_send() can result from an event originating from SCTP
 *  or for instance from a time event. In the first case the association must not be retrieved
 *  from the list, whereas in second it must.
 *  retrieveAssociationULP bypasses this problems by retrieving the association only if
 *  currentAssociation equals NULL. Furthermore the global parameter clearAssociation is
 *  set accordingly to signal the caller it should clear the currentAssociation after the
 *  ULP call has been handled.
 *
 *  @param assocID  association ID
 *  @return  pointer to the retrieved association, or NULL
 */
Association *retrieveAssociationULP(unsigned int assocID)
{
    Association *assoc;

    /* Retrieve association from list  */
    if (currentAssociation == NULL) {
        assoc = retrieveAssociation(assocID);
        clearAssociation = TRUE;
    } else {
        if (currentAssociation->tagLocal != assocID) {
            /* Darf nicht vorkommen */
            error_log(ERROR_FATAL, "retrieveAssociationULP: Assoc. set but ID's do not match");
        }
        clearAssociation = FALSE;
        assoc = currentAssociation;
    }
    return assoc;
}



/**
 *   retrieveAssociation retrieves a association from the list using the transport address as key.
 *   Returns NULL also if the association is marked "deleted" !
 *   CHECKME : Must return NULL, if no Address-Port combination does not occur in ANY existing assoc.
 *             If it occurs in one of these -> return it
 *
 *   @param  fromAddress address from which data arrived
 *   @param  fromPort SCTP port from which data arrived
 *   @return pointer to the retrieved association, or NULL
 */
Association *retrieveAssociationByTransportAddress(union sockunion * fromAddress,
                                                   unsigned short fromPort, unsigned short toPort)
{
    Association *assocr;
    Association *assocp;
    GList* result = NULL;

    tmpAssoc.noOfNetworks = 1;
    tmpAssoc.destinationAddresses = &tmpAddress;
    switch (sockunion_family(fromAddress)) {
    case AF_INET:
        event_logi(INTERNAL_EVENT_0,
                   "Looking for IPv4 Address %x, check NTOHX() ! ", sock2ip(fromAddress));
        tmpAssoc.destinationAddresses[0].sa.sa_family = AF_INET;
        tmpAssoc.destinationAddresses[0].sin.sin_addr.s_addr = sock2ip(fromAddress);
        tmpAssoc.remotePort = fromPort;
        tmpAssoc.localPort = toPort;
        tmpAssoc.deleted = FALSE;
        break;
#ifdef HAVE_IPV6
    case AF_INET6:
        tmpAssoc.destinationAddresses[0].sa.sa_family = AF_INET6;
        event_log(INTERNAL_EVENT_0, "Looking for IPv6 Address %x, check NTOHX() ! ");
        memcpy(tmpAssoc.destinationAddresses[0].sin6.sin6_addr.s6_addr,
               sock2ip6(fromAddress), sizeof(struct in6_addr));
        tmpAssoc.remotePort = fromPort;
        tmpAssoc.localPort = toPort;
        tmpAssoc.deleted = FALSE;
        break;
#endif
    default:
        error_logi(ERROR_FATAL,
                   "Unsupported Address Type %d in retrieveAssociationByTransportAddress()",
                   sockunion_family(fromAddress));
        break;

    }

    assocp = &tmpAssoc;

    event_log(INTERNAL_EVENT_0, "retrieving association by transport address from list");

    result = g_list_find_custom(AssociationList, assocp, equalAssociations);
    if (result != NULL){
        assocr = (Association *)result->data;
        if (assocr->deleted) assocr= NULL;
        if (assocr != NULL)
            event_logi(VERBOSE, "Found valid assoc assoc with id %u",assocr->tagLocal);
        return assocr;
    } else {
        event_log(INTERNAL_EVENT_0, "association indexed by transport address not in list");
    }
    return NULL;
}



/**
 *  checkForExistingAssociations checks wether a given association is already in the list using
 *  the equality condition given by function equalAssociations.
 *  TODO : this must still be implemented. Where is it used ??????????????
 *
 *  @param assoc_new the association to be compared with the association in the list.
 *  @return      1 if was association found, else  0
 */
static short checkForExistingAssociations(Association * assoc_new)
{

    /* get all Associations from the list that have the same ports ?! */

    /*  check if these have same number of transport addresses, and if these
       are equal for both associations. If so, the two associations are equal */
    error_log(ERROR_MAJOR,
              "TODO : checkForExistingAssociations() not implemented yet !");

    return 0;
}



/*------------------- Internal port management Functions -----------------------------------------*/

/**
 * seizePort return a free port number.
 * @return free port.
 */
static unsigned short seizePort(void)
{
    /* this is not nice, as you can guess the next port that will be chosen quite easily */
    /* maybe  replace that by a sort of bit mapped array, marking free/used ports,  that */
    /* we can choose randomly from ?!   */
    if (numberOfSeizedPorts >= 0xFBFF)
        return 0x0000;

    while (portsSeized[++lastPortSeized])
        if (lastPortSeized >= 0xFFFF)
            lastPortSeized = 0x00000400;

    numberOfSeizedPorts++;
    portsSeized[lastPortSeized] = 1;

    if (lastPortSeized > 0xFFFF)
        error_log(ERROR_MINOR, "Warning: lastPortSeized exceeds range of unsigned short");

    return (unsigned short) lastPortSeized;
}



/**
 * releasePort frees a previously used port.
 * @param portSeized port that is to be freed.
 */
static void releasePort(unsigned char portSeized)
{
    if (portsSeized[portSeized])
        error_log(ERROR_MINOR, "Warning: release of port that is not seized");

    numberOfSeizedPorts--;
    portsSeized[portSeized] = 0;
}



/*------------------- Other Internal Functions ---------------------------------------------------*/

/**
 * deleteAssociation removes the association from the list of associations, frees all data allocated
 *  for it and <calls moduleprefix>_delete*(...) function at all modules.
 *  @param assoc  pointer to the association to be deleted.
 */
static void mdi_deleteAssociation(Association * assoc)
{
    if (assoc != NULL) {
        event_logi(INTERNAL_EVENT_0, "Deleting association %08x ", assoc->tagLocal);

        /* free module data */
        if (assoc->tagRemote != 0) {
            /* association init was already completed */
            fc_delete_flowcontrol(assoc->flowControl);
            rtx_delete_reltransfer(assoc->reliableTransfer);
            rxc_delete_recvctrl(assoc->rx_control);
            se_delete_stream_engine(assoc->streamengine);
        }

        pm_deletePathman(assoc->pathMan);
        bu_delete(assoc->bundling);
        sci_deleteSCTP_control(assoc->sctp_control);
        releasePort(assoc->localPort);

        /* free association data */
        free(assoc->destinationAddresses);
        free(assoc);
    } else {
        error_log(ERROR_MAJOR, "mdi_deleteAssociation: association does not exist");
    }

    return;

}                               /* end: mdi_deleteAssociation */



/*------------------- Functions called by the Unix-Interface -------------------------------------*/


/**
 *  mdi_receiveMessage is the callback function of the SCTP-message distribution.
 *  It is called by the Unix-interface module when a new datagramm is received.
 *  This function also performs OOTB handling, tag verification etc.
 *  (see also RFC 2960, section 8.5.1.B)  and sends data to the bundling module of
 *  the right association
 *
 *  @param socket_fd          the socket file discriptor
 *  @param buffer             pointer to arrived datagram
 *  @param bufferlength       length of datagramm
 *  @param fromAddress        source address of DG
 *  @param portnum            bogus port number
 */
void
mdi_receiveMessage(gint socket_fd,
                   unsigned char *buffer,
                   int bufferLength,
                   unsigned char *hoststring,
                   unsigned short fromAddressLength)
{
    SCTP_message *message;
    SCTP_simple_chunk *chunk = NULL;
    SCTP_init_fixed *initChunk = NULL;
    guchar* initPtr;
    union sockunion fromAddress;
    unsigned int i, len, state;
    boolean sourceAddressExists;
    boolean sendAbort = FALSE;

    boolean initFound = FALSE, cookieEchoFound = FALSE, abortFound = FALSE;

    short shutdownCompleteCID;
    short abortCID;

    SCTP_instance temporary;
    GList* result = NULL;


    adl_str2sockunion(hoststring, &fromAddress);

    /* FIXME:  check this out, oif it works at all :-D */
    lastFromAddress = &fromAddress;

    lastFromPath = 0;


    message = (SCTP_message *) buffer;

    if (!validate_datagram(buffer, bufferLength)) {
        event_log(INTERNAL_EVENT_0, "received corrupted datagramm");
        lastFromAddress = NULL;
        return;
    }

    len = bufferLength - sizeof(SCTP_common_header);

    /* save from address for response if a remote address is not available otherwise.
       For instance initAck or cookieAck. */
    lastFromPort = ntohs(message->common_header.src_port);
    lastDestPort = ntohs(message->common_header.dest_port);

    event_logiiii(EXTERNAL_EVENT,
                  "mdi_receiveMessage : len %d, sourceaddress : %s, src_port %u,dest_port %u",
                  bufferLength, hoststring, lastFromPort, lastDestPort);

    /* OK - if this packet is for a server, we will find an SCTP instance, that shall
       handle it (i.e. we have the SCTP instance's localPort set and it matches the
       packet's destination port */
    temporary.localPort = lastDestPort;
    result = g_list_find_custom(InstanceList, &temporary, &CompareInstancePorts);
    if (result == NULL) {
        event_logi(VERBOSE, "Couldn't find SCTP Instance for Port %u in List !",lastDestPort);
        sctpInstance = NULL;
    }else{
        sctpInstance = result->data;
    }

    if (lastFromPort == 0 || lastDestPort == 0) {
        error_log(ERROR_MINOR, "received DG with invalid (i.e. 0) ports");
        lastFromAddress = NULL;
        lastFromPort = 0;
        lastDestPort = 0;
        return;
    }

    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationByTransportAddress(lastFromAddress, lastFromPort,lastDestPort);

    lastInitiateTag = ntohl(message->common_header.verification_tag);

    chunk = (SCTP_simple_chunk *) & message->sctp_pdu[0];

    /* check if sctp-message belongs to an existing association */
    if (currentAssociation == NULL) {
        if (sctpInstance == NULL) {
            event_logi(EXTERNAL_EVENT, "We have not registered an SCTP-Instance for Port %u - Dropping Packet !",lastDestPort);
            /* FIXME : Do we have to send an ABORT here ?????????? */
            return;
        }
        event_log(VVERBOSE, "mdi_receiveMsg: currentAssociation==NULL, start scanning !");
        /* This is not very elegant, but....only used when assoc is being build up, so :-D */
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_ABORT) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Found ABORT chunk, discarding it !");
            lastFromAddress = NULL;
            lastFromPort = 0;
            lastDestPort = 0;
            return;
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_SHUTDOWN_ACK) == TRUE) {
            event_log(INTERNAL_EVENT_0,
                      "mdi_receiveMsg: Found SHUTDOWN_ACK chunk, send SHUTDOWN_COMPLETE !");
            /* section 8.4.5 : return SHUTDOWN_COMPLETE with peers veri-tag and T-Bit set */
            shutdownCompleteCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NO_TCB);
            bu_put_Ctrl_Chunk(ch_chunkString(shutdownCompleteCID));
            /* should send it to last address */
            bu_sendAllChunks(NULL);
            /* free abort chunk */
            ch_deleteChunk(shutdownCompleteCID);

            /* send an ABORT with peers veri-tag, set T-Bit */
            event_log(VERBOSE, "mdi_receiveMsg: sending CHUNK_SHUTDOWN_COMPLETE  ");
            lastFromPort = 0;
            lastDestPort = 0;
            return;
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_SHUTDOWN_COMPLETE) == TRUE) {
            event_log(INTERNAL_EVENT_0,
                      "mdi_receiveMsg: Found SHUTDOWN_COMPLETE chunk, discarding it !");
            lastFromPort = 0;
            lastDestPort = 0;
            lastFromAddress = NULL;
            return;
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_COOKIE_ACK) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Found COOKIE_ACK chunk, discarding it !");
            lastFromPort = 0;
            lastDestPort = 0;
            lastFromAddress = NULL;
            return;
        }

        /* section 8.4.7) : Discard the datagram, if it contains a STALE-COOKIE ERROR */
        if (rbu_scanDatagramForError(message->sctp_pdu, len, ECC_STALE_COOKIE_ERROR) == TRUE) {
            event_log(INTERNAL_EVENT_0,
                      "mdi_receiveMsg: Found STALE COOKIE ERROR, discarding packet !");
            lastFromPort = 0;
            lastDestPort = 0;
            lastFromAddress = NULL;
            return;
        }
        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT)) != NULL) {
            if (lastDestPort != sctpInstance->localPort || sctpInstance->localPort == 0) {
                /* destination port is not the listening port of this this SCTP-instance. */
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: INIT Message ignored, dest. port does not fit");
                sendAbort = TRUE;
                /* as per section 5.1 :
                   If an endpoint receives an INIT, INIT ACK, or COOKIE ECHO chunk but
                   decides not to establish the new association due to missing mandatory
                   parameters in the received INIT or INIT ACK, invalid parameter values,
                   or lack of local resources, it MUST respond with an ABORT chunk */
            } else {
                event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: INIT Message - processing it !");

                /* TODO : check for all src address parameters in INIT, whether they correspond
                   to an association. If there are addresses previously unknown (that may have
                   been added or changed) discard the INIT. Might be an attack then.
                 */

            }
            initChunk = ((SCTP_init_fixed *) & ((SCTP_init *) message->sctp_pdu)->init_fixed);
            lastInitiateTag = ntohl(initChunk->init_tag);
            event_logi(VERBOSE, "setting lastInitiateTag to %x ", lastInitiateTag);

        } else if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_COOKIE_ECHO) == TRUE) {

            if (lastDestPort != sctpInstance->localPort || sctpInstance->localPort == 0) {
                /* destination port is not the listening port of this this SCTP-instance. */
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: COOKIE_ECHO ignored, dest. port does not fit");
                sendAbort = TRUE;
            } else {
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: COOKIE_ECHO Message - processing it !");
            }

        } else {
            /* section 8.4.8) send an ABORT with peers veri-tag, set T-Bit */
            event_log(INTERNAL_EVENT_0,
                      "mdi_receiveMsg: send ABORT -> message ignored (OOTB - see section 8.4.8) ");
            sendAbort = TRUE;
        }

        if (sendAbort == TRUE) {
            /* make and send abort message */
            abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NO_TCB);
            bu_put_Ctrl_Chunk(ch_chunkString(abortCID));
            /* should send it to last address */
            bu_sendAllChunks(NULL);
            /* free abort chunk */
            ch_deleteChunk(abortCID);
            /* send an ABORT with peers veri-tag, set T-Bit */
            event_log(VERBOSE, "mdi_receiveMsg: sending ABORT with T-Bit");
            lastFromAddress = NULL;
            lastFromPort = 0;
            lastDestPort = 0;
            /* and discard that packet */
            return;
        }

    } else { /* i.e. if(currentAssociation != NULL) */

        /* If the association exists, both ports of the message must be equal to the ports
           of the association and the source address must be in the addresslist of the peer
           of this association */
        /* check src- and dest-port and source address */
        if (lastFromPort != currentAssociation->remotePort
            || lastDestPort != currentAssociation->localPort) {
            error_logiiii(ERROR_FATAL,
                          "port mismatch in received DG (lastFromPort=%u, assoc->remotePort=%u, lastDestPort=%u, assoc->localPort=%u ",
                          lastFromPort, currentAssociation->remotePort,
                          lastDestPort, currentAssociation->localPort);
            currentAssociation = NULL;
            lastFromAddress = NULL;
            lastFromPort = 0;
            lastDestPort = 0;
            return;
        }

        if (sctpInstance == NULL) {
            /* retrieve SCTP-instance with SCTP-instance name in current association */
            temporary.sctpInstanceName = currentAssociation->sctpInstance->sctpInstanceName;
            event_logi(VERBOSE, "Searching for SCTP Instance with Name %u ", currentAssociation->sctpInstance->sctpInstanceName);
            result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
            if (result == NULL) {
                error_logi(ERROR_FATAL, "We have an Association, but no Instance with name %u in List, FIXME !",
                    currentAssociation->sctpInstance->sctpInstanceName);
            }
            sctpInstance = result->data;
        }

        /* check if source address is in address list of this association.
           tbd: check the draft if this is correct. */
        for (i = 0, sourceAddressExists = FALSE; i < currentAssociation->noOfNetworks; i++) {
            if (adl_equal_address
                (&(currentAssociation->destinationAddresses[i]), lastFromAddress) == TRUE) {
                sourceAddressExists = TRUE;
                break;
            }
        }

        if (!sourceAddressExists) {
            error_log(ERROR_MINOR,
                      "source address of received DG is not in the destination addresslist");
            currentAssociation = NULL;
            lastFromPort = 0;
            lastDestPort = 0;
            lastFromAddress = NULL;
            return;
        }

        if (sourceAddressExists)
            lastFromPath = i;

        /* check for verification tag rules --> see section 8.5 */
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_INIT) == TRUE) {
            /* check that there is ONLY init */
            initFound = TRUE;
            if (lastInitiateTag != 0) {
                currentAssociation = NULL;
                lastFromPort = 0;
                lastDestPort = 0;
                lastFromAddress = NULL;
                event_log(VERBOSE, "mdi_receiveMsg: scan found INIT, lastInitiateTag!=0, returning");
                return;
            }
            initChunk = ((SCTP_init_fixed *) & ((SCTP_init *) message->sctp_pdu)->init_fixed);
            /* make sure, if you send an ABORT later on (i.e. when peer requests 0 streams),
             * you pick the right tag */
            lastInitiateTag = ntohl(initChunk->init_tag);
            event_logi(VVERBOSE, "Got an INIT CHUNK with initiation-tag %u", lastInitiateTag);

        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_ABORT) == TRUE) {
            /* accept my-tag or peers tag, else drop packet */
            if ((lastInitiateTag != currentAssociation->tagLocal &&
                 lastInitiateTag != currentAssociation->tagRemote) || initFound == TRUE) {
                currentAssociation = NULL;
                lastFromPort = 0;
                lastDestPort = 0;
                lastFromAddress = NULL;
                return;
            }
            abortFound = TRUE;
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_SHUTDOWN_COMPLETE) == TRUE) {
            /* accept my-tag or peers tag, else drop packet */
            /* TODO : make sure that if it is the peer's tag also T-Bit is set */
            if ((lastInitiateTag != currentAssociation->tagLocal &&
                 lastInitiateTag != currentAssociation->tagRemote) || initFound == TRUE) {
                currentAssociation = NULL;
                lastFromPort = 0;
                lastDestPort = 0;
                lastFromAddress = NULL;
                return;
            }
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_SHUTDOWN_ACK) == TRUE) {
            if (initFound == TRUE) {
                currentAssociation = NULL;
                lastFromPort = 0;
                lastDestPort = 0;
                lastFromAddress = NULL;
                return;
            }
            state = sci_getState();
            if (state == COOKIE_ECHOED || state == COOKIE_WAIT) {
                /* see also section 8.5.E.) treat this like OOTB packet */
                event_logi(EXTERNAL_EVENT,
                           "mdi_receive_message: shutdownAck in state %u, send SHUTDOWN_COMPLETE ! ",
                           state);
                shutdownCompleteCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NO_TCB);
                bu_put_Ctrl_Chunk(ch_chunkString(shutdownCompleteCID));
                bu_sendAllChunks(NULL);
                ch_deleteChunk(shutdownCompleteCID);
                currentAssociation = NULL;
                lastFromPort = 0;
                lastDestPort = 0;
                lastFromAddress = NULL;
                return;
            }
        }
        if (rbu_scanDatagram(message->sctp_pdu, len, CHUNK_COOKIE_ECHO) == TRUE) {
               cookieEchoFound = TRUE;
        }

        if (!cookieEchoFound && !initFound && !abortFound && lastInitiateTag != currentAssociation->tagLocal) {
            error_logii(ERROR_MINOR,
                        "Tag mismatch in receive DG, received Tag = %u, local Tag = %u -> discarding",
                        lastInitiateTag, currentAssociation->tagLocal);

            currentAssociation = NULL;
            lastFromPort = 0;
            lastDestPort = 0;
            lastFromAddress = NULL;
            return;

        }

    }


    /* printString("Processing received SCTP-message : ", buffer, bufferLength); */

    /* forward DG to bundling */
    rbu_rcvDatagram(lastFromPath, message->sctp_pdu, bufferLength - sizeof(SCTP_common_header));

    lastInitiateTag = 0;
    currentAssociation = NULL;
    lastFromAddress = NULL;
    lastFromPath = -1;          /* only valid for functions called via mdi_receiveMessage */

}                               /* end: mdi_receiveMessage */




/*------------------- Functions called by the ULP ------------------------------------------------*/
/*------------------- Prototypes are defined in sctp.h -------------------------------------------*/

/**
 *  sctp_registerInstance is called to initialize one SCTP-instance.
 *  Each Adaption-Layer of the ULP must create its own SCTP-instance, and
 *  define and register appropriate callback functions.
 *  An SCTP-instance may define an own port, or zero here ! Servers and clients
 *  that care for their source port must chose a port, clients that do not really
 *  care which source port they use, chose ZERO, and have the implementation chose
 *  a free source port.
 *
 *  @param port                   wellknown port of this sctp-instance
 *  @param noOfLocalAddresses     number of local addresses
 *  @param localAddressList       local address list (pointer to a string-array)
 *  @param ULPcallbackFunctions   call back functions for primitives passed from sctp to ULP
 *  @return     instance name of this SCTP-instance or 0 in case of errors
 */
unsigned short
sctp_registerInstance(unsigned short port,
                           unsigned short noOfInStreams,
                           unsigned short noOfOutStreams,
                           unsigned int noOfLocalAddresses,
                           unsigned char localAddressList[][SCTP_MAX_IP_LEN],
                           SCTP_ulpCallbacks ULPcallbackFunctions)
{

    int adl_rscb_code;
    int i;
    struct timeval curTime;
    union sockunion su;
    gboolean with_ipv4 = FALSE;


#ifdef HAVE_IPV6
    gboolean with_ipv6 = FALSE;
#endif

    if ((port == 0) || (noOfInStreams==0) || (noOfOutStreams == 0) ||
        (noOfLocalAddresses == 0) || (localAddressList == NULL)) {
            error_log(ERROR_MAJOR, "Parameter Problem in sctp_registerInstance - Error !");
            return 0;
    }

    for (i=0; i< noOfLocalAddresses; i++) {
        if (adl_str2sockunion((localAddressList[i]), &su) < 0) {
            error_logi(ERROR_MAJOR, "Address Error in sctp_registerInstance(%s)", (localAddressList[i]));
            return 0;
        } else {
            if (su.sa.sa_family == AF_INET) with_ipv4 = TRUE;

#ifdef HAVE_IPV6
            if (su.sa.sa_family == AF_INET6) with_ipv6 = TRUE;
#endif
        }
    }

    event_logi(VERBOSE, "sctp_registerInstance : with_ipv4 : %s ",(with_ipv4==TRUE)?"TRUE":"FALSE" );
    /* if not IPv6 callback must be registered too ! */
#ifdef HAVE_IPV6
    event_logi(VERBOSE, "sctp_registerInstance : with_ipv6: %s ",(with_ipv6=TRUE)?"TRUE":"FALSE" );
#endif

    if ((with_ipv4 != TRUE)
#ifdef HAVE_IPV6
            && (with_ipv6 != TRUE)
#endif
                              ) {
            error_log(ERROR_MAJOR, "No valid address in sctp_registerInstance()");
            return 0;
    }

    sctpInstance = (SCTP_instance *) malloc(sizeof(SCTP_instance));
    if (!sctpInstance) {
        error_log_sys(ERROR_MAJOR, errno);
        return 0;
    }

    if (first_instance == TRUE) {

        first_instance = FALSE;
        /* initialize the output of event/error-log functions */
        read_tracelevels();

        event_log(EXTERNAL_EVENT, "sctp_registerInstance called");
        adl_init_adaptation_layer();

        /* Create list for associations - old, used to be here - now removed ! */

        /* initialize ports seized -- see comments above !!! */
        for (i = 0; i < 0x10000; i++)
            portsSeized[i] = 0;
        lastPortSeized = 0x00000400;
        numberOfSeizedPorts = 0x00000000;

        /* initialize random number generator */
        gettimeofday(&curTime, NULL);
        rstate[0] = curTime.tv_sec;
        rstate[1] = curTime.tv_usec;
        initstate(curTime.tv_sec, (char *) rstate, 8);
        setstate((char *) rstate);

        /* initialize bundling, i.e. the common buffer for sending chunks when no association
           exists. */
        bu_init_bundling();

        /* this block is to be executed only once for the lifetime of sctp-software */
        key_operation(KEY_INIT);

    }

#ifdef HAVE_IPV6
    if (with_ipv6 && ipv6_sctp_socket==0) {
         ipv6_sctp_socket = adl_get_sctpv6_socket();
         if (!ipv6_sctp_socket)
            error_log(ERROR_FATAL, "IPv6 socket creation failed");
        /*
         * here some operating system specialties may kick in (i.e. opening only ONE
         * socket MIGHT be enough, provided IPv6 socket implicitly reveives IPv4 packets, too
         */
         adl_rscb_code = adl_register_socket_cb(ipv6_sctp_socket,&mdi_receiveMessage);
         if (!adl_rscb_code)
             error_log(ERROR_FATAL, "register ipv6 socket call back function failed");
     }
    if (with_ipv6 == TRUE) ipv6_users++;
#endif
    if (with_ipv4 && sctp_socket==0) {
         sctp_socket = adl_get_sctpv4_socket();
         if (!sctp_socket)
            error_log(ERROR_FATAL, "IPv4 socket creation failed");
         adl_rscb_code = adl_register_socket_cb(sctp_socket,&mdi_receiveMessage);
         if (!adl_rscb_code)
             error_log(ERROR_FATAL, "registration of IPv4 socket call back function failed");
    }
    if (with_ipv4 == TRUE) ipv4_users++;

    sctpInstance->sctpInstanceName = lastSCTP_instanceName++;
    if (sctpInstance->sctpInstanceName == 0) {
        sctpInstance->sctpInstanceName = lastSCTP_instanceName++;
    }
    sctpInstance->localPort = port;
    sctpInstance->noOfInStreams = noOfInStreams;
    sctpInstance->noOfOutStreams = noOfOutStreams;

    sctpInstance->localAddressList =
            (union sockunion *) malloc(noOfLocalAddresses * sizeof(union sockunion));

    for (i=0; i< noOfLocalAddresses; i++) {
        adl_str2sockunion(localAddressList[i], &(sctpInstance->localAddressList[i]));
    }

    sctpInstance->noOfLocalAddresses = noOfLocalAddresses;

    sctpInstance->ULPcallbackFunctions = ULPcallbackFunctions;

    sctpInstance->default_rtoInitial = RTO_INITIAL;
    sctpInstance->default_validCookieLife = VALID_COOKIE_LIFE_TIME;
    sctpInstance->default_assocMaxRetransmits = ASSOCIATION_MAX_RETRANS;
    sctpInstance->default_pathMaxRetransmits = MAX_PATH_RETRANSMITS ;
    sctpInstance->default_maxInitRetransmits = MAX_INIT_RETRANSMITS;
    sctpInstance->default_myRwnd = RWND_CONST;
    sctpInstance->default_delay = SACK_DELAY;
    sctpInstance->default_ipTos = IPTOS_DEFAULT;
    sctpInstance->default_rtoMin = RTO_MIN;
    sctpInstance->default_maxSendQueue = 0;
    sctpInstance->default_maxRecvQueue = 0;

    InstanceList = g_list_insert_sorted(InstanceList, sctpInstance, &CompareInstanceNames);

    return sctpInstance->sctpInstanceName;

}                               /* end: sctp_registerInstance */



int sctp_unregisterInstance(unsigned short instance_name)
{
    // Look through the instance list, and delete instance, when
    // found, else return error.
    SCTP_instance temporary;
    SCTP_instance* instance;
    union sockunion* alist;
    guint16 num_addresses;
    guint32 n, fds;
    GList* result = NULL;
    gboolean with_ipv4=FALSE;
#ifdef HAVE_IPV6
    gboolean with_ipv6=FALSE;
#endif

    // FIXME : If last instance is removed, shouldn't we terminate program ????

    event_logi(INTERNAL_EVENT_0, "Removing SCTP Instance %u from list", instance_name);

    temporary.sctpInstanceName = instance_name;
    result = g_list_find_custom(InstanceList, &temporary,&CompareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
        alist = instance->localAddressList;
        num_addresses = instance->noOfLocalAddresses;
        for (n=0; n< num_addresses; n++) {
            if (sockunion_family(alist) == AF_INET) with_ipv4 = TRUE;

#ifdef HAVE_IPV6
            if (sockunion_family(alist) == AF_INET6) with_ipv6 = TRUE;
#endif
            /* this is pointer arithmetic, ughhh, ugly :-) */
            alist++;
        }
        event_logi(INTERNAL_EVENT_0, "sctp_unregisterInstance: SCTP Instance %u found !!!", instance_name);
#ifdef HAVE_IPV6
        event_logi(VERBOSE, "sctp_unregisterInstance : with_ipv6: %s ",(with_ipv6=TRUE)?"TRUE":"FALSE" );
        if (with_ipv6 == TRUE) ipv6_users--;
        event_logi(VERBOSE, "sctp_unregisterInstance : ipv6_users: %u ",ipv6_users);
#endif
        if (with_ipv4 == TRUE) ipv4_users--;
        event_logi(VERBOSE, "sctp_unregisterInstance : with_ipv4: %s ",(with_ipv4=TRUE)?"TRUE":"FALSE" );
        event_logi(VERBOSE, "sctp_unregisterInstance : ipv4_users: %u ",ipv4_users);

        if (sctp_socket != 0 &&  ipv4_users == 0) {
            fds = adl_remove_poll_fd(sctp_socket);
            event_logi(VVERBOSE, "sctp_unregisterInstance : Removed IPv4 cb, registered FDs: %u ",fds);
            /* if there are no ipv4_users, deregister callback for ipv4-socket, if it was registered ! */
            sctp_socket = 0;
        }
#ifdef HAVE_IPV6
        if (ipv6_sctp_socket != 0 &&  ipv6_users == 0) {
            fds = adl_remove_poll_fd(ipv6_sctp_socket);
           /* if there are no ipv6_users, deregister callback for ipv6-socket, if it was registered ! */
            event_logi(VVERBOSE, "sctp_unregisterInstance : Removed IPv4 cb, registered FDs: %u ",fds);
            ipv6_sctp_socket = 0;
        }
#endif
        event_log(INTERNAL_EVENT_0, "FIXME: are s there till associations with my port ? I remove instance now !");
        InstanceList = g_list_remove(InstanceList, result->data);
        return 0;
    } else {
        event_logi(INTERNAL_EVENT_0, "SCTP Instance %u not in list", instance_name);
    }
    return -1;
}


/**
 * This function should be called AFTER an association has indicated a
 * COMMUNICATION_LOST or a SHUTDOWN_COMPLETE, and the upper layer has
 * retrieved all data it is interested in (possibly using the currently
 * not implemented functions  sctp_receive_unsent() or sctp_receive_unacked())
 * it really removes all data belonging to the association, and removes the
 * association instance from the list, on explicit upper layer instruction !
 * @param  associationID the association ID of the assoc that shall be removed
 * @return error_code  0 for success, 1 if assoc is already gone, -1 if assocs
 *         deleted flag is not set (then assoc should be in a state different from CLOSED)
 */
int sctp_deleteAssociation(unsigned int associationID)
{
    Association *assoc;
    Association *assocFindP;
    GList* result = NULL;

    event_logi(INTERNAL_EVENT_0, "sctp_deleteAssociation: getting assoc %08x from list", associationID);

    tmpAssoc.tagLocal = associationID;
    tmpAssoc.deleted = FALSE;
    assocFindP = &tmpAssoc;
    assoc = NULL;

    result = g_list_find_custom(AssociationList, assocFindP, &compareAssociationIDs);
    if (result != NULL) {
        assoc = (Association *)result->data;
        if (!assoc->deleted) return -1;
        /* remove the association from the list */
        AssociationList = g_list_remove(AssociationList, assoc);
        /* free all association data */
        mdi_deleteAssociation(assoc);
        return 0;
    } else {
        event_logi(INTERNAL_EVENT_0, "association %08x not in list", associationID);
        return 1;
    }
    /* should not be reached */
    return 0;
}


/**
 * This function is called to setup an association.
 *  The ULP must specify the SCTP-instance to which this association belongs to.
 *  @param SCTP_InstanceName     the SCTP instance this association belongs to.
 *                               if the local port of this SCTP instance is zero, we will get a port num,
                                 else we will use the one from the SCTP instance !
 *  @param noOfOutStreams        number of output streams the ULP would like to have
 *  @param destinationAddress    destination address
 *  @param destinationPort       destination port
 *  @param ulp_data             pointer to an ULP data structure, will be passed with callbacks !
 *  @return association ID of this association (identical with local tag), 0 in case of failures
 */
unsigned int
sctp_associate(unsigned short SCTP_InstanceName,
               unsigned short noOfOutStreams,
               unsigned char destinationAddress[],
               unsigned short destinationPort,
               void* ulp_data)
{
    unsigned int assocID;
    unsigned short zlocalPort;
    union sockunion dest_su;
    SCTP_instance temporary;
    GList* result = NULL;

    if (currentAssociation != NULL) {
        error_log(ERROR_MINOR, "sctp_associate: association not cleared");
        return 0;
    }
    if (adl_str2sockunion(destinationAddress, &dest_su) < 0) {
        error_log(ERROR_MAJOR, "sctp_associate: could not convert destination adress");
        return 0;
    }
    event_log(EXTERNAL_EVENT, "sctp_associate called");
    event_logi(VERBOSE, "Looking for SCTP Instance %u in the list", SCTP_InstanceName);

    temporary.sctpInstanceName =  SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);

    if (result != NULL) {
        if (((SCTP_instance*)result->data)->localPort == 0)
            zlocalPort = seizePort();
        else
            zlocalPort = ((SCTP_instance*)result->data)->localPort;
    } else {
        error_log(ERROR_MAJOR, "sctp_associate: SCTP instance not in the list !!!");
        return 0;
    }
    event_logi(VERBOSE, "Chose local port of %u for associate !", zlocalPort);

    /* Create new association */
    if (mdi_newAssociation(SCTP_InstanceName, zlocalPort, /* local client port */
                           destinationPort, /* remote server port */
                           mdi_generateLocalTag(), 0, 1, &dest_su)) {
        error_log(ERROR_MAJOR, "Creation of association failed");
        return 0;
    }
    currentAssociation->ulp_dataptr = ulp_data;
    /* call associate at SCTP-control */
    scu_associate(noOfOutStreams, ((SCTP_instance*)result->data)->noOfInStreams);

    assocID = currentAssociation->tagLocal;

    /* rest currentAssociation */
    currentAssociation = NULL;

    return assocID;
}                               /* end: sctp_associate */


/**
 * sctp_shutdown initiates the shutdown of the specified association.
 *  @param    associationID  the ID of the addressed association.
 *  @return   0 for success, 1 for error (assoc. does not exist)
 */
int sctp_shutdown(unsigned int associationID)
{
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        /* Forward shutdown to the addressed association */
        scu_shutdown();
    } else {
        error_log(ERROR_MAJOR, "sctp_shutdown: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return 0;

}                               /* end: sctp_shutdown */



/**
 * sctp_abort initiates the abort of the specified association.
 * @param    associationID  the ID of the addressed association.
 * @return   0 for success, 1 for error (assoc. does not exist)
 */
int sctp_abort(unsigned int associationID)
{
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        /* Forward shutdown to the addressed association */
        scu_abort();
    } else {
        error_log(ERROR_MAJOR, "sctp_abort: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return 0;

}                               /* end: sctp_abort */



/**
 * sctp_send is used by the ULP to send data chunks.
 *
 *  @param    associationID  the ID of the addressed association.
 *  @param    streamID       identifies the stream on which the chunk is sent.
 *  @param    buffer         chunk data.
 *  @param    length         length of chunk data.
 *  @param    protocolId     the payload protocol identifier
 *  @param    path_id        index of destination address, if different from primary pat, negative for primary
 *  @param    context        ULP context, i.e. a pointer that will may be retunred with certain callbacks.
                             (in case of send errors).
 *  @param    lifetime       maximum time of chunk in send queue in msecs, 0 for infinite
 *  @param    unorderedDelivery chunk is delivered to peer without resequencing, if true (==1), else ordered (==0).
 *  @param    dontBundle     chunk must not be bundled with other data chunks.
 *                           boolean, 0==normal bundling, 1==do not bundle message
 *  @return   error code     -1 for send error, 1 for association error, 0 if successful
 */
int sctp_send(unsigned int associationID, unsigned short streamID,
                   unsigned char *buffer, unsigned int length, unsigned int protocolId, short path_id,
                   void*  context, /* optional (=SCTP_NO_CONTEXT=NULL if none) */
                   unsigned int lifetime, /* optional (zero -> infinite) */
                   int unorderedDelivery, /* boolean, 0==ordered, 1==unordered */
                   int dontBundle)      /* boolean, 0==normal bundling, 1==do not bundle message */
{
    int result = 0;
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        if (path_id >= 0) {
            if (path_id >= currentAssociation->noOfNetworks) {
                error_log(ERROR_MAJOR, "sctp_send: invalid destination address");
                return 1;
            }
        }
        event_log(INTERNAL_EVENT_1, "sctp_send: sending chunk");
        /* Forward chunk to the addressed association */
        result = se_ulpsend(streamID, buffer, length, protocolId, path_id,
                      context, lifetime, unorderedDelivery, dontBundle);

    } else {
        error_log(ERROR_MAJOR, "sctp_send: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return result;
}                               /* end: sctp_send */



/**
 * sctp_setPrimary changes the primary path of an association.
 * @param  associationID     ID of assocation.
 * @param  destAddressIndex  index to the new primary path
 * @return error code
 */
short sctp_setPrimary(unsigned int associationID, short path_id)
{
    short rv;
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        /* Forward shutdown to the addressed association */
        rv = pm_setPrimaryPath(path_id);
    } else {
        error_log(ERROR_MAJOR, "sctp_setPrimary: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return rv;

}                               /* end: sctp_setPrimary */




/**
 * sctp_receive is called in response to the dataArriveNotification to
 * get the received data.
 * The stream engine must copy the chunk data from a received  SCTP datagram to
 * a new byte string, because the SCTP datagram is overwritten when the next datagram
 * is received and the lifetime of a chunk in the streamengine might outlast the
 *  the reception of several SCTP datagrams.
 *  For this reasons and to avoid repeated copying of byte strings, a pointer to
 *  the byte string of chunkdata allocated by the streamengine is returned.
 *  According to the standard, the chunkdata should be copied to to a buffer provided
 *  by the ULP.
 *  @param   associationID  ID of association.
 *  @param   streamID       the stream on which the data chunk is received.
 *  @param   buffer         pointer to where payload data of arrived chunk will be copied
 *  @param   length         length of chunk data.
 *  @return  1 if association does not exist, 0 if okay
*/
unsigned short
sctp_receive(unsigned int associationID,
             unsigned short streamID, unsigned char *buffer, unsigned int *length)
{
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        /* retrieve data from streamengine instance */
        se_ulpreceive(buffer, length, streamID);
    } else {
        error_log(ERROR_MAJOR, "sctp_receive: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return 0;
}                               /* end: sctp_receive */



/**
 * sctp_changeHeartBeat turns the hearbeat on a path of an association on or
 * off, or modifies the interval
 * @param  associationID   ID of assocation.
 * @param  path_id         index of the path where to do heartbeat
 * @param  heartbeatON     turn heartbeat on or off
 * @param  timeIntervall   heartbeat time intervall in milliseconds
 * @return error code, 0 for success, 1 for error
 */
int
sctp_changeHeartBeat(unsigned int associationID,
                     short path_id, gboolean heartbeatON, unsigned int timeIntervall)
{
    int result;
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        /* Forward change HB to the addressed association */
        if (heartbeatON) {
            result = pm_enableHB(path_id, timeIntervall);
            event_logiii(VERBOSE,
                        "Setting HB interval for address %d to %u msecs, result: %d !",
                        path_id, timeIntervall, result);
        } else
            result = pm_disableHB(path_id);
            event_logii(VERBOSE,
                        "Disabling HB for address %d, result: %d !",
                        path_id, result);
    } else {
        error_log(ERROR_MAJOR, "sctp_changeHeartBeat: addressed association does not exist");
        return 1;
    }

    /* clear association */
    if (clearAssociation)
        currentAssociation = NULL;

    return result;

}                               /* end: sctp_changeHeartBeat */



/**
 * sctp_requestHeartbeat sends a heartbeat to the given address of an association.
 * @param  associationID  ID of assocation.
 * @param  path_id        destination address to which the heartbeat is sent.
 * @return error code (0 == success, 1 == error)
 */
int sctp_requestHeartbeat(unsigned int associationID, short path_id)
{
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        pm_doHB(path_id);
        event_logi(VERBOSE, "Sending HB on user request to path ID: %u !",
                        path_id);
    } else {
        error_log(ERROR_MAJOR, "sctp_requestHeartbeat: addressed association does not exist");
        return 1;
    }
    /* clear association */
    if (clearAssociation) currentAssociation = NULL;

    return 0;
}                               /* sctp_requestHeartbeat */

/**
 * sctp_getSrttReport returns a smoothed RTT value for a path to a given address
 * @param  associationID    ID of assocation.
 * @param  destAddressIndex destination address where to get SRTT from
 * @return SRTT of the address in msecs, 0 on error
 */
unsigned int sctp_getSrttReport(unsigned int associationID, short path_id)
{
    unsigned int srtt;
    /* Retrieve association from list  */
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL) {
        srtt = pm_readSRTT(path_id);
        event_logiii(VERBOSE, "sctp_getSrttReport(asoc=%u, address=%d) result: %u !",
                        associationID, path_id, srtt);
        if (srtt==0xffffffff) return 0;
        else return srtt;
    } else {
        error_log(ERROR_MAJOR, "sctp_getSrttReport: addressed association does not exist");
        return 0;
    }
    /* clear association */
    if (clearAssociation) currentAssociation = NULL;

    return 0;

}


/**
 *  sctp_setFailureThreshold is currently NOT implemented !
 *  Will be used to set the threshold for retransmissions on the given address of an
 *  association. If the threshold is exeeded, the the destination address is considered as
 *  unreachable.
 *  @param  associationID :            ID of assocation.
 *  @param  destAddressIndex :         destination address that gets a new failure threshold.
 *  @param  pathMaxRetransmissions :   threshold for retransmissions.
 *  @return -1
 */
int
sctp_setFailureThreshold(unsigned int associationID, unsigned short pathMaxRetransmissions)
{
    guint16 result;
    event_logii(VERBOSE, "sctp_setFailureThreshold: Association %u, pathMaxRetr. %u", associationID, pathMaxRetransmissions);

    result = mdi_setAssociationData(associationID);
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL && result == 0 ) {
        pm_setMaxPathRetransmisions(pathMaxRetransmissions);
    } else {
        error_logi(ERROR_MAJOR, "sctp_setFailureThreshold : association %u does not exist", associationID);
    }
    result = mdi_clearAssociationData(associationID);
    return result;

}                               /* end:  sctp_setFailureThreshold */



/**
 * sctp_getPathStatus : IP_TOS support is still missing !
 * Can be used to get path specific parameters in an existing association.
 *
 *  @param  associationID   ID of assocation.
 *  @param  path_id         path for which to get parameters
 *  @param  status      pointer to new parameters
 *  @return 0 for success, not zero for error
 */
int sctp_getPathStatus(unsigned int associationID, short path_id, SCTP_PathStatus* status)
{

    guint16 result;
    event_logii(VERBOSE, "sctp_getPathStatus: Association %u, Path %u", associationID, path_id);

    if (status == NULL) return -1;
    result = mdi_setAssociationData(associationID);

    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL && result == 0 && path_id >= 0 && path_id< currentAssociation->noOfNetworks) {
        adl_sockunion2str(&(currentAssociation->destinationAddresses[path_id]),
                          &(status->destinationAddress[0]), SCTP_MAX_IP_LEN);
        status->state = pm_readState(path_id);
        status->srtt = pm_readSRTT(path_id);
        status->rto = pm_readRTO(path_id);
        status->rttvar = pm_readRttVar(path_id);
        status->cwnd = fc_readCWND(path_id);
        status->cwnd2 = fc_readCWND2(path_id);
        status->partialBytesAcked = fc_readPBA(path_id);
        status->ssthresh = fc_readSsthresh(path_id);
        status->outstandingBytesPerAddress = fc_readOutstandingBytesPerAddress(path_id);
        status->mtu = fc_readMTU(path_id);
        status->ipTos = currentAssociation->ipTos;
    } else {
        error_logi(ERROR_MAJOR, "sctp_getPathStatus : association %u does not exist", associationID);
    }
    result = mdi_clearAssociationData(associationID);
    return result;
}

/**
 * sctp_setPathStatus is currently NOT implemented !
 * Can be used to set path specific parameters in an existing association.
 *
 *  @param  associationID   ID of assocation.
 *  @param  path_id         path for which to set parameters
 *  @param  new_status      pointer to new parameters
 *  @return -1
 */
int sctp_setPathStatus(unsigned int associationID, short path_id, SCTP_PathStatus* new_status)
{
    error_log(ERROR_MAJOR, "sctp_setPathStatus : unimplemented function");
    return -1;
}


/**
 * sctp_setAssocStatus allows for setting a number of association parameters.
 * _Not_ all values that the corresponding sctp_getAssocStatus-function returns
 * may be SET here !
 * Will set protocol parameters per SCTP association
 *
 *  @param  associationID   ID of assocation.
 *  @param  new_status      pointer to new parameters
 *  @return -1
*/
int sctp_setAssocStatus(unsigned int associationID, SCTP_AssociationStatus* new_status)
{
    guint16 result;

    if (new_status == NULL) return -1;

    result = mdi_setAssociationData(associationID);
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL && result == 0) {
        event_logi(VERBOSE, "sctp_setAssocStatus: Association %u", associationID);
        if (pm_setPrimaryPath(new_status->primaryAddressIndex)) {
            error_logi(ERROR_MINOR, "pm_setPrimary(%u) returned error", new_status->primaryAddressIndex);
            result = mdi_clearAssociationData(associationID);
            return -1;
        }
        if (pm_setRtoInitial(new_status->rtoInitial)) {
            error_logi(ERROR_MINOR, "pm_setRtoInitial(%u) returned error", new_status->rtoInitial);
            result = mdi_clearAssociationData(associationID);
            return -1;
        }
        if (pm_setRtoMin(new_status->rtoMin)) {
            error_logi(ERROR_MINOR, "pm_setRtoMin(%u) returned error", new_status->rtoMin);
            result = mdi_clearAssociationData(associationID);
            return -1;
        }
        if(pm_setMaxPathRetransmisions(new_status->pathMaxRetransmits)) {
            error_logi(ERROR_MINOR, "pm_getMaxPathRetransmisions(%u) returned error", new_status->pathMaxRetransmits);
            result = mdi_clearAssociationData(associationID);
            return -1;
        }
        sci_setCookieLifeTime(new_status->validCookieLife);

        sci_setMaxAssocRetransmissions(new_status->assocMaxRetransmits);
        sci_setMaxInitRetransmissions(new_status->maxInitRetransmits);

        rxc_set_local_receiver_window(new_status->myRwnd);
        rxc_set_sack_delay(new_status->delay);
        currentAssociation->ipTos = new_status->ipTos;

        result = mdi_clearAssociationData(associationID);
        return 0;
    } else {
        error_logi(ERROR_MAJOR, "sctp_getAssocStatus : association %u does not exist", associationID);
        result = mdi_clearAssociationData(associationID);
        return -1;
    }
    result = mdi_clearAssociationData(associationID);
    return 0;
}                               /* end: sctp_setAssocStatus */

/**
 * sctp_getAssocStatus is currently NOT implemented !
 * Will set protocol parameters per SCTP association
 *
 *  @param  associationID   ID of assocation.
 *  @return  pointer to a structure containing association parameters
*/
int sctp_getAssocStatus(unsigned int associationID, SCTP_AssociationStatus* status)
{
    guint16 result;

    if (status == NULL) return -1;

    result = mdi_setAssociationData(associationID);
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL && result == 0) {
        event_logi(VERBOSE, "sctp_getAssocStatus: Association %u", associationID);
        status->state = sci_getState();
        status->numberOfAddresses = currentAssociation->noOfNetworks;
        status->primaryAddressIndex = pm_readPrimaryPath();

        adl_sockunion2str(&(currentAssociation->destinationAddresses[status->primaryAddressIndex]),
                          &(status->primaryDestinationAddress[0]),SCTP_MAX_IP_LEN);

        se_readNumberOfStreams(&(status->inStreams), &(status->outStreams));
        status->currentReceiverWindowSize =  rtx_read_remote_receiver_window();
        status->outstandingBytes = fc_readOutstandingBytes();
        status->noOfChunksInSendQueue = fc_readNumberOfQueuedChunks();
        status->noOfChunksInRetransmissionQueue = rtx_readNumberOfUnackedChunks();
        status->noOfChunksInReceptionQueue = se_numOfQueuedChunks();
        status->rtoInitial = pm_getRtoInitial();
        status->rtoMin = pm_getRtoMin();
        status->validCookieLife = sci_getCookieLifeTime();
        status->assocMaxRetransmits = sci_getMaxAssocRetransmissions();
        status->pathMaxRetransmits = pm_getMaxPathRetransmisions();
        status->maxInitRetransmits = sci_getMaxInitRetransmissions();
        status->myRwnd = rxc_get_local_receiver_window();
        status->delay = rxc_get_sack_delay();
        status->maxSendQueue = 0;
        status->maxRecvQueue = 0;
        status->ipTos = 0;

    } else {
        error_logi(ERROR_MAJOR, "sctp_getAssocStatus : association %u does not exist", associationID);
        result = mdi_clearAssociationData(associationID);
        return -1;
    }
    result = mdi_clearAssociationData(associationID);
    return 0;
}                               /* end: sctp_getAssocStatus */

/**
 * sctp_setAssocDefaults allows for setting a few association default parameters !
 * Will set protocol default parameters per given SCTP instance
 *
 *  @param  SCTP_InstanceName   instance for which to set the parameters
 *  @param  params       pointer to parameter data structure
 *  @return -1
*/
int sctp_setAssocDefaults(unsigned short SCTP_InstanceName, SCTP_InstanceParameters* params)
{
    SCTP_instance temporary;
    SCTP_instance* instance;
    GList* result = NULL;

    event_logi(VERBOSE, "sctp_setInstanceParams: Instance %u", SCTP_InstanceName);

    temporary.sctpInstanceName = SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
    } else {
        error_logi(ERROR_MINOR, "sctp_setAssocDefaults : Did not find Instance Number %u", SCTP_InstanceName);
        return -1;
    }
    if (params == NULL) {
        error_log(ERROR_MINOR, "sctp_setAssocDefaults : Passed NULL Pointer !");
        return -1;
    }
    instance->default_rtoInitial =  params->rtoInitial;
    instance->default_rtoMin = params->rtoMin;
    instance->default_validCookieLife = params->validCookieLife;
    instance->default_assocMaxRetransmits =  params->assocMaxRetransmits;
    instance->default_pathMaxRetransmits = params->pathMaxRetransmits;
    instance->default_maxInitRetransmits =  params->maxInitRetransmits;
    instance->default_myRwnd =  params->myRwnd;
    instance->default_delay = params->delay;
    instance->default_ipTos = params->ipTos;
    instance->default_rtoMin =  params->rtoMin;
    instance->default_maxSendQueue = params->maxSendQueue;
    instance->default_maxRecvQueue = params->maxRecvQueue;
    return 0;
}                               /* end: sctp_setInstanceParams */

/**
 * sctp_getInstanceParams returns a struct with default parameter values !
 * Will get protocol parameters per given SCTP instance
 *
 *  @param  SCTP_InstanceName   instance for which to set the parameters
 *  @param  params       pointer to parameter data
 *  @return -1
*/
int sctp_getAssocDefaults(unsigned short SCTP_InstanceName, SCTP_InstanceParameters* params)
{
    SCTP_instance temporary;
    SCTP_instance* instance;
    GList* result = NULL;

    event_logi(VERBOSE, "sctp_getInstanceParams: Instance %u", SCTP_InstanceName);

    temporary.sctpInstanceName = SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
    } else {
        error_logi(ERROR_MINOR, "sctp_getAssocDefaults : Did not find Instance Number %u", SCTP_InstanceName);
        return -1;
    }
    if (params == NULL) {
        error_log(ERROR_MINOR, "sctp_getAssocDefaults : Passed NULL Pointer !");
        return -1;
    }
    params->rtoInitial = instance->default_rtoInitial;
    params->rtoMin  = instance->default_rtoMin;
    params->validCookieLife = instance->default_validCookieLife;
    params->assocMaxRetransmits = instance->default_assocMaxRetransmits;
    params->pathMaxRetransmits = instance->default_pathMaxRetransmits;
    params->maxInitRetransmits = instance->default_maxInitRetransmits;
    params->myRwnd = instance->default_myRwnd;
    params->delay = instance->default_delay ;
    params->ipTos = instance->default_ipTos ;
    params->rtoMin = instance->default_rtoMin;
    params->maxSendQueue = instance->default_maxSendQueue;
    params->maxRecvQueue = instance->default_maxRecvQueue;
    return 0;
}                               /* end: sctp_getAssocDefaults */


/**
 * sctp_receive_unsent is currently NOT implemented !
 * Will return messages that could not be sent for some reason
 *
 *  @param  associationID       ID of assocation.
 *  @param  buffer              pointer to a buffer that the application needs to pass. Data is copied there.
 *  @param  length              pointer to size of the buffer passed by application, contains actual length
 *                              of the copied chunk after the function call.
 *  @param  streamID            pointer to the stream id, where data should have been sent
 *  @param  streamSN            pointer to stream sequence number of the data chunk that was not sent
 *  @param  protocolID          pointer to the protocol ID of the unsent chunk
 *  @return number of unsent chunks still in the queue, else -1 on error (no chunks there etc.)
*/
int sctp_receiveUnsent(unsigned int associationID, unsigned char *buffer, unsigned int *length,
                        unsigned short *streamID, unsigned short *streamSN,unsigned int* protocolId)
{
    error_log(ERROR_MAJOR, "sctp_receiveUnsent: unimplemented funktion");
    return -1;

}

/**
 * sctp_receive_unacked is currently NOT implemented !
 * Will return messages that could not be sent for some reason
 *
 *  @param  associationID       ID of assocation.
 *  @param  buffer              pointer to a buffer that the application needs to pass. Data is copied there.
 *  @param  length              pointer to size of the buffer passed by application, contains actual length
 *                              of the copied chunk after the function call.
 *  @param  streamID            pointer to the stream id, where data should have been sent
 *  @param  streamSN            pointer to stream sequence number of the data chunk that was not acked
 *  @param  protocolID          pointer to the protocol ID of the unacked chunk
 *  @return number of unacked chunks still in the queue, else -1 on error (no chunks there etc.)
*/
int sctp_receiveUnacked(unsigned int associationID, unsigned char *buffer, unsigned int *length,
                        unsigned short *streamID, unsigned short *streamSN,unsigned int* protocolId)
{
    error_log(ERROR_MAJOR, "sctp_receiveUnacked: unimplemented funktion");
    return -1;

}


/**
 * sctp_getPrimary is currently NOT implemented !
 * Will return the index of the current primary path
 * @param  associationID       ID of assocation.
 * @return  the index of the current primary path, or -1 on error
 */
short sctp_getPrimary(unsigned int associationID)
{
    guint16 result;
    short primary = -1;

    result = mdi_setAssociationData(associationID);
    currentAssociation = retrieveAssociationULP(associationID);

    if (currentAssociation != NULL && result == 0) {
        event_logi(VERBOSE, "sctp_getPrimary: Association %u", associationID);
        primary = pm_readPrimaryPath();
    }else{
        error_logi(ERROR_MAJOR, "sctp_getPrimary : association %u does not exist", associationID);
    }
    result = mdi_clearAssociationData(associationID);
    if (result != 0) return -1;
    return primary;
}




/*------------------- Functions called by the SCTP bundling --------------------------------------*/

/**
 * Used by bundling to send a SCTP-datagramm.
 *
 * Bundling passes a static pointer and leaves space for common header, so
 * we can fill that header in up front !
 * Before calling send_message at the adaption-layer, this function does:
 * \begin{itemize}
 * \item add the SCTP common header to the message
 * \item convert the SCTP message to a byte string
 * \item retrieve the socket-file descriptor of the SCTP-instance
 * \item retrieve the destination address
 * \item retrieve destination port ???
 * \end{itemize}
 *
 *  @param SCTP_message     SCTP message as a struct (i.e. common header and chunks)
 *  @param length           length of complete SCTP message.
 *  @param destAddresIndex  Index of address in the destination address list.
 *  @return                 Errorcode (0 for good case: length bytes sent; 1 or -1 for error)
*/
int mdi_send_message(SCTP_message * message, unsigned int length, short destAddressIndex)
{
    union sockunion dest_su, *dest_ptr;
    SCTP_simple_chunk *chunk;
    unsigned char tos = 0;
    int txmit_len = 0;
    guchar hoststring[SCTP_MAX_IP_LEN];


    if (message == NULL) {
        error_log(ERROR_MINOR, "mdi_send_message: no message to send !!!");
        return 1;
    }

    chunk = (SCTP_simple_chunk *) & message->sctp_pdu[0];

    if (currentAssociation == NULL) {
        /* only possible case : initAck, no association exists yet,
           use last from address as destination address */

        if (lastFromAddress == NULL) {
            error_log(ERROR_MAJOR, "mdi_send_message: lastFromAddress does not exist for initAck");
            return 1;
        } else {
            /* only if the sctp-message received before contained an init-chunk */
            memcpy(&dest_su, lastFromAddress, sizeof(union sockunion));
            dest_ptr = &dest_su;
            message->common_header.verification_tag = htonl(lastInitiateTag);
            /* write invalid tag value to lastInitiateTag (reset it) */
            lastInitiateTag = 0;
            /* swap ports */
            message->common_header.src_port = htons(mdi_readLastDestPort());
            message->common_header.dest_port = htons(mdi_readLastFromPort());
            event_logiii(VVERBOSE,
                         "mdi_send_message (I) : tag = %x, src_port = %u , dest_port = %u",
                         lastInitiateTag, mdi_readLastDestPort(), mdi_readLastFromPort());

            tos =  sctpInstance->default_ipTos;
        }
    } else {
        if (destAddressIndex < 0 || destAddressIndex >= currentAssociation->noOfNetworks) {
            error_log(ERROR_MAJOR, "mdi_send_message: invalid destination address");
            return 1;
        }

        /* Use given destination address from current association */
        dest_ptr = &(currentAssociation->destinationAddresses[destAddressIndex]);

        if (isInitAckChunk(chunk)) {
            /* is true in case of an init-collision, normally when an initAck is sent
               no association exist and the last lastInitiateTag is used in the initAck. This
               is handled in the case above, where no association exists.
               Or when we respond to SHUTDOWN_ACK, see section 8.4.5)
             */
            if (lastInitiateTag == 0) {
                error_log(ERROR_MAJOR, "mdi_send_message: No verification tag");
                return 1;
            }

            message->common_header.verification_tag = htonl(lastInitiateTag);
        } else {
            message->common_header.verification_tag = htonl(currentAssociation->tagRemote);
        }

        message->common_header.src_port = htons(currentAssociation->localPort);
        message->common_header.dest_port = htons(currentAssociation->remotePort);
        event_logiii(VVERBOSE,
                     "mdi_send_message (II): tag = %x, src_port = %u , dest_port = %u",
                     ntohl(message->common_header.verification_tag),
                     currentAssociation->localPort, currentAssociation->remotePort);
        tos = currentAssociation->ipTos;
    }

    /* calculate and insert checksum */
    validate_insert_checksum((unsigned char *) message, length);

    switch (sockunion_family(dest_ptr)) {
    case AF_INET:
        txmit_len = adl_send_message(sctp_socket, message, length, dest_ptr, tos);
        break;
#ifdef HAVE_IPV6
    case AF_INET6:
        txmit_len = adl_send_message(ipv6_sctp_socket, message, length, dest_ptr, tos);
        break;
#endif
    default:
        error_log(ERROR_MAJOR, "mdi_send_message: Unsupported AF_TYPE");
        break;
    }

    adl_sockunion2str(dest_ptr, hoststring, SCTP_MAX_IP_LEN);
    event_logiii(INTERNAL_EVENT_0, "sent SCTP message of %d bytes to %s, result was %d",
                    length, hoststring, txmit_len);

    return (txmit_len == length) ? 0 : -1;

}                               /* end: mdi_send_message */



/*------------------- Functions called by the SCTP to forward primitives to ULP ------------------*/


/**
 *  indicates new data has arrived from peer (chapter 10.2.) destined for the ULP
 *
 *  @param streamID  received data belongs to this stream
 *  @param  length   so many bytes have arrived (may be used to reserve space)
 *  @param  protoID  the protocol ID of the arrived payload
 *  @param  unordered  unordered flag (TRUE==1==unordered, FALSE==0==normal,numbered chunk)
 */
void mdi_dataArriveNotif(unsigned int streamID, unsigned int length, unsigned int protoID, unsigned int unordered)
{

    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;

    if (currentAssociation != NULL) {


        /* Forward dataArriveNotif to the ULP */
        sctpInstance->ULPcallbackFunctions.dataArriveNotif(currentAssociation->tagLocal,
                                                           streamID,
                                                           length,
                                                           protoID,
                                                           unordered,
                                                           currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_dataArriveNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}                               /* end: mdi_dataArriveNotif */



/**
 * indicates a change of network status (chapter 10.2.C). Calls the respective ULP callback function.
 * @param  destinationAddress   index to address that has changed
 * @param  newState             state to which indicated address has changed (PM_ACTIVE/PM_INACTIVE)
 */
void mdi_networkStatusChangeNotif(short destinationAddress, unsigned short newState)
{
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;
    if (currentAssociation != NULL) {
        sctpInstance->ULPcallbackFunctions.networkStatusChangeNotif(currentAssociation->tagLocal,
                                                                    destinationAddress, newState, currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_networkStatusChangeNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}                               /* end: mdi_networkStatusChangeNotif */



/**
 * indicates a send failure (chapter 10.2.B). Calls the respective ULP callback function.
 * @param data          pointer to the data that has not been sent
 * @param dataLength    length of the data that has not been sent
 * @param context       from sendChunk (CHECKME : may be obsolete ?)
 */
void mdi_sendFailureNotif(unsigned char *data, unsigned int dataLength, unsigned int *context)
{
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;
    if (currentAssociation != NULL) {
        sctpInstance->ULPcallbackFunctions.sendFailureNotif(currentAssociation->tagLocal,
                                                            data, dataLength, context,
                                                            currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_sendFailureNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}                               /* end: mdi_sendFailureNotif */


/**
 * indicates that association has been gracefully shut down (chapter 10.2.H).
 * Calls the respective ULP callback function.
 */
void mdi_shutdownCompleteNotif()
{
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;
    if (currentAssociation != NULL) {
        sctpInstance->ULPcallbackFunctions.shutdownCompleteNotif(currentAssociation->tagLocal,
                                                                    currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_shutdownCompleteNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}


/**
 * indicates that a restart has occured(chapter 10.2.G).
 * Calls the respective ULP callback function.
 */
void mdi_restartNotif()
{
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;
    if (currentAssociation != NULL) {
        sctpInstance->ULPcallbackFunctions.restartNotif(currentAssociation->tagLocal,
                                                             currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_restartNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}



/**
 * indicates that communication was lost to peer (chapter 10.2.E). Calls the respective ULP callback function.
 *
 * @param  status  type of event, that has caused the association to be terminated
 */
void mdi_communicationLostNotif(unsigned short status)
{
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;
    if (currentAssociation != NULL) {
        sctpInstance->ULPcallbackFunctions.communicationLostNotif(currentAssociation->tagLocal,
                                                                       status,
                                                                       currentAssociation->ulp_dataptr);
    } else {
        error_log(ERROR_MAJOR, "mdi_communicationLostNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}                               /* end: mdi_communicationLostNotif */



/**
 * indicates that an association is established (chapter 10.2.D).
 *
 * @param status     type of event that caused association to come up;
 *                   either SCTP_COMM_UP_RECEIVED_VALID_COOKIE, SCTP_COMM_UP_RECEIVED_COOKIE_ACK
 *                   or SCTP_COMM_UP_RECEIVED_COOKIE_RESTART
 */
void mdi_communicationUpNotif(unsigned short status)
{
    union sockunion *lastAddress;
    short primaryPath;
    unsigned short noOfInStreams;
    unsigned short noOfOutStreams;
    SCTP_instance *old_Instance = sctpInstance;
    Association *old_assoc = currentAssociation;

    if (currentAssociation != NULL) {
        /* Find primary path */
        lastAddress = (union sockunion *) mdi_readLastFromAddress();

        if (lastAddress != NULL) {

            for (primaryPath = 0; primaryPath < currentAssociation->noOfNetworks; primaryPath++) {
                if (adl_equal_address
                    (&(currentAssociation->destinationAddresses[primaryPath]), lastAddress)) {
                    break;
                }
            }
        } else {
            primaryPath = 0;
        }
        if (primaryPath >= currentAssociation->noOfNetworks)
            primaryPath = 0;

        /* set number of paths and primary path at pathmanegement and start heartbeat */
        pm_setPaths(currentAssociation->noOfNetworks, primaryPath);

        se_readNumberOfStreams(&noOfInStreams, &noOfOutStreams);


        event_logiii(VERBOSE,
                     "Distribution: COMM-UP, assocId: %u, status: %u, noOfNetworks: %u",
                     currentAssociation->tagLocal, status, currentAssociation->noOfNetworks);
        event_logii(VERBOSE, "noOfInStreams: %u,noOfOutStreams  %u", noOfInStreams, noOfOutStreams);
        /* FIXME (???) : retreive sctp-instance from list */

        /* Forward mdi_communicationup Notification to the ULP */
        currentAssociation->ulp_dataptr = sctpInstance->ULPcallbackFunctions.communicationUpNotif(
                                                                currentAssociation->tagLocal,
                                                                status,
                                                                currentAssociation->noOfNetworks,
                                                                noOfInStreams, noOfOutStreams,
                                                                currentAssociation->ulp_dataptr);

        free(lastAddress);
    } else {
        error_log(ERROR_MAJOR, "mdi_communicationUpNotif: association not set");
    }
    sctpInstance = old_Instance;
    currentAssociation = old_assoc;
}                               /* end: mdi_communicationLostNotif */



/*------------------- Functions called by the SCTP to get current association data----------------*/

/* When processing external events from outside the SCTP (socket events, timer events and 
   function calls from the ULP), first the data of the addressed association are read
   from the list of associations and stored in a private but static datastructure.
   Elements of this association data can be read by the following functions.
*/


/* The following functions return pointer to data of modules of the SCTP. As only these
   modules know the exact type of these data structures, so the returned pointer are
   of type void.
*/

/**
 * function to return a pointer to the flow control module of this association
 * @return pointer to the flow control data structure,  null in case of error.
 */
void *mdi_readFlowControl()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readFlowControl: association not set");
        return NULL;
    } else {
        return currentAssociation->flowControl;
    }
}



/**
 * function to return a pointer to the reliable transfer-module of this association
 * @return pointer to the reliable transfer data structure, null in case of error.
 */
void *mdi_readReliableTransfer()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readReliableTransfer: association not set");
        return NULL;
    } else {
        return currentAssociation->reliableTransfer;
    }
}



/**
 * function to return a pointer to the receiver module of this association
 * @return pointer to the RX-control data structure, null in case of error.
 */
void *mdi_readRX_control()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readRX_control: association not set");
        return NULL;
    } else {
        return currentAssociation->rx_control;
    }
}



/**
 * function to return a pointer to the stream-engine module of this association
 * @return pointer to the stream engine data structure, null in case of error.
 */
void *mdi_readStreamEngine()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readStreamEngine: association not set");
        return NULL;
    } else {
        return currentAssociation->streamengine;
    }
}



/**
 * function to return a pointer to the path management module of this association
 * @return  pointer to the pathmanagement data structure, null in case of error.
 */
void *mdi_readPathMan()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readPathMan: association not set");
        return NULL;
    } else {
        return currentAssociation->pathMan;
    }
}


/**
 * function to return a pointer to the bundling module of this association
 * @return   pointer to the bundling data structure, null in case of error.
 */
void *mdi_readBundling()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readBundling: association not set");
        return NULL;
    } else {
        return currentAssociation->bundling;
    }
}



/**
 * function to return a pointer to the state machine controller of this association
 * @return pointer to the SCTP-control data structure, null in case of error.
 */
void *mdi_readSCTP_control()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readSCTP_control: association not set");
        return NULL;
    } else {
        return currentAssociation->sctp_control;
    }
}



/**
 * function to read the association id of the current association
 * @return   association-ID of the current association;
 *           0 means the association is not set (an error).
 */
unsigned int mdi_readAssociationID()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readAssociationID: association not set");
        return 0;
    } else {
        return currentAssociation->tagLocal;
    }
}



/**
 * function to read the tag that the peer within the current association uses
 * @return   tag value of the peer within the current association;
 *           CHECKME: can tag legally be 0 ?
 *           0 means the association is not set (an error).
 */
unsigned int mdi_readTagRemote()
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_readAssociationID: association not set");
        return 0;
    } else {
        return currentAssociation->tagRemote;
    }
}



/**
 * generates a random tag value for a new association, but not 0
 * @return   generates a random tag value for a new association, but not 0
 */
unsigned int mdi_generateLocalTag()
{
    unsigned int tag;

    while ((tag = random()) == 0);

    return tag;
}



/**
 * generates a random tsn value for a new association (may also be 0)
 * @return   generates a random tsn value for a new association (may also be 0)
 */
unsigned int mdi_generateStartTSN()
{
    return random();
}



/*------------- functions for the cookie mechanism --------------------------------------------*/


/**
 * read the address from which the last datagramm was received (host byte order).
 * @return the address from which the last datagramm was received (host byte order).
 */
union sockunion *mdi_readLastFromAddress()
{
    union sockunion *destAddressList;

    if (lastFromAddress == NULL) {
        error_log(ERROR_MINOR, "mdi_saveLastFromAddress: no last from address");
        return NULL;
    } else {
        destAddressList = (union sockunion *) malloc(sizeof(union sockunion));
        if (destAddressList != NULL) {
            memcpy(destAddressList, lastFromAddress, sizeof(union sockunion));
        } else {
            error_log(ERROR_MAJOR, "memory allocation failed ! Aborting ");
            return NULL;
        }
        return destAddressList;
    }
}


/**
 * read the index of the path from which the last DG was received (-1 if no DG was received)
 * @return index of the path from which the last DG was received (-1 if no DG was received)
 */
short mdi_readLastFromPath()
{
    return lastFromPath;
}

/**
 * read the port of the sender of the last received DG (host byte order)
 * @return the port of the sender of the last received DG (host byte order)
 */
unsigned short mdi_readLastFromPort()
{
    if (lastFromAddress == NULL) {
        error_log(ERROR_MINOR, "readLastFromPort: no last from address");
        return 0;
    } else {
        return lastFromPort;
    }
}


/**
 * read the port of the destination of the last received DG (host byte order)
 * @return the port of the destination of the last received DG (host byte order)
 */
unsigned short mdi_readLastDestPort()
{
    if (lastFromAddress == NULL) {
        error_log(ERROR_MINOR, "readLastDestPort: no last from address");

        return 0;
    } else {
        return lastDestPort;
    }
}

/* write the initiate tag of a-side to be used as verification tag for the initAck */
void mdi_writeLastInitiateTag(unsigned int initiateTag)
{
    lastInitiateTag = initiateTag;
}

/* rewrite the initiate tag of peer in case of a peer reset. */
void mdi_rewriteTagRemote(unsigned int newInitiateTag)
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_rewriteRemoteTag: association not set");
    } else {
        currentAssociation->tagRemote = newInitiateTag;
    }
}

/*------------- functions to write and read addresses --------------------------------------------*/

/**
 * copies destination addresses from the array passed as parameter to  the current association
 * @param addresses array that will hold the destination addresses after returning
 * @aram noOfAddresses number of addresses that the peer has (and sends along in init/initAck)
 */
void mdi_writeDestinationAddresses(union sockunion addresses[MAX_NUM_ADDRESSES], int noOfAddresses)
{

    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_writeDestinationAddresses: association not set");
        return;
    } else {
        if (currentAssociation->destinationAddresses != NULL)
            free(currentAssociation->destinationAddresses);

        currentAssociation->destinationAddresses =
            (union sockunion *) malloc(noOfAddresses * sizeof(union sockunion));

        if (currentAssociation->destinationAddresses == NULL)
            error_log(ERROR_FATAL, "mdi_writeDestinationAddresses: out of memory");

        memcpy(currentAssociation->destinationAddresses, addresses,
               noOfAddresses * sizeof(union sockunion));

        currentAssociation->noOfNetworks = noOfAddresses;

        return;
    }
}


/**
 * Function that returns the number of incoming streams that this instance
 * is willing to handle !
 * @return maximum number of in-streams
 */
unsigned short mdi_readLocalInStreams( )
{
    SCTP_instance temporary;
    GList* result = NULL;

    if (currentAssociation == NULL) {
        /* retrieve SCTP-instance with last destination port */
        lastDestPort = mdi_readLastDestPort();
        event_logi(VERBOSE, "Searching for SCTP Instance with Port %u ", lastDestPort);
        temporary.localPort = lastDestPort;
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstancePorts);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance for Port %u in List, FIXME !",lastDestPort);
        }
        sctpInstance = result->data;
    } else {
        /* retrieve SCTP-instance with SCTP-instance name in current association */
        temporary.sctpInstanceName = currentAssociation->sctpInstance->sctpInstanceName;
        event_logi(VERBOSE, "Searching for SCTP Instance with Name %u ", currentAssociation->sctpInstance->sctpInstanceName);
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance with name %u in List, FIXME !",
                currentAssociation->sctpInstance->sctpInstanceName);
        }
        sctpInstance = result->data;
    }
    return  sctpInstance->noOfInStreams;
}

/**
 * Function that returns the number of incoming streams that this instance
 * is willing to handle !
 * @return maximum number of in-streams
 */
unsigned short mdi_readLocalOutStreams( )
{
    SCTP_instance temporary;
    GList* result = NULL;

    if (currentAssociation == NULL) {
        /* retrieve SCTP-instance with last destination port */
        lastDestPort = mdi_readLastDestPort();
        event_logi(VERBOSE, "Searching for SCTP Instance with Port %u ", lastDestPort);
        temporary.localPort = lastDestPort;
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstancePorts);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance for Port %u in List, FIXME !",lastDestPort);
        }
        sctpInstance = result->data;
    } else {
        /* retrieve SCTP-instance with SCTP-instance name in current association */
        temporary.sctpInstanceName = currentAssociation->sctpInstance->sctpInstanceName;
        event_logi(VERBOSE, "Searching for SCTP Instance with Name %u ", currentAssociation->sctpInstance->sctpInstanceName);
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance with name %u in List, FIXME !",
                currentAssociation->sctpInstance->sctpInstanceName);
        }
        sctpInstance = result->data;
    }
    return  sctpInstance->noOfOutStreams;
}


/**
 * Copies local addresses of this instance into the array passed as parameter
 * CHECKME : does this function work in all circumstances ?
 * --> Under what conditions can we NOT find the SCTP instance ?
 *
 * @param addresses array that will hold the local host's addresses after returning
 * @aram noOfAddresses number of addresses that local host/current association has
 */
void mdi_readLocalAddresses(union sockunion addresses[MAX_NUM_ADDRESSES], guint16 * noOfAddresses)
{

    unsigned short lastDestPort;
    SCTP_instance temporary;
    GList* result = NULL;

    if (currentAssociation == NULL && mdi_readLastDestPort() == 0) {
        error_log(ERROR_MINOR, "mdi_readLocalAddresses: association not set");
        *noOfAddresses = 0;
        return;
    }
    if (currentAssociation == NULL) {
        /* retrieve SCTP-instance with last destination port */
        lastDestPort = mdi_readLastDestPort();
        event_logi(VERBOSE, "Searching for SCTP Instance with Port %u ", lastDestPort);
        temporary.localPort = lastDestPort;
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstancePorts);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance for Port %u in List, FIXME !",lastDestPort);
        }
        sctpInstance = result->data;
    } else {
        /* retrieve SCTP-instance with SCTP-instance name in current association */
        temporary.sctpInstanceName = currentAssociation->sctpInstance->sctpInstanceName;
        event_logi(VERBOSE, "Searching for SCTP Instance with Name %u ", currentAssociation->sctpInstance->sctpInstanceName);
        result = g_list_find_custom(InstanceList, &temporary, &CompareInstanceNames);
        if (result == NULL) {
            error_logi(ERROR_FATAL, "Could not find SCTP Instance with name %u in List, FIXME !",
                currentAssociation->sctpInstance->sctpInstanceName);
        }
        sctpInstance = result->data;
    }

    event_logi(VERBOSE, "mdi_readLocalAddresses() : returning %u addresses !",
               sctpInstance->noOfLocalAddresses);

    memcpy(addresses, sctpInstance->localAddressList,
           sctpInstance->noOfLocalAddresses * sizeof(union sockunion));

    *noOfAddresses = sctpInstance->noOfLocalAddresses;
}

int mdi_getDefaultRtoInitial(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_rtoInitial;
}
int mdi_getDefaultValidCookieLife(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_validCookieLife;
}
int mdi_getDefaultAssocMaxRetransmits(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_assocMaxRetransmits;
}
int mdi_getDefaultPathMaxRetransmits(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_pathMaxRetransmits;
}
int mdi_getDefaultMaxInitRetransmits(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_maxInitRetransmits;
}
int mdi_getDefaultMyRwnd()
{
    if (sctpInstance == NULL) return -1;
    else {
        event_logi(VVERBOSE, " mdi_getDefaultMyRwnd is %u", sctpInstance->default_myRwnd);
        return ((SCTP_instance*)sctpInstance)->default_myRwnd;
    }
}
int mdi_getDefaultRtoMin(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_rtoMin;
}

int mdi_getDefaultDelay(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_delay;
}

int mdi_getDefaultIpTos(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_ipTos;
}
int mdi_getDefaultMaxSendQueue(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_maxSendQueue;
}
int mdi_getDefaultMaxRecvQueue(void* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else
        return ((SCTP_instance*)sctpInstance)->default_maxRecvQueue;
}
/*------------- functions to set and clear the association data ----------------------------------*/

/**
 * Each module within SCTP that has timers implements its own timer call back
 *  functions. These are registered at the adaption layer when a timer is started
 *  and called directly at the module when the timer expires.
 *  setAssociationData allows SCTP-modules with timers to retrieve the data of the
 *  addressed association from the list of associations.
 *  For this purpose the association-ID must be included in one of the
 *  parameters of the start_timer function of the adaption-layer.
 *
 *  @param  associationID    the ID of the association
 *  @return 0 if successful, 1 if the association does not exist in the list
*/
unsigned short mdi_setAssociationData(unsigned int associationID)
{
    if (currentAssociation != NULL)
        error_log(ERROR_MINOR, "mdi_setAssociationData: previous assoc not cleared");

    /* retrieve association from list */
    currentAssociation = retrieveAssociation(associationID);
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_setAssociationData: association does not exist");
        return 1;
    }
    return 0;
}



/**
 * Clear the global association data.
 *  This function must be called after the association retrieved from the list
 *  with setAssociationData is no longer needed. This is the case after a time
 *  event has been handled.
 *
 *  @param  associationID    the ID of the association
 *  @return  0 if successful, 1 if association data has not been set, 2 wrong associationID
 */
unsigned short mdi_clearAssociationData(unsigned int associationID)
{
    if (currentAssociation == NULL) {
        error_log(ERROR_MINOR, "mdi_clearAssociationData: association not set");
        return 1;
    }

    if (currentAssociation->tagLocal != associationID) {
        error_log(ERROR_MINOR, "mdi_clearAssociationData: wrong associationID");
        return 2;
    }

    /* clear currentAssociation */
    currentAssociation = NULL;

    return 0;
}


/*------------------- Functions to create and delete associations --------------------------------*/

/**
 *  This function allocates memory for a new association.
 *  For the active side of an association, this function is called when ULP calls Associate
 *  For the passive side this function is called when a valid cookie message is received.
 *  It also creates all the modules path management, bundling and SCTP-control.
 *  The rest of the modules are created with mdi_initAssociation.
 *  The created association is put into the list of associations.
 *
 *  @param SCTP_InstanceName    identifier for an SCTP instance (if there are more)
 *  @param  local_port          src port (which this association listens to)
 *  @param  remote_port         destination port (peers source port)
 *  @param   tagLocal           randomly generated tag belonging to this association
 *  @param  primaryDestinitionAddress   index of the primary address
 *  @param  noOfDestinationAddresses    number of addresses the peer has
 *  @param  destinationAddressList      pointer to the array of peer's addresses
 *  @return 0 for success, else 1 for failure
 */
unsigned short
mdi_newAssociation(unsigned short SCTP_InstanceName,
                   unsigned short local_port,
                   unsigned short remote_port,
                   unsigned int tagLocal,
                   short primaryDestinitionAddress,
                   short noOfDestinationAddresses, union sockunion *destinationAddressList)
{

    SCTP_instance temporary;
    GList* result = NULL;
    /* tbd: retrieve SCTP-instance using local port from list */
    temporary.localPort = local_port;
    result = g_list_find_custom(InstanceList, &temporary, &CompareInstancePorts);
    if (result == NULL) {
        error_logi(ERROR_FATAL, "Could not find SCTP Instance for Port %u in List, call SCTP_register_instance FIRST !",local_port);
    }
    sctpInstance = result->data;

    /* check if SCTP-instance exists */
    if (!sctpInstance) {
        error_log(ERROR_MAJOR, "SCTP-protocol not ready");
        return 1;
    }

    /* If no SCTP-instance name is provided (z-side), use the instance name
       from the SCTP-instance found in list. */
    if (SCTP_InstanceName == 0) {
        SCTP_InstanceName = sctpInstance->sctpInstanceName;
        error_log(ERROR_MINOR, "SCTP Instance Name in call to mdi_newAssociation() was ZERO !!!");
    }

    /* check wether SCTP-instance names are equal (makes sense only for the a-side) */
    if (sctpInstance->sctpInstanceName != SCTP_InstanceName) {
        error_log(ERROR_MAJOR,
                  "Local Port does not match passed SCTP instance name -- FIX YOUR APPLICATION !!");
        return 1;
    }

    /* Do plausi checks on the addresses. */
    if (noOfDestinationAddresses <= 0 || destinationAddressList == NULL) {
        error_log(ERROR_MAJOR, "No destination address suppllied for new association");
        return 1;
    }
        else if (primaryDestinitionAddress < 0
                 || primaryDestinitionAddress >= noOfDestinationAddresses) {
        error_log(ERROR_MAJOR, "Invalid primary destination address for new association");
        return 1;
    }

    if (currentAssociation) {
        error_log(ERROR_MINOR, "current association not cleared");
    }

    currentAssociation = (Association *) malloc(sizeof(Association));

    if (!currentAssociation) {
        error_log_sys(ERROR_FATAL, errno);
        return 1;
    }

    currentAssociation->sctpInstance = sctpInstance;
    currentAssociation->localPort = local_port;
    currentAssociation->remotePort = remote_port;
    currentAssociation->tagLocal = tagLocal;
    currentAssociation->tagRemote = 0;
    currentAssociation->deleted = FALSE;

    currentAssociation->ulp_dataptr = NULL;
    currentAssociation->ipTos = sctpInstance->default_ipTos;


    currentAssociation->noOfNetworks = noOfDestinationAddresses;
    currentAssociation->destinationAddresses =
        (union sockunion *) malloc(noOfDestinationAddresses * sizeof(union sockunion));
    memcpy(currentAssociation->destinationAddresses, destinationAddressList,
           noOfDestinationAddresses * sizeof(union sockunion));

    /* initialize pointer to other modules of SCTP */
    currentAssociation->flowControl = NULL;
    currentAssociation->reliableTransfer = NULL;
    currentAssociation->rx_control = NULL;
    currentAssociation->streamengine = NULL;

    /* only pathman, bundling and sctp-control are created at this point, the rest is created
       with mdi_initAssociation */
    currentAssociation->bundling = bu_new();
    currentAssociation->pathMan = pm_newPathman(noOfDestinationAddresses,
                                                primaryDestinitionAddress, sctpInstance);
    currentAssociation->sctp_control = sci_newSCTP_control(sctpInstance);

    event_logi(INTERNAL_EVENT_1, "new Association created ID=%08x", currentAssociation->tagLocal);

    /* check if newly created association already exists. */
    if (checkForExistingAssociations(currentAssociation)) {
        error_log(ERROR_MAJOR, "tried to establish an existing association");
        /* FIXME : also free bundling, pathmanagement,sctp_control */
        free(currentAssociation);
        currentAssociation = NULL;
        return 1;
    }

    /* Enter association into list */
    enterAssociation(currentAssociation);

    return 0;

}                               /* end: mdi_newAssociation */


/**
 * This is the second function needed to fully create and initialize an association (after
 * mdi_newAssociation()) THe association is created in two steps because data become available
 * at the a-side in two steps
 * \begin{enumerate}
 * \item associate
 * \item init acknowledgement
 * \end{enumerate}
 * At the z-side, with the cookie message all data is available at once. So mdi_newAssociation
 * and mdi_initAssociation must be called when the initAck with valid Cookie is received.
 *
 * @param  remoteSideReceiverWindow  rwnd size that the peer allowed in this association
 * @param  noOfInStreams  number of incoming (receive) streams after negotiation
 * @param  noOfOutStreams number of outgoing (send) streams after negotiation
 * @param  remoteInitialTSN     initial  TSN of the peer
 * @param  tagRemote            tag of the peer
 * @param  localInitialTSN      my initial TSN, needed for initializing my flow control
 * @return 0 for success, else 1 for error
*/
unsigned short
mdi_initAssociation(unsigned int remoteSideReceiverWindow,
                    unsigned short noOfInStreams,
                    unsigned short noOfOutStreams,
                    unsigned int remoteInitialTSN,
                    unsigned int tagRemote, unsigned int localInitialTSN)
{
    if (!currentAssociation) {
        error_log(ERROR_MAJOR,
                  "mdi_initAssociation: current association does not exist, can not initialize");
        return 1;
    }

    /* if  mdi_initAssociation has already be called, delete modules and make new ones
       with possibly new data. Multiple calls of of mdi_initAssociation can occur on the
       a-side in the case of stale cookie errors. */
    if (currentAssociation->tagRemote != 0) {
        event_log(INTERNAL_EVENT_1,
                  "Deleting Modules in mdi_initAssociation() -- then recreating them !!!!");
        /* association init was already completed */
        fc_delete_flowcontrol(currentAssociation->flowControl);
        rtx_delete_reltransfer(currentAssociation->reliableTransfer);
        rxc_delete_recvctrl(currentAssociation->rx_control);
        se_delete_stream_engine(currentAssociation->streamengine);
    }

    /* TODO : check number of input and output streams (although that should be fixed now) */

    currentAssociation->tagRemote = tagRemote;
    currentAssociation->reliableTransfer =
        (void *) rtx_new_reltransfer(currentAssociation->noOfNetworks);
    currentAssociation->flowControl =
        (void *) fc_new_flowcontrol(remoteSideReceiverWindow, localInitialTSN,
                                    currentAssociation->noOfNetworks);

    currentAssociation->rx_control = (void *) rxc_new_recvctrl(remoteInitialTSN,
                                                               currentAssociation->sctpInstance);
    currentAssociation->streamengine = (void *) se_new_stream_engine(noOfInStreams, noOfOutStreams);
    event_logi(INTERNAL_EVENT_1,
               "second step of association initialisation performed ID=%08x",
               currentAssociation->tagLocal);

    return 0;

}                               /* end: mdi_initAssociation */


/**
 *  mdi_deleteCurrentAssociation deletes the current association.
 *
 *  The association will not be deleted at once, but is only marked for deletion. This is done in
 *  this way to allow other modules to finish their current activities. To prevent them to start
 *  new activities, the currentAssociation pointer is set to NULL.
 */
void mdi_deleteCurrentAssociation()
{
    short pathID;

    if (currentAssociation != NULL) {
        if (currentAssociation->tagRemote != 0) {
            /* stop timers */
            for (pathID = 0; pathID < currentAssociation->noOfNetworks; pathID++)
                pm_disableHB(pathID);

            fc_stop_timers();
            rxc_stop_sack_timer();
        }

        /* mark association as deleted, it will be deleted when retrieveAssociation(..) encounters
           a "deleted" association. */
        currentAssociation->deleted = TRUE;
        event_logi(INTERNAL_EVENT_1, "association ID=%08x marked for deletion",
                   currentAssociation->tagLocal);
        mdi_clearAssociationData(currentAssociation->tagLocal);
    } else {
        error_log(ERROR_MAJOR,
                  "mdi_deleteAssociation: current association does not exist, can not delete");
    }
}
