/*
 *  $Id: flowcontrol.c,v 1.10 2001/03/16 17:49:44 ajung Exp $
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module implements most parts of the flow control mechanisms
 *
 */

#include "flowcontrol.h"
#include "bundling.h"
#include "adaptation.h"
#include "recvctrl.h"
#include <stdlib.h>             /* for malloc() and friends */
#include <stdio.h>

/* here we should not have to worry about wrap */
#define max(x,y)            (x)>(y)?(x):(y)
#define min(x,y)            (x)<(y)?(x):(y)


/**
 * this struct contains all relevant congestion control parameters for
 * one PATH to the destination/association peer endpoint
 */
typedef struct __congestion_parameters
{
    //@{
    ///
    unsigned int cwnd;
    ///
    unsigned int cwnd2;
    ///
    unsigned int partial_bytes_acked;
    ///
    unsigned int ssthresh;
    ///
    unsigned int outstanding_bytes_per_address;
    ///
    unsigned int mtu;
    //@}
} cparm;

typedef struct flowcontrol_struct
{
    //@{
    ///
    unsigned int outstanding_bytes;
    ///
    unsigned int number_of_addresses;
    /** pointer to array of congestion window parameters */
    cparm *cparams;
    ///
    unsigned int current_tsn;
    ///
    List *chunk_list;
    ///
    struct timeval time_of_adjustment;
    ///
    unsigned int last_active_address;
    ///
    TimerID cwnd_timer;
    /** one timer may be running per destination address */
    TimerID *T3_timer;
    /** for passing as parameter in callback functions */
    unsigned int *addresses;
    ///
    unsigned int my_association;
    ///
    boolean shutdown_received;
    ///
    boolean waiting_for_sack;
    ///
    boolean t3_retransmission_sent;
    ///
    boolean one_packet_inflight;
    //@}
} fc_data;


/* ---------------  Function Prototypes -----------------------------*/
int fc_check_for_txmit(void *fc_instance);
int fc_do_timerbased_rtxmit(void *fc_instance, unsigned int num_of_chunks,
                            unsigned int timed_out_address);
/* ---------------  Function Prototypes -----------------------------*/


/**
 * Creates new instance of flowcontrol module and returns pointer to it
 * TODO : should parameter be unsigned short ?
 * TODO : get and update MTU (guessed values ?) per destination address
 * @param  peer_rwnd receiver window that peer allowed us when setting up the association
 * @param  my_iTSN my initial TSN value
 * @param  number_of_destination_addresses the number of paths to the association peer
 * @return  pointer to the new fc_data instance
*/
void *fc_new_flowcontrol(unsigned int peer_rwnd,
                         unsigned int my_iTSN, unsigned int number_of_destination_addresses)
{
    fc_data *tmp;
    unsigned int count;
    DLL_Return ExitCode;

    tmp = malloc(sizeof(fc_data));
    if (!tmp)
        error_log(ERROR_FATAL, "Malloc failed");
    tmp->current_tsn = my_iTSN;

    event_logi(VERBOSE,
               "========= Num of number_of_destination_addresses = %d \n",
               number_of_destination_addresses);

    tmp->cparams = malloc(number_of_destination_addresses * sizeof(cparm));
    if (!tmp->cparams)
        error_log(ERROR_FATAL, "Malloc failed");

    tmp->T3_timer = malloc(number_of_destination_addresses * sizeof(TimerID));
    if (!tmp->T3_timer)
        error_log(ERROR_FATAL, "Malloc failed");

    tmp->addresses = malloc(number_of_destination_addresses * sizeof(unsigned int));
    if (!tmp->addresses)
        error_log(ERROR_FATAL, "Malloc failed");

    for (count = 0; count < number_of_destination_addresses; count++) {
        tmp->T3_timer[count] = 0; /* i.e. timer not running */
        tmp->addresses[count] = count;
        (tmp->cparams[count]).cwnd = 2 * MAX_MTU_SIZE;
        (tmp->cparams[count]).cwnd2 = 0L;
        (tmp->cparams[count]).partial_bytes_acked = 0L;
        (tmp->cparams[count]).ssthresh = peer_rwnd;
        (tmp->cparams[count]).outstanding_bytes_per_address = 0L;
        (tmp->cparams[count]).mtu = MAX_SCTP_PDU;
    }
    tmp->outstanding_bytes = 0;
    tmp->last_active_address = pm_readPrimaryPath();
    tmp->number_of_addresses = number_of_destination_addresses;
    tmp->waiting_for_sack = FALSE;
    tmp->shutdown_received = FALSE;
    tmp->t3_retransmission_sent = FALSE;
    tmp->one_packet_inflight = FALSE;

    rtx_set_remote_receiver_window(peer_rwnd);

    if (DLL_CreateList(&(tmp->chunk_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if ((ExitCode = DLL_InitializeList(tmp->chunk_list, sizeof(chunk_data)))
        != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    gettimeofday(&(tmp->time_of_adjustment), NULL);

    tmp->my_association = mdi_readAssociationID();
    event_logi(VVERBOSE, "FlowControl : Association-ID== %d \n", tmp->my_association);
    if (tmp->my_association == 0)
        error_log(ERROR_FATAL, "Association was not set, should be......");
    return tmp;
}

/**
 * this function stops all currently running timers, and may be called when
 * the shutdown is imminent
 * @param  new_rwnd new receiver window of the association peer
 */
void fc_restart(guint32 new_rwnd)
{
    fc_data *tmp;
    guint32 count;
    DLL_Return ExitCode;

    tmp = (fc_data *) mdi_readFlowControl();
    event_log(INTERNAL_EVENT_0, "fc_restart()... ");
    if (!tmp) {
        error_log(ERROR_MINOR, "fc_data instance not set !");
        return;
    }
    fc_stop_timers();
    for (count = 0; count < tmp->number_of_addresses; count++) {
        (tmp->cparams[count]).cwnd = 2 * MAX_MTU_SIZE;
        (tmp->cparams[count]).cwnd2 = 0L;
        (tmp->cparams[count]).partial_bytes_acked = 0L;
        (tmp->cparams[count]).ssthresh = new_rwnd;
        (tmp->cparams[count]).outstanding_bytes_per_address = 0L;
        (tmp->cparams[count]).mtu = MAX_SCTP_PDU;
    }
    tmp->outstanding_bytes = 0;
    tmp->waiting_for_sack = FALSE;
    tmp->shutdown_received = FALSE;
    tmp->t3_retransmission_sent = FALSE;
    tmp->one_packet_inflight = FALSE;
    rtx_set_remote_receiver_window(new_rwnd);
    if (DLL_IsListEmpty(tmp->chunk_list) == DLL_FALSE)
        /* TODO : pass chunks in this list back up to the ULP ! */
        error_log(ERROR_MAJOR, "FLOWCONTROL RESTART : List is deleted with chunks still queued...");

    DLL_DestroyList(&(tmp->chunk_list));
    /* recreate new list */
    if (DLL_CreateList(&(tmp->chunk_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if ((ExitCode = DLL_InitializeList(tmp->chunk_list, sizeof(chunk_data)))
        != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    gettimeofday(&(tmp->time_of_adjustment), NULL);


}

/**
 * Deletes data occupied by a flow_control data structure
 * @param fc_instance pointer to the flow_control data structure
 */
void fc_delete_flowcontrol(void *fc_instance)
{
    fc_data *tmp;

    tmp = (fc_data *) fc_instance;
    event_log(INTERNAL_EVENT_0, "deleting flowcontrol");
    free(tmp->cparams);
    free(tmp->T3_timer);
    free(tmp->addresses);
    if (DLL_IsListEmpty(tmp->chunk_list) == DLL_FALSE)
        error_log(ERROR_MINOR, "FLOWCONTROL : List is deleted with chunks still queued...");
    DLL_DestroyList(&(tmp->chunk_list));
    free(fc_instance);
}


/**
 * helper function for sorting list of chunks in tsn order
 * @param  one pointer to chunk data
 * @param  two pointer to other chunk data
 * @return 0 if chunks have equal tsn, -1 if tsn1 < tsn2, 1 if tsn1 > tsn2
 */
int fc_sort_tsn(chunk_data * one, chunk_data * two)
{
    if (before(one->chunk_tsn, two->chunk_tsn)) {
        return -1;
    } else if (after(one->chunk_tsn, two->chunk_tsn)) {
        return 1;
    } else                      /* one==two */
        return 0;
}

/**
 * function to print debug data (flow control parameters of all paths)
 *  @param event_log_level  INTERNAL_EVENT_0 INTERNAL_EVENT_1 EXTERNAL_EVENT_X EXTERNAL_EVENT
 */
void fc_debug_cparams(short event_log_level)
{
    fc_data *fc;
    unsigned int count;

    if (event_log_level <= Current_event_log_) {
        fc = (fc_data *) mdi_readFlowControl();
        if (!fc) {
            error_log(ERROR_MAJOR, "fc_data instance not set !");
            return;
        }
        event_log(event_log_level,
                  "----------------------------------------------------------------------");
        event_log(event_log_level, "Debug-output for Congestion Control Parameters ! ");
        event_logii(event_log_level,
                    "outstanding_bytes == %u; current_tsn == %u; ",
                    fc->outstanding_bytes, fc->current_tsn);
        event_logii(event_log_level,
                    "chunks queued in flowcontrol== %lu; last_active_address == %u ",
                    DLL_GetNumberOfRecords(fc->chunk_list), fc->last_active_address);
        event_logii(event_log_level,
                    "shutdown_received == %s; waiting_for_sack == %s",
                    ((fc->shutdown_received == TRUE) ? "TRUE" : "FALSE"),
                    ((fc->waiting_for_sack == TRUE) ? "TRUE" : "FALSE"));

        event_logi(event_log_level, "t3_retransmission_sent == %s ",
                   ((fc->t3_retransmission_sent == TRUE) ? "TRUE" : "FALSE"));
        for (count = 0; count < fc->number_of_addresses; count++) {
            event_logiiii(event_log_level,
                          "%u : outstanding per address=%u   cwnd=%u   ssthresh=%u",
                          count,
                          (fc->cparams[count]).outstanding_bytes_per_address,
                          (fc->cparams[count]).cwnd, (fc->cparams[count]).ssthresh);
            event_logiiiii(event_log_level,
                           "%u :  mtu=%u   T3=%u   cwnd2=%u   pb_acked=%u",
                           count, (fc->cparams[count]).mtu,
                           fc->T3_timer[count], (fc->cparams[count]).cwnd2,
                           (fc->cparams[count]).partial_bytes_acked);
        }
        event_log(event_log_level,
                  "----------------------------------------------------------------------");

    }
    return;
}


/**
 * this function should be called to signal to flowcontrol, that our ULP
 * has initiated a shutdown procedure. We must only send unacked data from
 * now on ! The association is about to terminate !
 */
void fc_shutdown()
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();
    event_log(VERBOSE, "fc_shutdown()... ");
    if (!fc) {
        error_log(ERROR_MINOR, "fc_data instance not set !");
        return;
    }
    fc->shutdown_received = TRUE;
    return;
}

/**
 * this function stops all currently running timers of the flowcontrol module
 * and may be called when the shutdown is imminent
 */
void fc_stop_timers()
{
    fc_data *fc;
    int count, result;

    fc = (fc_data *) mdi_readFlowControl();
    event_log(INTERNAL_EVENT_0, "fc_stop_timers()... ");
    if (!fc) {
        error_log(ERROR_MINOR, "fc_data instance not set !");
        return;
    }
    for (count = 0; count < fc->number_of_addresses; count++) {
        if (fc->T3_timer[count] != 0) {
            result = sctp_stopTimer(fc->T3_timer[count]);
            fc->T3_timer[count] = 0;
            if (result == 1)
                error_log(ERROR_MINOR, "Timer not correctly reset to 0 !");
            event_logii(VVERBOSE, "Stopping T3-Timer(%d) = %d ", count, result);
        }
    }
    if (fc->cwnd_timer != 0) {
        result = sctp_stopTimer(fc->cwnd_timer);
        fc->cwnd_timer = 0;
        if (result == 1)
            error_log(ERROR_MINOR, "Timer not correctly reset to 0 !");
        event_logi(VVERBOSE, "Stopping cwnd_timer = %d ", result);
    }

}


/**
 *  timer controlled callback function, that reduces cwnd, if data is not sent within a certain time.
 *  As all timer callbacks, it takes three arguments, the timerID, and two pointers to relevant data
 *  @param  tid the id of the timer that has gone off
 *  @param  assoc  pointer to the association structure, where cwnd needs to be reduced
 *  @param  data2  currently unused == NULL
 */
void fc_timer_cb_reduce_cwnd(TimerID tid, void *assoc, void *data2)
{
    /* TODO : verify that last_active_address is always set correctly */
    unsigned int ad_idx;
    fc_data *fc;
    int res;
    unsigned int rto;
    event_log(INTERNAL_EVENT_0, "----------------> fc_timer_cb_reduce_cwnd <------------------");
    event_logi(VVERBOSE, "Association ID == %d \n", *(unsigned int *) assoc);
    res = mdi_setAssociationData(*(unsigned int *) assoc);
    if (res == 1)
        error_log(ERROR_MAJOR, " association does not exist !");
    if (res == 2) {
        error_log(ERROR_MAJOR, "Association was not cleared..... !!!");
        /* failure treatment ? */
    }

    fc = (fc_data *) mdi_readFlowControl();
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return;
    }
    fc->cwnd_timer = 0L;
    /* function does nothing and effectively stops this timer if list is not empty */
    ad_idx = fc->last_active_address;
    if (DLL_IsListEmpty(fc->chunk_list) == DLL_TRUE) {
        rto = pm_readRTO(ad_idx);
        event_logii(INTERNAL_EVENT_0, "RTO for address %d was : %d msecs\n", ad_idx, rto);
        if (rto < 2)
            rto = 2;
        /* we have nothing to send, so cut cwnd in half ! */
        fc->cparams[ad_idx].cwnd = max(2 * MAX_MTU_SIZE, (fc->cparams[ad_idx].cwnd) / 2);
        event_logii(INTERNAL_EVENT_0,
                    "updating cwnd[%d], setting it to : %d\n", ad_idx, fc->cparams[ad_idx].cwnd);
        /* is this call inside the if, or outside the if ???? */
        /* do these pointers survive ? */
        /* timer is only restarted now, if list is empty */
        fc->cwnd_timer = sctp_startTimer(rto, &fc_timer_cb_reduce_cwnd, assoc, data2);
    }
    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */
    mdi_clearAssociationData(*(unsigned int *) assoc);
    return;
}



/**
 * function that selects destination index for data chunks when they are sent,
 * or possibly new address when they are retransmitted.
 * @param  fc   pointer to the flow control structure
 * @param  dat  pointer to the data chunk that is to be sent
 * @param  data_retransmitted   has the chunk already been transmitted ?
 * @param  old_destination      if so, we pass a pointer to the index of the last address used, else NULL
 * @return index of the address where we should send this chunk to, now
 */
unsigned int
fc_select_destination(fc_data * fc, chunk_data * dat,
                      boolean data_retransmitted, unsigned int *old_destination)
{
    /* TODO : check for number_of_addresses == 1, ==2 */
    unsigned int count, count2;
    short prim, initial,last;
    boolean use_old_destination = TRUE, use_primary = TRUE;

    prim = pm_readPrimaryPath();

    event_logiii(VVERBOSE,
                 "fc_select_destination: chunk-tsn=%u, retrans=%s, primary path=%d ",
                 dat->chunk_tsn, ((data_retransmitted == TRUE) ? "TRUE" : "FALSE"), prim);
    if (old_destination) {
        event_logi(VERBOSE, "fc_select_destination: old_dest = %u\n", *old_destination);
    } else {
        event_log(VERBOSE, "fc_select_destination: old_dest = NULL Pointer \n");
    }

    /* return  a value that is equal to old_destination, if possible */
    if (old_destination) {
        for (count = 0; (count < dat->num_of_transmissions) && (count < MAX_DEST); count++) {
            if ((dat->used_destinations[count]) == *old_destination)
                use_old_destination = FALSE;
        }
        if ((use_old_destination == TRUE)
            && (pm_readState(*old_destination) == PM_ACTIVE))
            return *old_destination;
    }

    /* try primary or user selected address */
    if (dat->initial_destination == -1)
        initial = prim;
    else
        initial = (short) dat->initial_destination;

    if ((data_retransmitted == FALSE) && (pm_readState(initial) == PM_ACTIVE))
        return initial;

    /* or else an active address not in the list of already used addresses */
    for (count = 0; (count < dat->num_of_transmissions) && (count < MAX_DEST); count++) {
        /* did we already use primary ??? */
        if ((dat->used_destinations[count]) == prim)
            use_primary = FALSE;
    }
    if ((use_primary == TRUE) && (pm_readState(prim) == PM_ACTIVE))
        return prim;


    if ((data_retransmitted == TRUE) && (dat->num_of_transmissions >= 1))
        last = dat->used_destinations[dat->num_of_transmissions-1];
    else
        last = 0;

    count2 = 0;
    do {
        last = (last + 1) %  fc->number_of_addresses;
        count2++;
    } while ((pm_readState(last) == PM_INACTIVE) && (count2<fc->number_of_addresses));

    if (count2 < fc->number_of_addresses) {
        event_logi(VERBOSE,"fc_select_destination: returning address=%u \n",last);
        return last;
    }

   event_logi(VERBOSE,"fc_select_destination: returning last active address=%u \n",
            fc->last_active_address);
        /* last resort..... */
   return fc->last_active_address;
}


/**
 *  timer controlled callback function, called when T3 timer expires and data must be retransmitted
 *  This timer also adjusts the slow start threshold and cwnd values
 *  As all timer callbacks, it takes three arguments, the timerID, and two pointers to relevant data
 *  @param  tid the id of the timer that has gone off
 *  @param  assoc  pointer to the association structure to which this T3 timer belongs
 *  @param  data2  pointer to the index of the address where  T3 timer had been running
 */
void fc_timer_cb_t3_timeout(TimerID tid, void *assoc, void *data2)
{
    fc_data *fc;
    DLL_Return ExitCode;
    unsigned int ad_idx, res, retransmitted_bytes = 0;
    unsigned int peer_rwnd;
    int count;
    int num_of_chunks;
    chunk_data **chunks;
    int (*pfun)() = NULL;

    pfun = rtx_sort_tsn;

    res = mdi_setAssociationData(*(unsigned int *) assoc);
    if (res == 1)
        error_log(ERROR_MAJOR, " association does not exist !");
    if (res == 2) {
        error_log(ERROR_MAJOR, "Association was not cleared..... !!!");
        /* failure treatment ? */
    }

    fc = (fc_data *) mdi_readFlowControl();
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return;
    }
    ad_idx = *((unsigned int *) data2);

    event_logi(INTERNAL_EVENT_0, "===============> fc_timer_cb_t3_timeout(address=%u) <========", ad_idx);

    fc->T3_timer[ad_idx] = 0;
    /* adjust ssthresh, cwnd - section 6.3.3.E1, respectively 7.2.3) */
    fc->cparams[ad_idx].ssthresh = max(fc->cparams[ad_idx].cwnd / 2, 2 * fc->cparams[ad_idx].mtu);
    fc->cparams[ad_idx].cwnd = fc->cparams[ad_idx].mtu;

    num_of_chunks = rtx_readNumberOfUnackedChunks();
    event_logii(INTERNAL_EVENT_0, "Address-Index : %u, Number of Chunks==%d",
                ad_idx, num_of_chunks);

    if (num_of_chunks == 0) {
        event_log(VERBOSE, "Number of Chunks was 0 BEFORE calling rtx_t3_timeout - returning");
        if (fc->shutdown_received == TRUE)
            error_log(ERROR_MAJOR,
                      "T3 Timeout with 0 chunks in rtx-queue,  sci_allChunksAcked() should have been called !");
        mdi_clearAssociationData(*(unsigned int *) assoc);
        return;
    }
    chunks = malloc(num_of_chunks * sizeof(chunk_data *));
    num_of_chunks = rtx_t3_timeout(&(fc->my_association), ad_idx, fc->cparams[ad_idx].mtu, chunks);
    if (num_of_chunks == 0) {
        event_log(VERBOSE, "Number of Chunks was 0 AFTER calling rtx_t3_timeout - returning");
        free(chunks);
        mdi_clearAssociationData(*(unsigned int *) assoc);
        return;
    }
    for (count = 0; count < num_of_chunks; count++) {
        retransmitted_bytes += chunks[count]->chunk_len;
        event_logi(VERBOSE, "fc_timer_cb_t3_timeout: Got TSN==%u for RTXmit\n",
                   chunks[count]->chunk_tsn);

    }
    /* section 6.2.1.C */
    peer_rwnd = rtx_read_remote_receiver_window();
    peer_rwnd += retransmitted_bytes;
    rtx_set_remote_receiver_window(peer_rwnd);

    /* insert chunks to be retransmitted at the beginning of the list */
    ExitCode = DLL_CurrentPointerToHead(fc->chunk_list);
    for (count = num_of_chunks - 1; count >= 0; count--) {
        ExitCode = DLL_AddRecordUnique(fc->chunk_list, chunks[count], pfun);
        /*  make sure, these go into the TRANSMISSION queue only once ! */
        if ((ExitCode != DLL_NORMAL) && (ExitCode != DLL_NOT_MODIFIED)) {
            DLL_error_log(ERROR_FATAL, ExitCode);
            return;
        } else if (ExitCode == DLL_NOT_MODIFIED) {
            event_logi(VERBOSE, "Chunk number %u already in list, skipped adding it", chunks[count]->chunk_tsn);
        }
    }
    event_log(VVERBOSE, "FlowControl (T3 timeout): Chunklist after reinserting chunks \n");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    free(chunks);

    /* section 7.2.3 : assure that only one data packet is in flight, until a new sack is received */
    fc->waiting_for_sack = TRUE;
    fc->t3_retransmission_sent = FALSE; /* we may again send one packet ! */
    fc->one_packet_inflight = FALSE;

    fc_debug_cparams(VERBOSE);

    fc_do_timerbased_rtxmit(fc, num_of_chunks, ad_idx);
    mdi_clearAssociationData(*(unsigned int *) assoc);
    return;
}

/**
 * function increases chunk's number of transmissions, stores used destination, updates counts per addresses
 */
void fc_update_chunk_data(fc_data * fc, chunk_data * dat, unsigned int destination)
{
    unsigned int rwnd;

    rwnd = rtx_read_remote_receiver_window();
    dat->num_of_transmissions++;
    event_logiii(VERBOSE,
                 "fc_update_chunk_data(),dat->TSN=%u, dat->num_of_transmissions %d , dest %d\n",
                 dat->chunk_tsn, dat->num_of_transmissions, destination);
    if (dat->num_of_transmissions >= MAX_DEST) {
        error_log(ERROR_MINOR, "Maximum number of assumed transmissions exceeded ");
        dat->num_of_transmissions = MAX_DEST - 1;
    }
    if (dat->num_of_transmissions < 1)
        error_log(ERROR_MAJOR, "Somehow dat->num_of_transmissions became less than 1 !");

    /* this time we will send dat to destination */
    dat->used_destinations[dat->num_of_transmissions - 1] = destination;

    if (dat->num_of_transmissions == 1) /* chunk is being transmitted the first time */
        fc->outstanding_bytes += dat->chunk_len;
    fc->cparams[destination].outstanding_bytes_per_address += dat->chunk_len;

    if (dat->num_of_transmissions > 1) { /* chunk is being retransmitted */
        fc->cparams[dat->used_destinations[dat->num_of_transmissions - 2]].outstanding_bytes_per_address =
            (fc->cparams[dat->used_destinations[dat->num_of_transmissions-2]].outstanding_bytes_per_address < dat->chunk_len) ?
                0 : (fc->cparams[dat->used_destinations[dat->num_of_transmissions-2]].outstanding_bytes_per_address -
                                    dat->chunk_len);

        event_logii(VVERBOSE,
                    "fc_update_chunk_data(retransmission) : outstanding_bytes_per_address(%u)=%u bytes\n",
                    dat->used_destinations[dat->num_of_transmissions - 2],
                    fc->cparams[dat->used_destinations[dat->num_of_transmissions -
                                                       2]].outstanding_bytes_per_address);
    }

    /* section 6.2.1.B */
    if (dat->chunk_len >= rwnd)
        rtx_set_remote_receiver_window(0);
    else
        rtx_set_remote_receiver_window(rwnd - dat->chunk_len);

    event_logii(VERBOSE, "outstanding_bytes_per_address(%u)=%u bytes\n",
                destination, fc->cparams[destination].outstanding_bytes_per_address);
    return;
}


/**
 *  function that checks whether we may transmit data that is currently in the send queue.
 *  Any time that some data chunk is added to the send queue, we must check, whether we can send
 *  the chunk, or must wait until cwnd opens up.
 *  @param fc_instance  pointer to the flowcontrol instance used here
 *  @return  0 for success, -1 for error
 */
int fc_check_for_txmit(void *fc_instance)
{
    int result;
    fc_data *fc;
    DLL_Return ExitCode;
    chunk_data *dat;
    unsigned int total_size, bundling_size, destination, tmp, chunk_len, peer_rwnd, rto_time;
    boolean data_is_retransmitted = FALSE;
    boolean data_is_submitted = FALSE;

    peer_rwnd = rtx_read_remote_receiver_window();

    event_logi(INTERNAL_EVENT_0, "Entering fc_check_for_txmit(rwnd=%u)... ", peer_rwnd);

    fc = (fc_data *) fc_instance;
    /* ------------------ DEBUGGING ----------------------------- */
    /*  chunk_list_debug(VVERBOSE, fc->chunk_list); */
    fc_debug_cparams(VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    if ((ExitCode = DLL_CurrentPointerToHead(fc->chunk_list)) != DLL_NORMAL) {
        /* this function should only be called IF there IS data in that list */
        DLL_error_log(ERROR_MAJOR, ExitCode);
        return -1;
    }
    dat = DLL_GetCurrentPointer(fc->chunk_list);
    if (dat->num_of_transmissions >= 1)
        data_is_retransmitted = TRUE;
    destination = fc_select_destination(fc, dat, data_is_retransmitted, NULL);

    total_size = 0;
    bundling_size = 0;
    chunk_len = dat->chunk_len;
    event_logiii(VERBOSE,
                 "Called fc_select_destination == %d, chunk_len=%u, outstanding(p.add)=%u",
                 destination, chunk_len, fc->cparams[destination].outstanding_bytes_per_address);
    event_logiiii(VERBOSE, "cwnd(%u) == %u, mtu == %u, MAX_MTU = %d ",
                  destination, fc->cparams[destination].cwnd,
                  fc->cparams[destination].mtu, MAX_MTU_SIZE);

    if (peer_rwnd == 0) {       /* section 6.1.A */
        if (fc->cparams[destination].cwnd <= fc->cparams[destination].outstanding_bytes_per_address)
            return 0;
        if (fc->one_packet_inflight == TRUE)
            return 0;
    } else if (fc->cparams[destination].cwnd <= /* section 6.1.B */
               fc->cparams[destination].outstanding_bytes_per_address) {
        event_logiii(VERBOSE, "NOT SENDING (cwnd=%u, outstanding(%u)=%u",
                     fc->cparams[destination].cwnd, destination,
                     fc->cparams[destination].outstanding_bytes_per_address);
        return 0;
    }

    if (fc->waiting_for_sack == TRUE && data_is_retransmitted == TRUE) {
        /* make sure we send only one retransmission after T3 timeout */
        if (fc->t3_retransmission_sent == TRUE)
            return 0;
    }


    while ((dat != NULL) &&
           (total_size +
            fc->cparams[destination].outstanding_bytes_per_address <
            fc->cparams[destination].cwnd) && (rtx_read_remote_receiver_window() > chunk_len)) {

        /* size is used to see, whether we may send this next chunk, too */
        total_size += dat->chunk_len;
        bundling_size += dat->chunk_len;

        event_logiii(VVERBOSE, "Chunk: len=%u, tsn=%u, gap_reports=%u",
                     dat->chunk_len, dat->chunk_tsn, dat->gap_reports);
        event_logii(VVERBOSE, "Chunk: ack_time=%d, num_of_transmissions=%u",
                    dat->ack_time, dat->num_of_transmissions);
        adl_gettime(&(dat->transmission_time));
        result = bu_put_Data_Chunk((SCTP_simple_chunk *) dat->data);
        data_is_submitted = TRUE;
        event_logi(VERBOSE, "sent chunk (tsn=%u) to bundling", dat->chunk_tsn);
        event_log(VERBOSE, "Calling fc_update_chunk_data \n");
        fc_update_chunk_data(fc, dat, destination);
        if (dat->num_of_transmissions == 1) {
            event_log(INTERNAL_EVENT_0, "Calling rtx_save_retrans \n");
            result = rtx_save_retrans_chunks(dat);
        } else {
            event_log(INTERNAL_EVENT_0, "Calling rtx_update_retrans \n");
            result = rtx_update_retrans_chunks(dat, destination);
        }
        ExitCode = DLL_DeleteCurrentRecord(fc->chunk_list);

        if (ExitCode == DLL_NORMAL)
            dat = DLL_GetCurrentPointer(fc->chunk_list);
        else
            dat = NULL;

        if (dat != NULL) {
            tmp = fc_select_destination(fc, dat, data_is_retransmitted, &destination);
            chunk_len = dat->chunk_len;
            event_logi(VERBOSE, "Called fc_select_destination == %d\n", destination);
            if (destination != tmp)
                break;
            if (chunk_len + bundling_size >= fc->cparams[destination].mtu) {
                fc->one_packet_inflight = TRUE;
                bu_sendAllChunks(&destination);
                rto_time = pm_readRTO(destination);
                if (fc->cwnd_timer == 0) {
                    fc->cwnd_timer =
                        sctp_startTimer(rto_time, &fc_timer_cb_reduce_cwnd,
                                        &(fc->my_association), NULL);
                    event_logi(INTERNAL_EVENT_0,
                               "fc_check_for_txmit...started reduce-cwnd-Timer going off %u msecs from now",
                               rto_time);
                } else {
                    fc->cwnd_timer = sctp_restartTimer(fc->cwnd_timer, rto_time);
                    event_logi(INTERNAL_EVENT_0,
                               "fc_check_for_txmit...re-started reduce-cwnd-Timer going off %u msecs from now",
                               rto_time);
                }
                data_is_submitted = FALSE;
                fc->last_active_address = destination;
                bundling_size = 0;
            }
        }
    }

    if ((fc->waiting_for_sack == TRUE) && (fc->t3_retransmission_sent == FALSE)) {
        error_log(ERROR_MAJOR, "Possible Retransmission Condition in fc_check_for_txmit !!!!!!!! ");
        if (data_is_submitted == TRUE) {
            /* Keep me from retransmitting more than once */
            fc->t3_retransmission_sent = TRUE;
        }
    }

    event_log(VVERBOSE, "Printing Chunk List / Congestion Params in FlowControl\n");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(VVERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    if (fc->T3_timer[destination] == 0) { /* see section 5.1 */
        fc->T3_timer[destination] =
            sctp_startTimer(pm_readRTO(destination), &fc_timer_cb_t3_timeout,
                            &(fc->my_association), &(fc->addresses[destination]));
        event_logiii(INTERNAL_EVENT_0,
                     "started T3 Timer with RTO(%u)==%u msecs on address %u",
                     destination, pm_readRTO(destination), fc->addresses[destination]);
    } else {                    /* restart only if lowest TSN is being retransmitted, else leave running */
        if (data_is_retransmitted == TRUE) {
            /* see section 6.1 */
            event_logiii(INTERNAL_EVENT_0,
                         "Restarted T3 Timer with RTO(%u)==%u msecs on address %u",
                         destination, pm_readRTO(destination), fc->addresses[destination]);
            fc->T3_timer[destination] =
                sctp_restartTimer(fc->T3_timer[destination], pm_readRTO(destination));
        }
    }

    if (data_is_submitted == TRUE) {
        fc->one_packet_inflight = TRUE;
        bu_sendAllChunks(&destination);
        rto_time = pm_readRTO(destination);
        if (fc->cwnd_timer == 0) {
            fc->cwnd_timer =
                sctp_startTimer(rto_time, &fc_timer_cb_reduce_cwnd, &(fc->my_association), NULL);
            event_logi(INTERNAL_EVENT_0,
                       "fc_check_for_txmit...started reduce-cwnd-Timer going off %u msecs from now",
                       rto_time);
        } else {
            fc->cwnd_timer = sctp_restartTimer(fc->cwnd_timer, rto_time);
            event_logi(INTERNAL_EVENT_0,
                       "fc_check_for_txmit...re-started reduce-cwnd-Timer going off %u msecs from now",
                       rto_time);
        }
        data_is_submitted = FALSE;
        fc->last_active_address = destination;
    }
    return 0;
}

/**
 *  This function does the retransmission algorithm, after T3 timeout.
 *  One packet worth one MTU may be sent at ONCE ! After that, send data
 *  according to cwnd, mtu, outstanding bytes and what else is there
 *  @param  fc_instance    pointer to the used flow control structure
 *  @param  num_of_chunks   number of chunks we may transmit at once
 *                          which are supposed to be less than path mtu
 *  @param  timed_out_address   index of address, where timeout occurrs
 *  @return 0 for success, less than zero for error
 */
int fc_do_timerbased_rtxmit(void *fc_instance, unsigned int num_of_chunks,
                        unsigned int timed_out_address)
{
    int result;
    fc_data *fc;
    DLL_Return ExitCode;
    chunk_data *dat;
    unsigned int size = 0, destination, tmp, number_put = 0;
    gboolean data_is_retransmitted = FALSE;
    gboolean removed_association = FALSE;

    event_logi(INTERNAL_EVENT_0, "Entering fc_do_timerbased_rtxmit(%u chunks)", num_of_chunks);

    fc = (fc_data *) fc_instance;
    /* ------------------ DEBUGGING ----------------------------- */
    /*  chunk_list_debug(VVERBOSE, fc->chunk_list); */
    fc_debug_cparams(VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    if (num_of_chunks == 0) {
        error_log(ERROR_MAJOR,
                  "There were no chunks to retransmit...timer should have been canceled");
        return -1;
    }

    if ((ExitCode = DLL_CurrentPointerToHead(fc->chunk_list)) != DLL_NORMAL) {
        /* this function should only be called IF there IS data in that list */
        DLL_error_log(ERROR_MAJOR, ExitCode);
        return -2;
    }
    dat = DLL_GetCurrentPointer(fc->chunk_list);
    data_is_retransmitted = TRUE;
    destination = fc_select_destination(fc, dat, data_is_retransmitted, NULL);

    event_logiii(VERBOSE, "Called fc_select_destination == %d, size=%u, outstanding(p.add)=%u",
                 destination, size, fc->cparams[destination].outstanding_bytes_per_address);
    event_logiiii(VERBOSE, "cwnd(%u) == %u, mtu == %u, MAX_MTU = %d ",
                    destination, fc->cparams[destination].cwnd,
                    fc->cparams[destination].mtu, MAX_MTU_SIZE);

    while ((dat != NULL) &&
           (number_put < num_of_chunks) ) {
        event_logiii(VVERBOSE, "Chunk: len=%u, tsn=%u, gap_reports=%u",
                     dat->chunk_len, dat->chunk_tsn, dat->gap_reports);
        event_logii(VVERBOSE, "Chunk: ack_time=%d, num_of_transmissions=%u",
                    dat->ack_time, dat->num_of_transmissions);
        adl_gettime(&(dat->transmission_time));
        result = bu_put_Data_Chunk((SCTP_simple_chunk *) dat->data);
        number_put++;
        event_logi(VERBOSE, "sent chunk (tsn=%u) to bundling", dat->chunk_tsn);
        event_log(VERBOSE, "Calling fc_update_chunk_data \n");
        fc_update_chunk_data(fc, dat, destination);
        if (dat->num_of_transmissions == 1) {
            error_log(ERROR_MAJOR, "Chunks has number of transmissions set too low\n");
        } else {
            event_log(VERBOSE, "Calling rtx_update_retrans \n");
            result = rtx_update_retrans_chunks(dat, destination);
        }
        ExitCode = DLL_DeleteCurrentRecord(fc->chunk_list);
        if (ExitCode == DLL_NORMAL)
            dat = DLL_GetCurrentPointer(fc->chunk_list);
        else
            dat = NULL;

        if (dat != NULL) {
            tmp = fc_select_destination(fc, dat, data_is_retransmitted, &destination);
            event_logi(VERBOSE, "Called fc_select_destination == %d\n", destination);
            if (destination != tmp) {
                error_log(ERROR_MINOR,
                          "Next Chunk could not be retransmitted to the same address as the rest");
                break;
            }
        }
    }

    if (number_put != num_of_chunks)
        error_log(ERROR_MAJOR,
                  "Put less chunks in packet for retransmission than we got from rtx_t3_timeout()");


    if ((fc->waiting_for_sack == TRUE) && (fc->t3_retransmission_sent == FALSE)) {
        if (number_put > 0) {
            /* Keep me from retransmitting more than once */
            fc->t3_retransmission_sent = TRUE;
        }
    }

    event_log(VVERBOSE, "Printing    List / Congestion Params in FlowControl\n");
    chunk_list_debug(VVERBOSE, fc->chunk_list);
    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(VVERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    if (number_put > 0) {
        fc->one_packet_inflight = TRUE;
        bu_sendAllChunks(&destination);
        fc->last_active_address = destination;
        removed_association = pm_chunksRetransmitted((short) timed_out_address);
        if (removed_association) {
            event_log(INTERNAL_EVENT_0, "fc_do_timerbased_rtxmit: Association was terminated by pm_chunksRetransmitted()");
            return 0;
        }
    }

    pm_rto_backoff(destination);

    if (fc->T3_timer[destination] == 0) { /* see section 5.1 */
        fc->T3_timer[destination] =
            sctp_startTimer(pm_readRTO(destination), &fc_timer_cb_t3_timeout,
                            &(fc->my_association), &(fc->addresses[destination]));
        event_logiii(INTERNAL_EVENT_0,
                     "started T3 Timer with RTO(%u)==%u msecs on address %u",
                     destination, pm_readRTO(destination), fc->addresses[destination]);
    }

    /* fc_check_for_txmit(fc); */

    return 0;
}

/*
  this function checks whether T3 may be stopped, restarted or left running
 @param ad_idx  index of the destination address concerned (which may have a T3 timer running)
 @param all_acked   has all data been acked ?
 @param new_acked   have new chunks been acked ? CHECKME : has the ctsna advanced ?
*/
void fc_check_t3(unsigned int ad_idx, boolean all_acked, boolean new_acked)
{
    fc_data *fc;
    int result;
    fc = (fc_data *) mdi_readFlowControl();

    event_logiii(INTERNAL_EVENT_0,
                 "fc_check_t3(%u,all_acked=%s, new_acked=%s)... ", ad_idx,
                 (all_acked == TRUE) ? "true" : "false", (new_acked == TRUE) ? "true" : "false");

    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return;
    }
    if (all_acked == TRUE) {
        if (fc->T3_timer[ad_idx] != 0) {
            result = sctp_stopTimer(fc->T3_timer[ad_idx]);
            event_logi(INTERNAL_EVENT_0, "Stopped T3 Timer, Result was %d ", result);
            fc->T3_timer[ad_idx] = 0;
        }
        return;
    }

    /*
       6.3.2 R3) Whenever a SACK is received that acknowledges new data chunks
       including the one with the earliest outstanding TSN on that address,
       restart T3-rxt timer of that address with its current RTO. */

    if (new_acked == TRUE) {
        /* 6.2.4.4) Restart T3, if SACK acked lowest outstanding tsn, OR
         *                      we are retransmitting the first outstanding data chunk
         */
        if (fc->T3_timer[ad_idx] != 0) {
            fc->T3_timer[ad_idx] =
                sctp_restartTimer(fc->T3_timer[ad_idx], pm_readRTO(ad_idx));
            event_logii(INTERNAL_EVENT_0,
                        "Restarted T3 Timer with RTO==%u msecs on address %u",
                        pm_readRTO(ad_idx), ad_idx);
        }
        return;
    }
    event_log(INTERNAL_EVENT_0, "Left T3 Timer running...");
    /* else leave T3 running  */
    return;
}

/**
 * Function called by stream engine to enqueue data chunks in the flowcontrol
 * module. After function returns, we should be able to  delete the pointer
 * to the data (i.e. some lower module must have copied the data...e.g. the
 * Flowcontrol, ReliableTransfer, or Bundling
 * @param  chunk    pointer to the data chunk to be sent
 * @param destAddressIndex index to address to send data structure to...
 * @param  lifetime NULL if unused, else pointer to a value of msecs,
           after which data will not be sent anymore
 * @param   dontBundle NULL if unused, by default bundling is allowed,
            else pointer to boolean indicating whether it is or it is not allowed.
 * @return -1 on error, 0 on success, (1 if problems occurred ?)
 */
int fc_send_data_chunk(SCTP_data_chunk * chunk, short destAddressIndex,
                   unsigned int lifetime, gboolean dontBundle, gpointer context)
{
    fc_data *fc;
    DLL_Return ExitCode;
    chunk_data dat;
    SCTP_data_chunk_se *data_chunk;

    event_log(INTERNAL_EVENT_0, "fc_send_data_chunk is being executed.");

    fc = (fc_data *) mdi_readFlowControl();
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return (-1);
    }
    data_chunk = (SCTP_data_chunk_se *) chunk;

    if (fc->shutdown_received == TRUE) {
        error_log(ERROR_MAJOR,
                  "fc_send_data_chunk() called, but shutdown_received==TRUE - send not allowed !");
        return -1;
    }

    event_log(VVERBOSE, "Printing Chunk List / Congestion Params in FlowControl\n");
    chunk_list_debug(VVERBOSE, fc->chunk_list);

    event_log(VERBOSE, "FlowControl got a Data Chunk to send ");
    DLL_CurrentPointerToTail(fc->chunk_list);

    chunk->data_chunk_header.tsn = htonl(fc->current_tsn++);
    dat.chunk_len = CHUNKP_LENGTH(data_chunk);
    /* FIXME : reject chunk if its size is too big ! Undo state changes here (tsn etc.) */
    if (dat.chunk_len > MAX_SCTP_PDU)
        error_log(ERROR_MAJOR, "Chunk size too big !");
    dat.chunk_tsn = ntohl(data_chunk->tsn);
    dat.gap_reports = 0L;
    dat.ack_time = 0;
    dat.context = context;
    if (destAddressIndex >= 0)
        dat.initial_destination = destAddressIndex;
    else
        dat.initial_destination = -1;

    if (lifetime > 0) {
        adl_gettime(&(dat.lifetime));
        adl_add_msecs_totime(&(dat.lifetime), lifetime);
    } else
        timerclear(&(dat.lifetime));

    dat.dontBundle = dontBundle;

    dat.num_of_transmissions = 0;

    memcpy(&dat.data, data_chunk, dat.chunk_len * sizeof(unsigned char));
    /* insert chunk at the list's tail */
    if ((ExitCode = DLL_AddRecord(fc->chunk_list, &dat, NULL)) != DLL_NORMAL) {
        DLL_error_log(ERROR_FATAL, ExitCode);
        return -1;
    }

    event_log(VVERBOSE, "Printing Chunk List / Congestion Params in FlowControl");
    chunk_list_debug(VVERBOSE, fc->chunk_list);

    fc_check_for_txmit(fc);

    return 0;
}

/**
 * function called by Reliable Transfer, when it requests retransmission
 * in SDL diagram this signal is called (Req_RTX, RetransChunks)
 * @param  all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   num_acked_per_address array of integers, that hold number of bytes acked for each address
 * @param   number_of_rtx_chunks number indicatin, how many chunks are to be retransmitted in on datagram
 * @param   chunks  array of pointers to data_chunk structures. These are to be retransmitted
 * @return   -1 on error, 0 on success, (1 if problems occurred ?)
 */
int fc_do_retransmission(unsigned int address_index, unsigned int arwnd,
                     unsigned int rtx_bytes, boolean all_data_acked,
                     boolean new_data_acked, unsigned int num_acked,
                     unsigned int number_of_addresses,
                     unsigned int *num_acked_per_address,
                     int number_of_rtx_chunks, chunk_data ** chunks)
{
    fc_data *fc;
    DLL_Return ExitCode;
    int count;
    int (*pfun) () = NULL;

    unsigned rtt_time, peer_rwnd;
    struct timeval last_update, now;
    chunk_data temp;
    int diff;

    fc = (fc_data *) mdi_readFlowControl();
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return (-1);
    }


    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */
    pfun = fc_sort_tsn;

    fc->t3_retransmission_sent = FALSE; /* we have received a SACK, so reset this */
    if (num_acked > 0)
        fc->waiting_for_sack = FALSE;

    for (count = 0; count < number_of_rtx_chunks; count++)
        event_logi(VERBOSE, "fc_do_retransmission: Got TSN==%u for RTXmit\n",
                   chunks[count]->chunk_tsn);

    fc->outstanding_bytes =
        (fc->outstanding_bytes <= num_acked) ? 0 : (fc->outstanding_bytes - num_acked);
    for (count = 0; count < number_of_addresses; count++) {
        fc->cparams[count].outstanding_bytes_per_address =
            (fc->cparams[count].outstanding_bytes_per_address <
             num_acked_per_address[count]) ? 0 : (fc->cparams
                                                  [count].outstanding_bytes_per_address
                                                  - num_acked_per_address[count]);
    }

    /* see section 6.2.1, section 6.2.2 */
    if (fc->cparams[address_index].cwnd <= fc->cparams[address_index].ssthresh) { /* SLOW START */
        for (count = 0; count < number_of_addresses; count++) {
            event_logii(VERBOSE, "SLOW START : outstanding bytes on(%d)=%u ",
                        count, fc->cparams[count].outstanding_bytes_per_address);
            event_logii(VERBOSE, "SLOW START : numacked per address(%d)=%u ",
                        count, num_acked_per_address[count]);
            fc->cparams[count].partial_bytes_acked = 0;
        }
        if (new_data_acked == TRUE) {
            fc->cparams[address_index].cwnd += min(MAX_MTU_SIZE, num_acked);
        }
    } else {                    /* CONGESTION AVOIDANCE, as per section 6.2.2 */

        fc->cparams[address_index].partial_bytes_acked += num_acked;
        rtt_time = pm_readSRTT(address_index);
        memcpy(&last_update, &(fc->time_of_adjustment), sizeof(struct timeval));
        adl_add_msecs_totime(&last_update, rtt_time);
        adl_gettime(&now);
        diff = adl_timediff_to_msecs(&now, &last_update); /* a-b */
        event_logii(VVERBOSE, "CONG. AVOIDANCE : rtt_time=%u diff=%d", rtt_time, diff);

        if (diff >= 0) {
            if ((fc->cparams[address_index].partial_bytes_acked >= fc->cparams[address_index].cwnd)
                && (fc->outstanding_bytes >= fc->cparams[address_index].cwnd)) {
                fc->cparams[address_index].cwnd += MAX_MTU_SIZE;
                fc->cparams[address_index].partial_bytes_acked -= fc->cparams[address_index].cwnd;
                /* update time of window adjustment (i.e. now) */
                event_log(VVERBOSE,
                          "CONG. AVOIDANCE : updating time of adjustment !!!!!!!!!! NOW ! ");
                adl_gettime(&(fc->time_of_adjustment));
            }
            event_logii(VERBOSE,
                        "CONG. AVOIDANCE : updated counters: %u bytes outstanding, cwnd=%u",
                        fc->outstanding_bytes, fc->cparams[address_index].cwnd);
        }
        event_logii(VVERBOSE, "CONG. AVOIDANCE : partial_bytes_acked(%u)=%u ",
                    address_index, fc->cparams[address_index].partial_bytes_acked);

        for (count = 0; count < number_of_addresses; count++) {
            event_logii(VVERBOSE,
                        "CONG. AVOIDANCE : outstanding bytes on(%d)=%u ",
                        count, fc->cparams[count].outstanding_bytes_per_address);
            event_logii(VVERBOSE,
                        "CONG. AVOIDANCE : numacked per address(%d)=%u ",
                        count, num_acked_per_address[count]);
        }
        /* see section 7.2.2 */
        if (all_data_acked == TRUE)
            fc->cparams[address_index].partial_bytes_acked = 0;

    }

    /* We HAVE retransmission, so DO UPDATE OF WINDOW PARAMETERS , see section 6.2.3 */
    fc->cparams[address_index].ssthresh =
        max(fc->cparams[address_index].cwnd / 2, 2 * fc->cparams[address_index].mtu);
    fc->cparams[address_index].cwnd = fc->cparams[address_index].ssthresh;
    event_logiiii(VERBOSE,
                  "fc_do_retransmission: updated: %u bytes outstanding, %u bytes per address, cwnd=%u, ssthresh=%u",
                  fc->outstanding_bytes,
                  fc->cparams[address_index].outstanding_bytes_per_address,
                  fc->cparams[address_index].cwnd, fc->cparams[address_index].ssthresh);

    /* This is to be an ordered list containing no duplicate entries ! */
    for (count = number_of_rtx_chunks - 1; count >= 0; count--) {

        if (DLL_FindRecord(fc->chunk_list, &temp, chunks[count], pfun) == DLL_NORMAL) {
            event_logii(VERBOSE,
                        "chunk_tsn==%u, count==%u already in the list -- continue with next\n",
                        chunks[count]->chunk_tsn, count);
            continue;
        }
        event_logii(INTERNAL_EVENT_0,
                    "inserting chunk_tsn==%u, count==%u in the list\n",
                    chunks[count]->chunk_tsn, count);

        if ((ExitCode = DLL_AddRecord(fc->chunk_list, chunks[count], pfun)) != DLL_NORMAL) {
            DLL_error_log(ERROR_FATAL, ExitCode);
            return -1;
        }
    }
    fc->last_active_address = address_index;

    event_log(VVERBOSE, "fc_do_retransmission: FlowControl Chunklist after Re-Insertion \n");
    chunk_list_debug(VVERBOSE, fc->chunk_list);

    fc_check_t3(address_index, all_data_acked, new_data_acked);

    fc->one_packet_inflight = FALSE;

    /* section 6.2.1.D */
    peer_rwnd = arwnd - fc->outstanding_bytes;
    /* section 6.2.1.C */
    peer_rwnd += rtx_bytes;
    rtx_set_remote_receiver_window(peer_rwnd);


    /* request a SACK to be sent along */
    rxc_create_sack(NULL, FALSE);

    /* send as many to bundling as allowed, requesting new destination address */
    if (DLL_IsListEmpty(fc->chunk_list) == DLL_FALSE) {
        fc_check_for_txmit(fc);
    }
    return 1;
}


/**
 * function called by Reliable Transfer, after it has got a SACK chunk
 * in SDL diagram this signal is called SACK_Info
 * @param  all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   num_acked_per_address array of integers, that hold number of bytes acked for each address
 */
void fc_sack_info(unsigned int address_index, unsigned int arwnd,
             boolean all_data_acked, boolean new_data_acked,
             unsigned int num_acked, unsigned int number_of_addresses,
             unsigned int *num_acked_per_address)
{
    fc_data *fc;
    unsigned int count, rto_time, rtt_time, peer_rwnd;
    struct timeval last_update, now;
    int diff;

    fc = (fc_data *) mdi_readFlowControl();
    if (!fc) {
        error_log(ERROR_MAJOR, "fc_data instance not set !");
        return;
    }
    /* ------------------ DEBUGGING ----------------------------- */
    fc_debug_cparams(VERBOSE);
    /* ------------------ DEBUGGING ----------------------------- */

    event_logii(INTERNAL_EVENT_0,
                "fc_sack_info...bytes acked=%u on address %u ", num_acked, address_index);

    fc->t3_retransmission_sent = FALSE; /* we have received a SACK, so reset this */
    if (num_acked > 0)
        fc->waiting_for_sack = FALSE;

    fc->outstanding_bytes =
        (fc->outstanding_bytes <= num_acked) ? 0 : (fc->outstanding_bytes - num_acked);
    for (count = 0; count < number_of_addresses; count++) {
        fc->cparams[count].outstanding_bytes_per_address =
            (fc->cparams[count].outstanding_bytes_per_address <
             num_acked_per_address[count]) ? 0 : (fc->cparams
                                                  [count].outstanding_bytes_per_address
                                                  - num_acked_per_address[count]);
    }
    /* see section 6.2.1, section 6.2.2 */
    if (fc->cparams[address_index].cwnd <= fc->cparams[address_index].ssthresh) { /* SLOW START */
        for (count = 0; count < number_of_addresses; count++) {
            event_logii(VERBOSE, "SLOW START : outstanding bytes on(%d)=%u ",
                        count, fc->cparams[count].outstanding_bytes_per_address);
            event_logii(VERBOSE, "SLOW START : numacked per address(%d)=%u ",
                        count, num_acked_per_address[count]);
            fc->cparams[count].partial_bytes_acked = 0;
        }
        if (new_data_acked == TRUE) {
            fc->cparams[address_index].cwnd += min(MAX_MTU_SIZE, num_acked);
        }
        event_logiii(VERBOSE,
                     "SLOW START : updated counters: TOTAL %u bytes outstanding, cwnd(%u)=%u",
                     fc->outstanding_bytes, address_index, fc->cparams[address_index].cwnd);
    } else {                    /* CONGESTION AVOIDANCE, as per section 6.2.2 */

        fc->cparams[address_index].partial_bytes_acked += num_acked;

        rtt_time = pm_readSRTT(address_index);
        memcpy(&last_update, &(fc->time_of_adjustment), sizeof(struct timeval));
        adl_add_msecs_totime(&last_update, rtt_time);
        adl_gettime(&now);
        diff = adl_timediff_to_msecs(&now, &last_update); /* a-b */

        event_logii(VVERBOSE, "CONG. AVOIDANCE : rtt_time=%u diff=%d", rtt_time, diff);

        if (diff >= 0) {
            if ((fc->cparams[address_index].partial_bytes_acked >= fc->cparams[address_index].cwnd)
                && (fc->outstanding_bytes >= fc->cparams[address_index].cwnd)) {
                fc->cparams[address_index].cwnd += MAX_MTU_SIZE;
                fc->cparams[address_index].partial_bytes_acked -= fc->cparams[address_index].cwnd;
                event_log(VVERBOSE,
                          "CONG. AVOIDANCE : updating time of adjustment !!!!!!!!!! NOW ! ");
                adl_gettime(&(fc->time_of_adjustment));
            }
            event_logii(VERBOSE,
                        "CONG. AVOIDANCE : updated counters: %u bytes outstanding, cwnd=%u",
                        fc->outstanding_bytes, fc->cparams[address_index].cwnd);
            /* update time of window adjustment (i.e. now) */
        }
        event_logii(VVERBOSE, "CONG. AVOIDANCE : partial_bytes_acked(%u)=%u ",
                    address_index, fc->cparams[address_index].partial_bytes_acked);

        for (count = 0; count < number_of_addresses; count++) {
            event_logii(VVERBOSE,
                        "CONG. AVOIDANCE : outstanding bytes on(%d)=%u ",
                        count, fc->cparams[count].outstanding_bytes_per_address);
            event_logii(VVERBOSE,
                        "CONG. AVOIDANCE : numacked per address(%d)=%u ",
                        count, num_acked_per_address[count]);
        }
        /* see section 7.2.2 */
        if (all_data_acked == TRUE)
            fc->cparams[address_index].partial_bytes_acked = 0;

    }
    /* if we don't send data, cwnd = max(cwnd/2, 2*MTU), once per RTO */
    rto_time = pm_readRTO(address_index);
    fc->last_active_address = address_index;


    if ((fc->cwnd_timer == 0) && (DLL_IsListEmpty(fc->chunk_list) == DLL_TRUE)) {
        fc->cwnd_timer =
            sctp_startTimer(rto_time, &fc_timer_cb_reduce_cwnd, &(fc->my_association), NULL);
        event_logi(INTERNAL_EVENT_0,
                   "fc_sack_info...started reduce-cwnd-Timer going off %u msecs from now",
                   rto_time);
    }

    fc_check_t3(address_index, all_data_acked, new_data_acked);

    fc->one_packet_inflight = FALSE;

    /* section 6.2.1.C */
    peer_rwnd = arwnd - fc->outstanding_bytes;
    rtx_set_remote_receiver_window(peer_rwnd);

    if (DLL_IsListEmpty(fc->chunk_list) == DLL_FALSE) {
        fc_check_for_txmit(fc);
    }

    return;
}


/**
 * function returns number of chunks, that are waiting in the transmission queue
 * These have been submitted from the upper layer, but not yet been sent !
 * They are initially only queued in the transmission queue, not the
 * retransmission queue
 * @return size of the send queue of the current flowcontrol module
 */
unsigned int fc_readNumberOfQueuedChunks()
{
    unsigned int queue_len;
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return 0;
    }
    queue_len = DLL_GetNumberOfRecords(fc->chunk_list);
    event_logi(VERBOSE, "fc_readNumberOfQueuedChunks() returns %u", queue_len);
    return queue_len;
}




/**
 * Function returns cwnd value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd value, else -1
 */
int fc_readCWND(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }

    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].cwnd;
}


/**
 * Function returns cwnd2 value of a certain path.
 * @param path_id    path index of which we want to know the cwnd2
 * @return current cwnd2 value, else -1
 */
int fc_readCWND2(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }

    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].cwnd2;
}

/**
 * Function returns ssthresh value of a certain path.
 * @param path_id    path index of which we want to know the ssthresh
 * @return current ssthresh value, else -1
 */
int fc_readSsthresh(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }
    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].ssthresh;
}

/**
 * Function returns mtu value of a certain path.
 * @param path_id    path index of which we want to know the mtu
 * @return current MTU value, else -1
 */
int fc_readMTU(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }
    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].mtu;
}

/**
 * Function returns the partial bytes acked value of a certain path.
 * @param path_id    path index of which we want to know the PBA
 * @return current PBA value, else -1
 */
int fc_readPBA(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }
    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].partial_bytes_acked;
}

/**
 * Function returns the outstanding byte count value of a certain path.
 * @param path_id    path index of which we want to know the outstanding_bytes_per_address
 * @return current outstanding_bytes_per_address value, else -1
 */
int fc_readOutstandingBytesPerAddress(short path_id)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }
    if (path_id >= fc->number_of_addresses || path_id < 0) {
        error_logi(ERROR_MAJOR, "Association has only %u addresses !!! ", fc->number_of_addresses);
        return -1;
    }
    return (int)fc->cparams[path_id].outstanding_bytes_per_address;
}

/**
 * Function returns the outstanding byte count value of this association.
 * @return current outstanding_bytes value, else -1
 */
int fc_readOutstandingBytes(void)
{
    fc_data *fc;
    fc = (fc_data *) mdi_readFlowControl();

    if (!fc) {
        error_log(ERROR_MAJOR, "flow control instance not set !");
        return -1;
    }
    return (int)fc->outstanding_bytes;
}
