/*
 *  $Id: rbundling.c,v 1.4 2001/03/16 13:44:26 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This module treats the received chunks, gets the SCTP data from SCTP message-distribution,
 *          which are disassambled (debundled) into chunks. Depending on the chunk-type, the chunks
 *          are distributed to SCTP-control, RX-control, pathmanagement or reliable transfer.
 */

#include <stdio.h>
#include <stdlib.h>
#include "bundling.h"
#include "messages.h"
#include "recvctrl.h"
#include "reltransfer.h"
#include "errorhandler.h"

#define TOTAL_SIZE(buf)		((buf)->ctrl_position+(buf)->sack_position+(buf)->data_position- 2*sizeof(SCTP_common_header))



/**
 * looks for chunk_type in a newly received datagram
 *
 * All chunks within the datagram are lookes at, until one is found
 * that equals the parameter chunk_type.
 * @param  datagram     pointer to the newly received data
 * @param  chunk_type   chunk type to look for
 * @return true is chunk_type exists in SCTP datagram, false if it is not in there
 */
gboolean rbu_scanDatagram(guchar * datagram, guint len, gushort chunk_type)
{
    gushort processed_len = 0;
    guchar *current_position;
    guint pad_bytes;
    SCTP_simple_chunk *chunk;

    current_position = datagram; /* points to the first chunk in this pdu */
    while (processed_len < len) {

        event_logii(INTERNAL_EVENT_0,
                    "rbu_scanDatagram : len==%u, processed_len == %u", len, processed_len);

        chunk = (SCTP_simple_chunk *) current_position;
        switch (chunk->chunk_header.chunk_id == chunk_type) {
        case TRUE:
            return TRUE;
            break;
        default:
            processed_len += CHUNKP_LENGTH((SCTP_chunk_header *) chunk);
            pad_bytes = ((processed_len % 4) == 0) ? 0 : (4 - processed_len % 4);
            processed_len += pad_bytes;
            current_position +=
                (CHUNKP_LENGTH((SCTP_chunk_header *) chunk) + pad_bytes * sizeof(unsigned char));
            break;
        }
    }
    return FALSE;
}

/*
 * rbu_findChunk: looks for chunk_type in a newly received datagram
 *
 * All chunks within the datagram are looked at, until one is found
 * that equals the parameter chunk_type.
 * @param  datagram     pointer to the newly received data
 * @param  len          stop after this many bytes
 * @param  chunk_type   chunk type to look for
 * @return pointer to first chunk of chunk_type in SCTP datagram, else NULL
 */
guchar* rbu_findChunk(guchar * datagram, guint len, gushort chunk_type)
{
    gushort processed_len = 0;
    guchar *current_position;
    guint pad_bytes;
    SCTP_simple_chunk *chunk;

    current_position = datagram; /* points to the first chunk in this pdu */
    while (processed_len < len) {

        event_logii(INTERNAL_EVENT_0,
                    "rbu_findChunk : len==%u, processed_len == %u", len, processed_len);

        chunk = (SCTP_simple_chunk *) current_position;
        switch (chunk->chunk_header.chunk_id == chunk_type) {
        case TRUE:
            return current_position;
            break;
        default:
            processed_len += CHUNKP_LENGTH((SCTP_chunk_header *) chunk);
            pad_bytes = ((processed_len % 4) == 0) ? 0 : (4 - processed_len % 4);
            processed_len += pad_bytes;
            current_position +=
                (CHUNKP_LENGTH((SCTP_chunk_header *) chunk) + pad_bytes * sizeof(unsigned char));
            break;
        }
    }
    return NULL;
}


/**
 * looks for Error chunk_type in a newly received datagram
 * that contains a special error cause code
 *
 * All chunks within the datagram are lookes at, until one is found
 * that equals the parameter chunk_type.
 * @param  datagram     pointer to the newly received data
 * @param  len          stop after this many bytes
 * @param  error_cause  error cause code to look for
 * @return true is chunk_type exists in SCTP datagram, false if it is not in there
 */
gboolean rbu_scanDatagramForError(guchar * datagram, guint len, gushort error_cause)
{
    gushort processed_len = 0;
    gushort err_len = 0;

    guchar *current_position;
    guint pad_bytes;
    SCTP_simple_chunk *chunk;
    SCTP_staleCookieError *err_chunk;


    current_position = datagram; /* points to the first chunk in this pdu */
    while (processed_len < len) {

        event_logii(VERBOSE,
                    "rbu_scanDatagramForError : len==%u, processed_len == %u", len, processed_len);

        chunk = (SCTP_simple_chunk *) current_position;
        switch (chunk->chunk_header.chunk_id == CHUNK_ERROR) {
        case TRUE:
            event_log(INTERNAL_EVENT_0, "rbu_scanDatagramForError : Error Chunk Found");
            while (err_len < chunk->chunk_header.chunk_length) {
                err_chunk = (SCTP_staleCookieError *) & current_position[err_len];
                if (ntohs(err_chunk->vlparam_header.param_type) == error_cause) {
                    event_logi(VERBOSE,
                               "rbu_scanDatagramForError : Error Cause %u found -> Returning TRUE",
                               error_cause);
                    return TRUE;
                }
                err_len += ntohs(err_chunk->vlparam_header.param_length);
                while ((err_len % 4) != 0)
                    err_len++;
            }
        default:
            processed_len += CHUNKP_LENGTH((SCTP_chunk_header *) chunk);
            pad_bytes = ((processed_len % 4) == 0) ? 0 : (4 - processed_len % 4);
            processed_len += pad_bytes;
            current_position +=
                (CHUNKP_LENGTH((SCTP_chunk_header *) chunk) + pad_bytes * sizeof(unsigned char));
            break;
        }
    }
    event_logi(VERBOSE,
               "rbu_scanDatagramForError : Error Cause %u NOT found -> Returning FALSE",
               error_cause);
    return FALSE;
}


/**
 * Disassembles chunks from a received datagram
 *
 * All chunks within the datagram are dispatched and sent to the appropriate
 * module, i.e.: control chunks are sent to sctp_control/pathmanagement,
 * SACK chunks to reliable_transfer, and data_chunks to RX_control.
 * Those modules must get a pointer to the start of a chunk and
 * information about its size (without padding).
 * @param  address_index  index of address on which this data arrived
 * @param  datagram     pointer to first chunk of the newly received data
 * @param  len          length of payload (i.e. len of the concatenation of chunks)
 */
gint rbu_rcvDatagram(guint address_index, guchar * datagram, guint len)
{
    /* sctp common header header has been verified */
    /* tag (if association is established) and CRC is okay */
    /* get first chunk-id and length, pass pointers & len on to relevant module :
       - CHUNK_INIT, CHUNK_INIT_ACK,CHUNK_ABORT, CHUNK_SHUTDOWN,CHUNK_SHUTDOWN_ACK
       CHUNK_COOKIE_ECHO,CHUNK_COOKIE_ACK go to SCTP_CONTROL (change of association state)
       - CHUNK_HBREQ, CHUNK_HBACK go to PATH_MAN instance
       - CHUNK_SACK goes to RELIABLE_TRANSFER
       - CHUNK_ERROR probably to SCTP_CONTROL as well  (at least there !)
       - CHUNK_DATA goes to RX_CONTROL
     */
    guchar *current_position;
    gushort processed_len = 0, chunk_len;
    gushort pad_bytes;
    SCTP_simple_chunk *chunk;
    gboolean data_chunk_received = FALSE;
    gboolean removed_association = FALSE;
    gboolean send_it;

    bu_lock_sender();

    current_position = datagram; /* points to the first chunk in this pdu */
    event_log(INTERNAL_EVENT_0, "Entered rbu_rcvDatagram()...... ");
    /* CHECKME : beim Empfangen leerer Chunks tritt im Bundling eine Endlosschleife auf ??? */
    while (processed_len < len) {

        chunk = (SCTP_simple_chunk *) current_position;
        chunk_len = CHUNKP_LENGTH((SCTP_chunk_header *) chunk);
        event_logiiii(INTERNAL_EVENT_0,
                     "rbu_rcvDatagram(address=%u) : len==%u, processed_len = %u, chunk_len=%u",
                     address_index, len, processed_len, chunk_len);
        if (chunk_len > len) {
            error_logii(ERROR_MINOR, "Faulty chunklen=%u, total len=%u --> dropping rest of data !",
                                    chunk_len,len);
            return 1;
        }
        /*
         * TODO :   Add return values to the chunk-functions, where they can indicate what
         *          to do with the rest of the datagram (i.e. DISCARD after stale COOKIE_ECHO
         *          with tie tags that do not match the current ones)
         */
        switch (chunk->chunk_header.chunk_id) {
        case CHUNK_DATA:
            event_log(INTERNAL_EVENT_0, "Bundling received DATA chunk");
            rxc_data_chunk_rx((SCTP_data_chunk *) chunk, address_index);
            data_chunk_received = TRUE;
            break;
        case CHUNK_INIT:
            event_log(INTERNAL_EVENT_0, "Bundling received INIT chunk");
            removed_association = scr_init((SCTP_init *) chunk);
            break;
        case CHUNK_INIT_ACK:
            event_log(INTERNAL_EVENT_0, "Bundling received INIT ACK chunk");
            removed_association = scr_initAck((SCTP_init *) chunk);
            break;
        case CHUNK_SACK:
            event_log(INTERNAL_EVENT_0, "Bundling received SACK chunk");
            rtx_process_sack(address_index, chunk);
            break;
        case CHUNK_HBREQ:
            event_log(INTERNAL_EVENT_0, "Bundling received HB_REQ chunk");
            pm_heartbeat((SCTP_heartbeat *) chunk, address_index);
            break;
        case CHUNK_HBACK:
            event_log(INTERNAL_EVENT_0, "Bundling received HB_ACK chunk");
            pm_heartbeatAck((SCTP_heartbeat *) chunk);
            break;
        case CHUNK_ABORT:
            event_log(INTERNAL_EVENT_0, "Bundling received ABORT chunk");
            removed_association = scr_abort();
            break;
        case CHUNK_SHUTDOWN:
            event_log(INTERNAL_EVENT_0, "Bundling received SHUTDOWN chunk");
            removed_association = scr_shutdown((SCTP_simple_chunk *) chunk);
            break;
        case CHUNK_SHUTDOWN_ACK:
            event_log(INTERNAL_EVENT_0, "Bundling received SHUTDOWN ACK chunk");
            removed_association = scr_shutdownAck();
            break;
        case CHUNK_ERROR:
            event_log(INTERNAL_EVENT_0, "Error Chunk");
            eh_recv_chunk(chunk);
            break;
        case CHUNK_COOKIE_ECHO:
            event_log(INTERNAL_EVENT_0, "Bundling received COOKIE ECHO chunk");
            scr_cookie_echo((SCTP_cookie_echo *) chunk);
            break;
        case CHUNK_COOKIE_ACK:
            event_log(INTERNAL_EVENT_0, "Bundling received COOKIE ACK chunk");
            scr_cookieAck((SCTP_simple_chunk *) chunk);
            break;
        case CHUNK_ECNE:
        case CHUNK_CWR:
            event_logi(INTERNAL_EVENT_0,
                       "Chunktype %u not Supported Yet !!!!!!!!", chunk->chunk_header.chunk_id);
            break;
        case CHUNK_SHUTDOWN_COMPLETE:
            event_log(INTERNAL_EVENT_0, "Bundling received SHUTDOWN_COMPLETE chunk");
            removed_association = scr_shutdownComplete();
            break;
        default:
            event_log(EXTERNAL_EVENT_X, "Received unknown chunk type in bundling.c");
            break;
        }

        processed_len += chunk_len;
        pad_bytes = ((processed_len % 4) == 0) ? 0 : (4 - processed_len % 4);
        processed_len += pad_bytes;
        current_position += (chunk_len + pad_bytes) * sizeof(unsigned char);
				
		if (removed_association == TRUE) processed_len = len;
				
        event_logiiii(VVERBOSE, "processed_len=%u, pad_bytes=%u, current_position=%u, chunk_len=%u",
            processed_len, pad_bytes, current_position,chunk_len);
    }
    if (data_chunk_received == TRUE) {
        /* update SACK structure */
        rxc_all_chunks_processed();
    }

    if (removed_association == FALSE) {
        if (data_chunk_received == TRUE)
            send_it = rxc_create_sack(&address_index, FALSE);
        bu_unlock_sender(&address_index);
    }

    return 0;

}


