/*
 *  $Id: recvctrl.c,v 1.8 2001/03/08 16:25:29 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module creates SACK data structures, that may be used to
 * indicate received and lost data to the association peer.
 *
 */

#include "recvctrl.h"
#include "adaptation.h"
#include "bundling.h"
#include "distribution.h"
#include "streamengine.h"
#include "dll_main.h"
#include <stdlib.h>
#include <string.h>

/**
 * this struct contains all necessary data for creating SACKs from received data chunks
 */
typedef struct rxc_buffer_struct
{
    //@{
    ///
    void *sack_chunk;
    ///
    List *frag_list;
    ///
    List *dup_list;
    /** cumulative TSN acked */
    unsigned int ctsna;
    /** store lowest tsn value for dups (!) */
    unsigned int lowest;
    /** stores highest tsn received so far, taking care of wraps
        i.e. highest < lowest indicates a wrap */
    unsigned int highest;
    ///
    boolean contains_valid_sack;
    ///
    boolean timer_running;
    /** indicates whether a chunk was recvd that is truly new */
    boolean new_chunk_received;
    /// timer for delayed sacks
    TimerID sack_timer;
    int datagrams_received;
     /* either 1 (= sack each data chunk) or 2 (=sack every second chunk)*/
    unsigned int sack_flag;
    ///
    unsigned int last_address;
    ///
    unsigned int my_association;
    ///
    unsigned int my_rwnd;
    /// delay for delayed ACK in msecs
    unsigned int delay;
    //@}
} rxc_buffer;


/**
 * function creates and allocs new rxc_buffer structure.
 * There is one such structure per established association
 * @param  remote_initial_TSN initial tsn of the peer
 * @return pointer to the newly created structure
 */
void *rxc_new_recvctrl(unsigned int remote_initial_TSN, void* sctpInstance)
{
    rxc_buffer *tmp;
    DLL_Return ExitCode;
    tmp = malloc(sizeof(rxc_buffer));
    if (!tmp)
        error_log(ERROR_FATAL, "Malloc failed");
    tmp->sack_chunk = malloc(sizeof(SCTP_sack_chunk));
    tmp->ctsna = remote_initial_TSN - 1; /* as per section 4.1 */
    tmp->lowest = remote_initial_TSN - 1;
    tmp->highest = remote_initial_TSN - 1;
    /* initialize and set up lists */
    if (DLL_CreateList(&(tmp->frag_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if (DLL_CreateList(&(tmp->dup_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if ((ExitCode = DLL_InitializeList(tmp->frag_list, sizeof(fragment32))) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    if ((ExitCode = DLL_InitializeList(tmp->dup_list, sizeof(duplicate))) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);

    tmp->contains_valid_sack = FALSE;
    tmp->timer_running = FALSE;
    tmp->datagrams_received = -1;
    tmp->sack_flag = 2;
    tmp->last_address = 0;
    tmp->my_rwnd = mdi_getDefaultMyRwnd();
    tmp->delay = mdi_getDefaultDelay(sctpInstance);
    tmp->my_association = mdi_readAssociationID();
    event_logi(VVERBOSE, "RecvControl : Association-ID== %d ", tmp->my_association);
    if (tmp->my_association == 0)
        error_log(ERROR_FATAL, "Association was not set, should be......");
    return (tmp);
}

/**
 * function deletes a rxc_buffer structure (when it is not needed anymore)
 * @param rxc_instance pointer to a rxc_buffer, that was previously created
 */
void rxc_delete_recvctrl(void *rxc_instance)
{
    rxc_buffer *tmp;
    tmp = (rxc_buffer *) rxc_instance;
    event_log(INTERNAL_EVENT_0, "deleting receivecontrol");
    free(tmp->sack_chunk);
    DLL_DestroyList(&(tmp->frag_list));
    DLL_DestroyList(&(tmp->dup_list));
    free(tmp);
}


/**
 * function to find out, whether a chunk is duplicate or not
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 * @return the boolean response
 */
boolean rxc_chunk_is_duplicate(rxc_buffer * rbuf, unsigned int chunk_tsn)
{
    DLL_Return ExitCode;
    unsigned int low = rbuf->lowest;
    unsigned int hi = rbuf->highest;
    unsigned int ctsna = rbuf->ctsna;
    fragment32 frag;
    unsigned int num_of_frags;

    /* Assume, lowest and highest have already been updated */
    if (between(low, chunk_tsn, ctsna))
        return TRUE;
    if (!between(ctsna, chunk_tsn, hi))
        return FALSE;

    /* Now check, whether chunk_tsn is in the (sorted !) list of fragments */
    num_of_frags = DLL_GetNumberOfRecords(rbuf->frag_list);
    if (num_of_frags == 0)
        return FALSE;           /* no fragments ! */
    /* if we are still here, we need to check, whether chunk_tsn is between any fragment bounds */
    if ((ExitCode = DLL_CurrentPointerToHead(rbuf->frag_list)) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    do {
        if ((ExitCode = DLL_GetCurrentRecord(rbuf->frag_list, &frag)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);

        if (between(frag.start_tsn, chunk_tsn, frag.stop_tsn))
            return TRUE;        /* and leave list pointer here */
        /* assuming an ordered list of fragments */
        if (frag.stop_tsn > chunk_tsn)
            return FALSE;       /* and leave list pointer there... */
    }
    while ((ExitCode = DLL_IncrementCurrentPointer(rbuf->frag_list)) != DLL_NOT_FOUND);

    /* never reached */
    error_log(ERROR_MAJOR, "while loop went past end of list....should not have happened !");
    return FALSE;
}

/**
 * Helper function to do the correct update of rxc->lowest
 * Function is only called, if that is necessary !
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 * @return boolean indicating whether lowest was updated or not
 */
boolean rxc_update_lowest(rxc_buffer * rbuf, unsigned int chunk_tsn)
{
    unsigned int low = rbuf->lowest;
    if (before(chunk_tsn, low)) {
        rbuf->lowest = chunk_tsn;
        /* and it must be a duplicate ! */
        return TRUE;
    } else
        return FALSE /* no update of lowest */ ;
}

/**
 * Helper function to do the correct update of rxc->highest
 * Function is only called, if that is necessary !
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 * @return boolean indicating whether highest was updated or not
 */
boolean rxc_update_highest(rxc_buffer * rbuf, unsigned int chunk_tsn)
{
    unsigned int hi = rbuf->highest;
    if (after(chunk_tsn, hi)) {
        rbuf->highest = chunk_tsn;
        return TRUE;
    } else
        return FALSE /* no update of highest */ ;
}

int rxc_sort_duplicates(duplicate * one, duplicate * two)
{
    if (before(one->duplicate_tsn, two->duplicate_tsn)) {
        return -1;
    } else if (after(one->duplicate_tsn, two->duplicate_tsn)) {
        return 1;
    } else                      /* one==two */
        return 0;
}

/**
 * Helper function for inserting chunk_tsn in the list of duplicates
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 */
void rxc_update_duplicates(rxc_buffer * rbuf, unsigned int ch_tsn)
{
    DLL_Return ExitCode;
    duplicate match, tmp;
    int (*pfun) () = NULL;

    match.duplicate_tsn = ch_tsn;
    pfun = rxc_sort_duplicates;

    ExitCode = DLL_FindRecord(rbuf->dup_list, &tmp, &match, pfun);
    if (ExitCode == DLL_NORMAL)
        return;
    if (ExitCode == DLL_NULL_LIST || ExitCode == DLL_NOT_FOUND) {
        if ((ExitCode = DLL_AddRecord(rbuf->dup_list, &match, pfun)) != DLL_NORMAL)
            DLL_error_log(ERROR_FATAL, ExitCode);
        return;
    }

}

/**
 * Helper function to do the correct update of rxc->ctsna
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 */
void rxc_update_ctsna(rxc_buffer * rbuf)
{
    unsigned int num_of_frags;
    fragment32 frag;
    DLL_Return ExitCode;
    event_log(INTERNAL_EVENT_0, "Entering rxc_update_ctsna... ");

    num_of_frags = DLL_GetNumberOfRecords(rbuf->frag_list);
    if (num_of_frags == 0)
        return;

    if ((ExitCode = DLL_CurrentPointerToHead(rbuf->frag_list)) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    do {
        if ((ExitCode = DLL_GetCurrentRecord(rbuf->frag_list, &frag)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);
        if (rbuf->ctsna + 1 == frag.start_tsn)
            rbuf->ctsna = frag.stop_tsn;
        else
            return;
    }
    while ((ExitCode = DLL_IncrementCurrentPointer(rbuf->frag_list)) != DLL_NOT_FOUND);

}

/**
 * Helper function to do the correct update fragments
 * @param rbuf	instance of rxc_buffer
 * @param chunk_tsn	tsn we just received
 * @return boolean indicating we need update of ctsna (==TRUE) or (definitely) NOT (==FALSE)
 */
boolean rxc_update_fragments(rxc_buffer * rbuf, unsigned int ch_tsn)
{
    DLL_Return ExitCode;
    unsigned int lo, hi, gapsize;
    fragment32 frag, lo_frag;

    event_logi(INTERNAL_EVENT_0, "Entering rxc_update_fragments.tsn==%u.. ", ch_tsn);

    lo = rbuf->ctsna + 1;
    if ((ExitCode = DLL_CurrentPointerToHead(rbuf->frag_list)) == DLL_NORMAL) {
        do {
            if ((ExitCode = DLL_GetCurrentRecord(rbuf->frag_list, &frag)) != DLL_NORMAL)
                DLL_error_log(ERROR_MAJOR, ExitCode);
            hi = frag.start_tsn - 1;

            event_logiii(VVERBOSE, "while-loop: lo=%u, tsn=%u, hi=%u, \n", lo, ch_tsn, hi);

            if (between(lo, ch_tsn, hi)) {
                gapsize = hi - lo + 1;
                if (gapsize > 1) {
                    event_logi(INTERNAL_EVENT_0, "Value of Gapsize (should be > 1 :) ", gapsize);
                    if (ch_tsn == hi) {
                        event_log(VVERBOSE, "ch_tsn==hi....");
                        frag.start_tsn = ch_tsn;
                        ExitCode = DLL_UpdateCurrentRecord(rbuf->frag_list, &frag);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        rbuf->new_chunk_received = TRUE;
                        return TRUE;
                    } else if (ch_tsn == lo) {
                        event_logii(VVERBOSE,
                                    "ch_tsn==lo==%u....rbuf->ctsna==%u....", lo, rbuf->ctsna);
                        if (ch_tsn == (rbuf->ctsna + 1)) {
                            rbuf->ctsna++;
                            rbuf->new_chunk_received = TRUE;
                            return TRUE;
                        }
                        if ((ExitCode = DLL_DecrementCurrentPointer(rbuf->frag_list))
                            != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        if ((ExitCode = DLL_GetCurrentRecord(rbuf->frag_list, &frag)) != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        frag.stop_tsn = ch_tsn;
                        ExitCode = DLL_UpdateCurrentRecord(rbuf->frag_list, &frag);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        rbuf->new_chunk_received = TRUE;
                        return TRUE;
                    } else {    /* a fragment in between */
                        frag.start_tsn = frag.stop_tsn = ch_tsn;
                        event_log(VVERBOSE, "Inserting new fragment....");
                        if (
                            (ExitCode =
                             DLL_InsertRecord(rbuf->frag_list, &frag,
                                              DLL_ABOVE)) != DLL_NORMAL)
                     DLL_error_log(ERROR_MAJOR, ExitCode);
                        rbuf->new_chunk_received = TRUE;
                        return FALSE;
                    }
                } else {        /*gapsize == 1 */
                    event_logi(INTERNAL_EVENT_0, "Value of Gapsize (should be 1 :) %u", gapsize);
                    /* delete fragment, return TRUE */
                    if (lo == rbuf->ctsna + 1) {
                        rbuf->ctsna = frag.stop_tsn;
                        ExitCode = DLL_DeleteCurrentRecord(rbuf->frag_list);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        rbuf->new_chunk_received = TRUE;
                        return TRUE;
                    } else {
                        ExitCode = DLL_StoreCurrentPointer(rbuf->frag_list);
                        if ((ExitCode = DLL_DecrementCurrentPointer(rbuf->frag_list))
                            != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        if (
                            (ExitCode =
                             DLL_GetCurrentRecord(rbuf->frag_list,
                                                  &lo_frag)) !=
                            DLL_NORMAL) DLL_error_log(ERROR_MAJOR, ExitCode);
                        frag.start_tsn = lo_frag.start_tsn;
                        ExitCode = DLL_DeleteCurrentRecord(rbuf->frag_list);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        ExitCode = DLL_RestoreCurrentPointer(rbuf->frag_list);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        ExitCode = DLL_UpdateCurrentRecord(rbuf->frag_list, &frag);
                        if (ExitCode != DLL_NORMAL)
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        rbuf->new_chunk_received = TRUE;
                        return TRUE;
                    }
                }
            } else {            /* ch_tsn is not in the gap between these two fragments */
                lo = frag.stop_tsn + 1;
                event_logi(VVERBOSE, "rxc_update_fragments: Setting lo to %u ", lo);
            }
        }
        while ((ExitCode = DLL_IncrementCurrentPointer(rbuf->frag_list)) != DLL_NOT_FOUND);
    }
    /* (ExitCode == DLL_NULL_LIST)  OR  (End of Fragment List was passed) */
    if (ch_tsn == lo) {
        /* just increase ctsna, handle rest in separate update_ctsna() */
        if (ch_tsn == rbuf->ctsna + 1) {
            event_logi(VVERBOSE, "Updating rbuf->ctsna==%u", ch_tsn);
            rbuf->ctsna = ch_tsn;
            rbuf->new_chunk_received = TRUE;
            return TRUE;
        }
        /* Update last fragment....increase stop_tsn by one */
        ExitCode = DLL_CurrentPointerToTail(rbuf->frag_list);
        if (ExitCode == DLL_NULL_LIST)
            DLL_error_log(ERROR_MAJOR, ExitCode);

        if ((ExitCode = DLL_GetCurrentRecord(rbuf->frag_list, &frag)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);
        frag.stop_tsn = frag.stop_tsn++;
        event_logiii(VVERBOSE,
                     "Updating last fragment frag.start==%u, frag.stop==%u, tsn=%u",
                     frag.start_tsn, frag.stop_tsn, ch_tsn);
        ExitCode = DLL_UpdateCurrentRecord(rbuf->frag_list, &frag);
        rbuf->new_chunk_received = TRUE;
        return FALSE;
    } else {                    /* a new fragment altogether */
        ExitCode = DLL_CurrentPointerToTail(rbuf->frag_list);
        frag.start_tsn = ch_tsn;
        frag.stop_tsn = ch_tsn;
        event_logiii(VVERBOSE,
                     "Inserting new  fragment at end...frag.start==%u,frag.stop==%u, tsn=%u",
                     frag.start_tsn, frag.stop_tsn, ch_tsn);
        if ((ExitCode = DLL_InsertRecord(rbuf->frag_list, &frag, DLL_BELOW)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);

        rbuf->new_chunk_received = TRUE;
        return FALSE;           /* no ctsna update necessary whatsoever */
    }

}


/**
 * For now this function treats only one incoming data chunk' tsn
 * @param chunk the data chunk that was received by the bundling
 */
int rxc_data_chunk_rx(SCTP_data_chunk * chunk, unsigned int ad_idx)
{
    SCTP_data_chunk_se *se_chk;
    rxc_buffer *rxc;
    unsigned int chunk_tsn;
    unsigned int chunk_len;
    boolean result = FALSE;

    event_log(INTERNAL_EVENT_0, "Entering function rxc_data_chunk_rx");
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return (-1);
    }

    /* resetting it */
    rxc->new_chunk_received = FALSE;

    rxc->last_address = ad_idx;

    /*
     * if any received data chunks have not been acked, sender
     * should create a SACK and bundle it with the outbound data
     */
    rxc->contains_valid_sack = FALSE;

    se_chk = (SCTP_data_chunk_se *) chunk;

    chunk_tsn = ntohl(se_chk->tsn);
    chunk_len = ntohs(se_chk->chunk_length);

    /* TODO :  Duplicates : see Note in section 6.2 :  */
    /*  Note: When a datagram arrives with duplicate DATA chunk(s) and no new
       DATA chunk(s), the receiver MUST immediately send a SACK with no
       delay. Normally this will occur when the original SACK was lost, and
       the peers RTO has expired. The duplicate TSN number(s) SHOULD be
       reported in the SACK as duplicate.
     */
    event_logii(VERBOSE, "rxc_data_chunk_rx : chunk_tsn==%u, chunk_len=%u", chunk_tsn, chunk_len);
    if (rxc_update_lowest(rxc, chunk_tsn) == TRUE) {
        /* tsn is even lower than the lowest one received so far */
        rxc_update_duplicates(rxc, chunk_tsn);
    } else if (rxc_update_highest(rxc, chunk_tsn) == TRUE) {
        rxc->new_chunk_received = TRUE;
        result = rxc_update_fragments(rxc, chunk_tsn);
    } else if (rxc_chunk_is_duplicate(rxc, chunk_tsn) == TRUE)
        rxc_update_duplicates(rxc, chunk_tsn);
    else
        result = rxc_update_fragments(rxc, chunk_tsn);

    if (result == TRUE)
        rxc_update_ctsna(rxc);

    event_logi(VVERBOSE, "rxc_data_chunk_rx: after rxc_update_ctsna, rxc->ctsna=%u", rxc->ctsna);

    if (rxc->new_chunk_received == TRUE)
        se_recvDataChunk(se_chk, chunk_len);
    /* resetting it */
    rxc->new_chunk_received = FALSE;
    return 1;
}

/**
 * Function triggered by flowcontrol, tells recvcontrol to
 * send SACK to bundling using bu_put_SACK_Chunk() function.
 * @return boolean to indicate, whether a SACK was generated, and should be sent !
 */
boolean rxc_create_sack(unsigned int *destination_address, boolean send_at_once)
{
    rxc_buffer *rxc;
    int result;
    unsigned int num_of_frags;

    event_logii(VVERBOSE,
                "Entering rxc_create_sack(address==%u, send_at_once==%s",
                ((destination_address != NULL) ? *destination_address : 0),
                ((send_at_once == TRUE) ? "TRUE" : "FALSE"));

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return FALSE;
    }

    if (rxc->contains_valid_sack == FALSE) {
        error_log(ERROR_MINOR, "SACK structure was not updated (should have been)");
        rxc_all_chunks_processed();
    }


    num_of_frags = DLL_GetNumberOfRecords(rxc->frag_list);

    if (num_of_frags > 0)
        rxc_send_sack_everytime();
    else
        rxc_send_sack_every_second_time();


    /* send sacks along every second time, generally */
    /* first sack is sent at once, since datagrams_received==-1 */
    if (send_at_once == FALSE) {
        event_logii(VVERBOSE, "rxc->datagrams_received==%d, rxc->sack_flag=%d",
                    rxc->datagrams_received, rxc->sack_flag);
        if (rxc->datagrams_received % rxc->sack_flag != 0) {
            event_log(VVERBOSE, "Did not send SACK here - returning");
            return FALSE;
        }
    }

    /* in case we have not yet got any data, we will not want to send a SACK */
    if (rxc->datagrams_received == -1)
        return FALSE;


    rxc->lowest = rxc->ctsna;
    DLL_DeleteEntireList(rxc->dup_list);
    result = bu_put_SACK_Chunk(rxc->sack_chunk);
    if (send_at_once == TRUE) {
        /* really unnecessary, as this is only TRUE for timer based function calls */
        /* TODO : clean this up..... :-)        */
        rxc_stop_sack_timer();
        result = bu_sendAllChunks(destination_address);
        return FALSE;
    }
    if (rxc->datagrams_received == 0 && send_at_once == FALSE)
        return TRUE;            /* first send, fixme: wrap */
    event_logii(VVERBOSE, "rxc->datagrams_received==%d, rxc->sack_flag=%d",
                rxc->datagrams_received, rxc->sack_flag);
    if (rxc->datagrams_received % rxc->sack_flag == 0)
        return TRUE;
    return (send_at_once == FALSE) ? FALSE : TRUE;
}


/**
 * the callback function when the sack timer goes off, and we must sack previously
 * received data (e.g. after 200 msecs)
 * Has three parameters as all timer callbacks
 * @param   tid id of the timer that has gone off
 * @param   assoc  pointer to the association this event belongs to
 * @param   dummy  pointer that is not used here
 */
void rxc_sack_timer_cb(TimerID tid, void *assoc, void *dummy)
{
    unsigned short res;
    rxc_buffer *rxc;
    event_log(INTERNAL_EVENT_1, "Timer Callback Function activated -> initiate sending of a SACK");
    res = mdi_setAssociationData(*(unsigned int *) assoc);
    if (res == 1)
        error_log(ERROR_MAJOR, " association does not exist !");
    if (res == 2) {
        error_log(ERROR_MAJOR, "Association was not cleared..... !!!");
        /* failure treatment ? */
    }
    /* all should be well now */
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return;
    }
    rxc->timer_running = FALSE;
    /* sending sack */
    /* FIXME : maybe choose different address ??? */
    rxc_create_sack(NULL, TRUE);
    mdi_clearAssociationData(*(unsigned int *) assoc);
    return;
}

/**
 * function called by bundling when a SACK is actually sent, to stop
 * a possibly running  timer
 */
void rxc_stop_sack_timer(void)
{
    rxc_buffer *rxc;
    int result;

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MINOR, "rxc_buffer instance not set !");
        return;
    }
    if (rxc->timer_running == TRUE) {
        result = sctp_stopTimer(rxc->sack_timer);
        event_logi(INTERNAL_EVENT_0, "Stopped Timer, Result was %d", result);
        rxc->timer_running = FALSE;
    }
    return;
}


/**
  called by bundling, after new data has been processed (so we may start building a sack chunk)
 */
void rxc_all_chunks_processed(void)
{

    /* now go and create SACK structure from the array */
    rxc_buffer *rxc;
    SCTP_sack_chunk *sack;
    unsigned short num_of_frags, num_of_dups;
    unsigned short len16, count, frag_start16, frag_stop16;
    unsigned int pos, len32;
    DLL_Return ExitCode;
    duplicate d;
    fragment32 f32;
    fragment chunk_frag;

    event_log(INTERNAL_EVENT_0, "Entering funtion rxc_all_chunks_processed... ");

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return;
    }
    rxc->datagrams_received++;

    if (DLL_GetNumberOfRecords(rxc->frag_list) > 0xFFFF)
        error_log(ERROR_FATAL, "Fragment List too big");
    if (DLL_GetNumberOfRecords(rxc->dup_list) > 0xFFFF)
        error_log(ERROR_FATAL, "Dup List too big");
    num_of_frags = DLL_GetNumberOfRecords(rxc->frag_list);
    num_of_dups = DLL_GetNumberOfRecords(rxc->dup_list);
    event_logii(VVERBOSE, "len of frag_list==%u, len of dup_list==%u", num_of_frags, num_of_dups);
    /* FIXME : Limit number of Fragments/Duplicates according to PATH MTU */

    sack = rxc->sack_chunk;
    sack->chunk_header.chunk_id = CHUNK_SACK;
    sack->chunk_header.chunk_flags = 0;
    len32 =
        sizeof(SCTP_chunk_header) + (2 + num_of_dups) * sizeof(unsigned int) +
        (2 * num_of_frags + 2) * sizeof(unsigned short);
    if (len32 > 0xFFFF)
        error_log(ERROR_FATAL, "Chunk Length becomes too big");
    len16 = (unsigned short) len32;
    sack->chunk_header.chunk_length = htons(len16);
    sack->cumulative_tsn_ack = htonl(rxc->ctsna);
    /* FIXME : deduct size of data still in queue, that is waiting to be picked up by an ULP */
    sack->a_rwnd = htonl(rxc->my_rwnd);
    sack->num_of_fragments = htons(num_of_frags);
    sack->num_of_duplicates = htons(num_of_dups);
    pos = 0L;
    ExitCode = DLL_CurrentPointerToHead(rxc->frag_list);
    ExitCode = DLL_CurrentPointerToHead(rxc->dup_list);

    for (count = 0; count < num_of_frags; count++) {
        if ((ExitCode = DLL_GetCurrentRecord(rxc->frag_list, &f32)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);
        event_logiii(VVERBOSE,
                     "ctsna==%u, fragment.start==%u, fragment.stop==%u",
                     rxc->ctsna, f32.start_tsn, f32.stop_tsn);
        if (((f32.start_tsn - rxc->ctsna) > 0xFFFF)
            || ((f32.stop_tsn - rxc->ctsna) > 0xFFFF)) {
            error_log(ERROR_MINOR, "Fragment offset becomes too big");
            break;
        }
        frag_start16 = (unsigned short) (f32.start_tsn - rxc->ctsna);
        frag_stop16 = (unsigned short) (f32.stop_tsn - rxc->ctsna);
        event_logii(VVERBOSE, "frag_start16==%u, frag_stop16==%u", frag_start16, frag_stop16);

        chunk_frag.start = htons(f32.start_tsn - rxc->ctsna);
        chunk_frag.stop = htons(f32.stop_tsn - rxc->ctsna);
        event_logii(VVERBOSE, "chunk_frag.start=%u,chunk_frag.stop ==%u",
                    ntohs(chunk_frag.start), ntohs(chunk_frag.stop));
        memcpy(&sack->fragments_and_dups[pos], &chunk_frag, sizeof(fragment));
        pos += sizeof(fragment);
        if ((ExitCode = DLL_IncrementCurrentPointer(rxc->frag_list)) != DLL_NORMAL)
            break;
    }

    for (count = 0; count < num_of_dups; count++) {
        if ((ExitCode = DLL_GetCurrentRecord(rxc->dup_list, &d)) != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);
        d.duplicate_tsn = htonl(d.duplicate_tsn);
        memcpy(&sack->fragments_and_dups[pos], &d, sizeof(duplicate));
        pos += sizeof(duplicate);
        if ((ExitCode = DLL_IncrementCurrentPointer(rxc->dup_list)) != DLL_NORMAL)
            break;
    }
    /* start sack_timer set to 200 msecs */
    if (rxc->timer_running != TRUE) {
        rxc->sack_timer = sctp_startTimer(rxc->delay, &rxc_sack_timer_cb, &(rxc->my_association), NULL);
        event_log(INTERNAL_EVENT_0, "Started SACK Timer !");
        rxc->timer_running = TRUE;
    }
    rxc->contains_valid_sack = TRUE;
    return;
}


/**
  @return my current receiver window (32 bit unsigned value)
 */
unsigned int rxc_get_local_receiver_window(void)
{
    rxc_buffer *rxc;
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MINOR, "rxc_buffer instance not set - returning 0");
        return (0);
    }
    event_logi(VERBOSE, "function rxc_get_my_receiver_window() returns %u", rxc->my_rwnd);
    return rxc->my_rwnd;
}


/**
  @return my current sack delay in msecs
 */
int rxc_get_sack_delay(void)
{
    rxc_buffer *rxc;
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MINOR, "rxc_buffer instance not set - returning default");
        return (-1);
    }
    event_logi(VERBOSE, "function rxc_get_sack_delay() returns %u", rxc->delay);
    return ((int)rxc->delay);
}

/**
  @return my current sack delay in msecs
 */
int rxc_set_sack_delay(unsigned int new_delay)
{
    rxc_buffer *rxc;
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MINOR, "rxc_buffer instance not set - returning default");
        return (-1);
    }
    rxc->delay = new_delay;
    event_logi(VERBOSE, "Setting new sack delay  to %u msecs", rxc->delay);
    return 0;
}

/**
 Set the size of my receiver window. This needs to reflect buffer sizes.
 Beware, this is really only a DUMMY function, too !
 @param  new local receiver window (32 bit unsigned value)
 @return 0 on success, else -1 on failure
 */
int rxc_set_local_receiver_window(unsigned int new_window)
{
    rxc_buffer *rxc;
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return (-1);
    }
    event_logi(VERBOSE, "function rxc_set_my_receiver_window(%u)", new_window);
    rxc->my_rwnd = new_window;
    return 0;
}


/**
  Get the number of the current cumulative TSN, that we may ack
  @return my current ctsna (32 bit unsigned value)
 */
unsigned int rxc_read_cummulativeTSNacked(void)
{
    rxc_buffer *rxc;

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return (0);
    }
    return (rxc->ctsna);
}

/**
 * Helper function called, when we have gap reports in incoming
 * SACK chunks....
 */
void rxc_send_sack_everytime(void)
{
    rxc_buffer *rxc;

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return;
    }
    rxc->sack_flag = 1;
}

/**
 * Helper function called, when we have no gap reports in incoming
 * SACK chunks....
 */
void rxc_send_sack_every_second_time(void)
{
    rxc_buffer *rxc;

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return;
    }
    rxc->sack_flag = 2;

}

/**
 * Helper function to get last address on which data arrived
 */
short rxc_read_last_active_address(void)
{
    rxc_buffer *rxc;
    short idx;
    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return 0;
    }
    if (rxc->last_address > 0xFFFF)
        error_log(ERROR_MAJOR, "Address index out of range");
    idx = (short) rxc->last_address;
    if (rxc->datagrams_received == -1)
        return pm_readPrimaryPath();
    else
        return idx;

}


/**
 function only called in a restart case.
 Beware : this has been largely untested !
 @param  new_remote_TSN new tsn value of peer that has restarted
 */
void rxc_restart_receivecontrol(unsigned int new_remote_TSN)
{
    rxc_buffer *rxc;
    DLL_Return ExitCode;

    rxc = (rxc_buffer *) mdi_readRX_control();
    if (!rxc) {
        error_log(ERROR_MAJOR, "rxc_buffer instance not set !");
        return;
    }
    rxc_stop_sack_timer();
    DLL_DestroyList(&(rxc->frag_list));
    DLL_DestroyList(&(rxc->dup_list));
    rxc->ctsna = new_remote_TSN - 1;
    rxc->lowest = new_remote_TSN - 1;
    rxc->highest = new_remote_TSN - 1;
    /* initialize and set up lists */
    if (DLL_CreateList(&(rxc->frag_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if (DLL_CreateList(&(rxc->dup_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if ((ExitCode = DLL_InitializeList(rxc->frag_list, sizeof(fragment32))) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    if ((ExitCode = DLL_InitializeList(rxc->dup_list, sizeof(duplicate))) != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);

    rxc->contains_valid_sack = FALSE;
    rxc->timer_running = FALSE;
    rxc->datagrams_received = -1;
    rxc->sack_flag = 2;
    rxc->last_address = 0;
    rxc->my_rwnd = 10000;
    rxc->my_association = mdi_readAssociationID();
    return;

}
