/*
 *  $Id: reltransfer.c,v 1.3 2001/03/13 14:14:45 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module implements the retransmission mechanism, and stores
 * all data that has not been acknowledged by the peer.
 *
 */

#include "adaptation.h"
#include "reltransfer.h"
#include "flowcontrol.h"
#include "recvctrl.h"
#include "pathmanagement.h"
#include "distribution.h"
#include "SCTP-control.h"
#include "dll_main.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/* here we should not have to worry about wrap */
#define max(x,y)            (x)>(y)?(x):(y)
#define min(x,y)            (x)<(y)?(x):(y)

#define MAX_NUM_OF_CHUNKS   375 /* for 1500 MTU */

static chunk_data *rtx_chunks[MAX_NUM_OF_CHUNKS];

/**
 * this struct contains all necessary data for retransmissions
 * and processing of received SACKs
 */
typedef struct rtx_buffer_struct
{
    //@{
    /** storing the lowest tsn that is in the list */
    unsigned int lowest_tsn;
    ///
    unsigned int highest_tsn;
    ///
    unsigned int num_of_chunks;
    /** a list that is ordered by ascending tsn values */
    List *chunk_list;
    ///
    struct timeval sack_arrival_time;
    ///
    struct timeval save_send_time;
    /** this val stores 0 if retransmitted chunks have been acked, else 1 */
    unsigned int save_num_of_txm;
    ///
    unsigned int newly_acked_bytes;
    ///
    unsigned int num_of_addresses;
    ///
    unsigned int *newly_acked_per_address;
    ///
    unsigned int my_association;
    ///
    unsigned int peer_arwnd;
    ///
    boolean shutdown_received;
    //@}
} rtx_buffer;


/**
 * after submitting results from a SACK to flowcontrol, the counters in
 * reliable transfer must be reset
 * @param rtx   pointer to a rtx_buffer, where acked bytes per address will be reset to 0
 */
void rtx_reset_bytecounters(rtx_buffer * rtx)
{
    unsigned int count;

    rtx->newly_acked_bytes = 0L;
    for (count = 0; count < rtx->num_of_addresses; count++)
        rtx->newly_acked_per_address[count] = 0L;
    return;
}


/**
 * function creates and allocs new rtx_buffer structure.
 * There is one such structure per established association
 * @param   number_of_destination_addresses     number of paths to the peer of the association
 * @return pointer to the newly created structure
 */
void *rtx_new_reltransfer(unsigned int number_of_destination_addresses)
{
    rtx_buffer *tmp;
    DLL_Return ExitCode;

    tmp = malloc(sizeof(rtx_buffer));
    if (!tmp)
        error_log(ERROR_FATAL, "Malloc failed");

    event_logi(VVERBOSE,
               "================== Reltransfer: number_of_destination_addresses = %d",
               number_of_destination_addresses);

    if (DLL_CreateList(&(tmp->chunk_list)) == NULL)
        DLL_error_log(ERROR_FATAL, DLL_MEM_ERROR);
    if ((ExitCode = DLL_InitializeList(tmp->chunk_list, sizeof(chunk_data)))
        != DLL_NORMAL)
        DLL_error_log(ERROR_FATAL, ExitCode);
    tmp->lowest_tsn = 0L;
    tmp->highest_tsn = 0L;
    tmp->newly_acked_bytes = 0L;
    tmp->num_of_chunks = 0L;
    tmp->save_num_of_txm = 0L;
    tmp->peer_arwnd = 0L;
    tmp->shutdown_received = FALSE;
    tmp->num_of_addresses = number_of_destination_addresses;
    tmp->newly_acked_per_address = malloc(number_of_destination_addresses * sizeof(unsigned int));
    tmp->my_association = mdi_readAssociationID();
    event_logi(VVERBOSE, "RTX : Association-ID== %d ", tmp->my_association);
    if (tmp->my_association == 0)
        error_log(ERROR_FATAL, "Association was not set, should be......");
    rtx_reset_bytecounters(tmp);
    return (tmp);
}

/**
 * function deletes a rtx_buffer structure (when it is not needed anymore)
 * @param rtx_instance pointer to a rtx_buffer, that was previously created
            with rtx_new_reltransfer()
 */
void rtx_delete_reltransfer(void *rtx_instance)
{
    rtx_buffer *rtx;
    rtx = (rtx_buffer *) rtx_instance;
    event_log(INTERNAL_EVENT_0, "deleting reliable transfer");
    if (DLL_IsListEmpty(rtx->chunk_list) == DLL_FALSE)
        error_log(ERROR_MINOR, "List is being deleted, but chunks are still queued...");
    DLL_DestroyList(&(rtx->chunk_list));
    free(rtx->newly_acked_per_address);
    free(rtx_instance);
}


/**
 * helper function that calls pm_chunksAcked()
 * and tells path management, if new chunks have  been acked, and new RTT may be guessed
 * @param  adr_idx  CHECKME : address where chunks have been acked (is this correct ?);
            may we take src address of the SACK, or must we take destination address of our data ?
 * @param    rtx    pointer to the currently active rtx structure
 */
void rtx_rtt_update(unsigned int adr_idx, rtx_buffer * rtx)
{
    /* FIXME : check this routine !!!!!!!!!!! */
    int rtt;
    event_logi(INTERNAL_EVENT_0, "rtx_update_rtt(address=%u... ", adr_idx);
    if (rtx->save_num_of_txm == 1) {
        rtt = adl_timediff_to_msecs(&(rtx->sack_arrival_time), &(rtx->save_send_time));
        if (rtt != -1) {
            event_logii(ERROR_MINOR, "Calling pm_chunksAcked(%u, %d)...", adr_idx, rtt);
            pm_chunksAcked(adr_idx, rtt);
        }
    } else {
        event_logi(VERBOSE, "Calling pm_chunksAcked(%u, 0)...", adr_idx);
        pm_chunksAcked(adr_idx, 0L);
    }
    return;
}


/**
 * helper function for sorting list of chunks in tsn order
 * @param  one pointer to chunk data
 * @param  two pointer to other chunk data
 * @return 0 if chunks have equal tsn, -1 if tsn1 < tsn2, 1 if tsn1 > tsn2
 */
int rtx_sort_tsn(chunk_data * one, chunk_data * two)
{
    if (before(one->chunk_tsn, two->chunk_tsn)) {
        return -1;
    } else if (after(one->chunk_tsn, two->chunk_tsn)) {
        return 1;
    } else                      /* one==two */
        return 0;
}

/**
 * Function takes out chunks up to ctsna, updates newly acked bytes
 * @param   ctsna   the ctsna value, that has just been received in a sack
 * @return -1 if error (such as ctsna > than all chunk_tsn), 0 on success
 */
int rtx_dequeue_up_to(unsigned int ctsna)
{
    DLL_Return ExitCode;
    rtx_buffer *rtx;
    chunk_data dat;
    boolean deleted_chunk = FALSE;

    event_logi(INTERNAL_EVENT_0, "rtx_dequeue_up_to...%u ", ctsna);

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return (-1);
    }
    if ((ExitCode = DLL_CurrentPointerToHead(rtx->chunk_list)) != DLL_NORMAL) {
        event_log(INTERNAL_EVENT_0, "List is NULL in rtx_dequeue_up_to()");
        return -1;
    }

    while ((ExitCode = DLL_GetCurrentRecord(rtx->chunk_list, &dat)) == DLL_NORMAL) {
        /*              event_log(INTERNAL_EVENT_0,"while loop...tsn== %u",dat.chunk_tsn); */
        deleted_chunk = FALSE;
        event_logiiii(VVERBOSE,
                      " dat.num_of_transmissions==%u, dat.chunk_tsn==%u, chunk_len=%u, ctsna==%u ",
                      dat.num_of_transmissions, dat.chunk_tsn, dat.chunk_len, ctsna);
        if (before(dat.chunk_tsn, ctsna) || (dat.chunk_tsn == ctsna)) {
            rtx->newly_acked_bytes += dat.chunk_len;
            if (dat.num_of_transmissions < 1)
                error_log(ERROR_MAJOR, "Somehow dat.num_of_transmissions is less than 1 !");

            rtx->newly_acked_per_address[dat.used_destinations
                                         [dat.num_of_transmissions - 1]] += dat.chunk_len;
            if (dat.num_of_transmissions > 1) {
                rtx->save_num_of_txm = 0;
            } else if (dat.num_of_transmissions == 1) {
                memcpy(&rtx->save_send_time, &(dat.transmission_time), sizeof(struct timeval));
                event_logiii(VERBOSE,
                             "Saving Time (after dequeue) : %lu secs, %06lu usecs for tsn=%u",
                             dat.transmission_time.tv_sec,
                             dat.transmission_time.tv_usec, dat.chunk_tsn);
            }

            ExitCode = DLL_DeleteCurrentRecord(rtx->chunk_list);
            event_logi(INTERNAL_EVENT_0, "Deleted record...%u", dat.chunk_tsn);
            if (ExitCode != DLL_NORMAL) {
                DLL_error_log(ERROR_MAJOR, ExitCode);
                break;
            }
            deleted_chunk = TRUE;
        }
        /* it is a sorted list, so it is safe to get out in these cases */
        if (DLL_IsListEmpty(rtx->chunk_list) == DLL_TRUE)
            break;
        if (after(dat.chunk_tsn, ctsna))
            break;
        if (deleted_chunk == FALSE) {
            /* increment pointer only if chunk was not deleted */
            if ((ExitCode = DLL_IncrementCurrentPointer(rtx->chunk_list)) != DLL_NORMAL)
                break;
        }
    }
    return 0;
}



/**
 * this is called by bundling, when a SACK needs to be processed. This is a LONG function !
 * FIXME : check correct update of rtx->lowest_tsn !
 * CHECK : did SACK ack lowest outstanding tsn, restart t3 timer (section 7.2.4.4) )
 * @param  adr_index   index of the address where we got that sack
 * @param  sack_chunk  pointer to the sack chunk
 * @return -1 on error, 0 if okay.
 */
int rtx_process_sack(unsigned int adr_index, void *sack_chunk)
{
    DLL_Return ExitCode;
    rtx_buffer *rtx;
    SCTP_sack_chunk *sack;
    unsigned int advertised_rwnd;
    int result;
    fragment *frag;
    chunk_data dat;
    boolean rtx_necessary = FALSE, all_acked = FALSE, new_acked = FALSE;
    unsigned int low, hi, ctsna, pos, chunk_len, var_len, old_own_ctsna;
    /* this is an array of POINTERS to the chunk_data structs... */
    int chunks_to_rtx = 0;
    unsigned int max_rtx_arraysize;
    unsigned int rtx_length = 0L, retransmitted_bytes = 0L;
    short idx = 0;
    boolean is_tail;

    event_logi(INTERNAL_EVENT_0, "rtx_process_sack(address==%u)", adr_index);

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return (-1);
    }

    /*      chunk_list_debug(rtx->chunk_list); */

    sack = (SCTP_sack_chunk *) sack_chunk;
    ctsna = ntohl(sack->cumulative_tsn_ack);
    old_own_ctsna = rtx->lowest_tsn;
    event_logii(VVERBOSE, "Received ctsna==%u, old_own_ctsna==%u", ctsna, old_own_ctsna);

    rtx->save_num_of_txm = 1;
    adl_gettime(&(rtx->sack_arrival_time));

    event_logii(VERBOSE, "SACK Arrival Time : %lu secs, %06lu usecs",
                rtx->sack_arrival_time.tv_sec, rtx->sack_arrival_time.tv_usec);

    /* a false value here may do evil things !!!!! */
    chunk_len = ntohs(sack->chunk_header.chunk_length);
    /* maybe add sanity checks  !!! */
    advertised_rwnd = ntohl(sack->a_rwnd);
    var_len =
        chunk_len - sizeof(SCTP_chunk_header) - 2 * sizeof(unsigned int) -
        2 * sizeof(unsigned short);
    event_logiii(VVERBOSE, "chunk_len==%u, a_rwnd==%u, var_len==%u", chunk_len,
                 advertised_rwnd, var_len);
    if (after(ctsna, rtx->lowest_tsn) || (ctsna == rtx->lowest_tsn)) {
        event_logiii(VVERBOSE,
                     "after(%u, %u) == true, call rtx_dequeue_up_to(%u)",
                     ctsna, rtx->lowest_tsn, ctsna);
        result = rtx_dequeue_up_to(ctsna);
        if (result < 0) {
            event_log(EXTERNAL_EVENT_X,
                      "Bad ctsna arrived in SACK or no data in queue - discarding SACK");
            return -1;
        }
        rtx->lowest_tsn = ctsna;
        event_logi(VVERBOSE, "Updated rtx->lowest_tsn==ctsna==%u", ctsna);
    }

/*  	chunk_list_debug(VVERBOSE, rtx->chunk_list); */

    if (ntohs(sack->num_of_fragments) != 0) {
        event_logi(VERBOSE, "Processing %u fragment reports", ntohs(sack->num_of_fragments));
        max_rtx_arraysize = DLL_GetNumberOfRecords(rtx->chunk_list);
        if (max_rtx_arraysize == 0) {
            /*rxc_send_sack_everytime(); */
            event_log(VERBOSE,
                      "Size of retransmission list was zero, but we received fragment report");
        } else {
            /* use     num_of_rtxm as index to that array.... */
            /*              rxc_send_sack_everytime();     */
            /* this will be expensive !!!!!!!!!!!!!!!! */
            pos = 0;

            ExitCode = DLL_CurrentPointerToHead(rtx->chunk_list);
            ExitCode = DLL_GetCurrentRecord(rtx->chunk_list, &dat);
            if (ExitCode == DLL_NORMAL) {
                do {
                    frag = (fragment *) & (sack->fragments_and_dups[pos]);
                    low = ctsna + ntohs(frag->start);
                    hi = ctsna + ntohs(frag->stop);
                    event_logiii(VVERBOSE, "chunk_tsn==%u, lo==%u, hi==%u", dat.chunk_tsn, low, hi);
                    if (after(dat.chunk_tsn, hi)) {
                        event_logii(VVERBOSE, "after(%u,%u)==true", dat.chunk_tsn, hi);
                        pos += sizeof(fragment);
                        if (pos >= var_len)
                            break;
                        continue;
                    }
                    if (before(dat.chunk_tsn, low)) {
                        /* this chunk is in a gap... */
                        dat.gap_reports++;
                        event_logiii(VVERBOSE,
                                     "Chunk in a gap: before(%u,%u)==true -- Marking it up (%u Gap Reports)!",
                                     dat.chunk_tsn, low, dat.gap_reports);
                        if (dat.gap_reports == 4) {
                            event_logi(VVERBOSE,
                                       "Got four gap_reports, scheduling %u for RTX",
                                       dat.chunk_tsn);
                            rtx_necessary = TRUE;
                            /* FIXME : Get MTU of address, where RTX is to take place, instead of MAX_SCTP_PDU */
                            if ((rtx_length + dat.chunk_len) < MAX_SCTP_PDU) {
                                /* check sum of chunk sizes (whether it exceeds MTU for current address */
                                rtx_chunks[chunks_to_rtx] = DLL_GetCurrentPointer(rtx->chunk_list);
                                chunks_to_rtx++;
                                /* preparation for what is in section 6.2.1.C */
                                retransmitted_bytes += dat.chunk_len;
                            }
                        }
                        ExitCode = DLL_UpdateCurrentRecord(rtx->chunk_list, &dat);
                        if (ExitCode != DLL_NORMAL) {
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                            break;
                        }
                        /* read next chunk */
                        ExitCode = DLL_GetNextRecord(rtx->chunk_list, &dat);
                        if (ExitCode == DLL_NOT_FOUND)
                            break; /* was the last chunk in the list */
                        if (chunks_to_rtx == MAX_NUM_OF_CHUNKS)
                            break;
                        else if (ExitCode == DLL_NULL_LIST) /* just in case */
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                        else
                            continue;
                    } else if (between(low, dat.chunk_tsn, hi)) {
                        event_logiii(VVERBOSE, "between(%u,%u,%u)==true", low, dat.chunk_tsn, hi);
                        rtx->newly_acked_bytes += dat.chunk_len;
                        if (dat.num_of_transmissions < 1)
                            error_log(ERROR_MAJOR,
                                      "Somehow dat.num_of_transmissions is less than 1 !");

                        rtx->newly_acked_per_address[dat.used_destinations
                                                     [dat.num_of_transmissions
                                                      - 1]] += dat.chunk_len;

                        if (dat.num_of_transmissions == 1) {
                            memcpy(&rtx->save_send_time,
                                   &(dat.transmission_time), sizeof(struct timeval));
                        } else if (dat.num_of_transmissions > 1)
                            rtx->save_num_of_txm = 0;
                        /* test, if current record is at the tail of the list */
                        is_tail = FALSE;
                        if (DLL_IsCurrentTail(rtx->chunk_list))
                            is_tail = TRUE;
                        event_logi(VVERBOSE, "Deleting chunk_tsn(%u) from rtx-list", dat.chunk_tsn);
                        ExitCode = DLL_DeleteCurrentRecord(rtx->chunk_list);
                        if (ExitCode != DLL_NORMAL) {
                            DLL_error_log(ERROR_MAJOR, ExitCode);
                            break;
                        }
                        if (is_tail == TRUE)
                            break;
                        ExitCode = DLL_GetCurrentRecord(rtx->chunk_list, &dat);
                        continue;
                    } else if (after(dat.chunk_tsn, hi)) {
                        error_log(ERROR_MINOR, "Problem with fragment boundaries (low_2 <= hi_1)");
                        break;
                    }

                }
                while ((pos < var_len));

            } /* if ((ExitCode == DLL_NORMAL) */
            else {
                event_log(EXTERNAL_EVENT,
                          "Received duplicated SACK for Chunks that are not in the queue anymore");
            }
        }

    } else {                    /* no gaps reported in this SACK */

        /* rxc_send_sack_every_second_time(); */
    }
    event_log(INTERNAL_EVENT_0, "Marking of Chunks done in rtx_process_sack()");
    chunk_list_debug(VVERBOSE, rtx->chunk_list);

    /* also tell pathmanagement, that we got a SACK, possibly updating RTT/RTO. */
    rtx_rtt_update(adr_index, rtx);

    /*
     * new_acked==TRUE means our own ctsna has advanced :
     * also see section 6.2.1 (Note)
     */
    if (DLL_IsListEmpty(rtx->chunk_list) == DLL_TRUE) {
        all_acked = TRUE;
        /* section 6.2.1.D.ii) */
        rtx->peer_arwnd = advertised_rwnd;
        rtx->lowest_tsn = rtx->highest_tsn;
        if (after(rtx->lowest_tsn, old_own_ctsna))
            new_acked = TRUE;

        /* in the case where shutdown was requested by the ULP, and all is acked (i.e. ALL queues are empty) ! */
        if (rtx->shutdown_received == TRUE) {
            if (fc_readNumberOfQueuedChunks() == 0) {
                sci_allChunksAcked();
            }
        }
    } else {
        if ((ExitCode = DLL_CurrentPointerToHead(rtx->chunk_list)) != DLL_NORMAL) {
            DLL_error_log(ERROR_MINOR, ExitCode);
            return -1;
        }
        /* there are still chunks in that queue */
        ExitCode = DLL_GetCurrentRecord(rtx->chunk_list, &dat);
        if (ExitCode != DLL_NORMAL)
            DLL_error_log(ERROR_MAJOR, ExitCode);
        rtx->lowest_tsn = dat.chunk_tsn;
        /* new_acked is true, when own  ctsna advances... */
        if (rtx->lowest_tsn > old_own_ctsna) {
            new_acked = TRUE;
        }
    }

    if (rtx->shutdown_received == TRUE)
        rxc_send_sack_everytime();

    event_logiiii(VVERBOSE,
                  "rtx->lowest_tsn==%u, new_acked==%s, all_acked==%s, rtx_necessary==%s\n",
                  rtx->lowest_tsn, ((new_acked == TRUE) ? "TRUE" : "FALSE"),
                  ((all_acked == TRUE) ? "TRUE" : "FALSE"),
                  ((rtx_necessary == TRUE) ? "TRUE" : "FALSE"));
    if (rtx_necessary == FALSE) {
        fc_sack_info(adr_index, advertised_rwnd, all_acked, new_acked,
                     rtx->newly_acked_bytes, rtx->num_of_addresses, rtx->newly_acked_per_address);
        rtx_reset_bytecounters(rtx);
    } else {
        /* retval = */ fc_do_retransmission(adr_index, advertised_rwnd,
                                            retransmitted_bytes,
                                            all_acked, new_acked,
                                            rtx->newly_acked_bytes,
                                            rtx->num_of_addresses,
                                            rtx->newly_acked_per_address,
                                            chunks_to_rtx, rtx_chunks);
        rtx_reset_bytecounters(rtx);
        if (adr_index > 0xFFFF){
            error_log(ERROR_FATAL, "Type Problems with Addresses");
        } else {
            idx = (short) adr_index;
        }
        for (pos = 0; pos < chunks_to_rtx; pos++) {
            /* adjust in flowcontrol...rtx_chunks[pos]->num_of_transmissions++; */
            rtx_chunks[pos]->gap_reports = 0;
        }
    }
    return 0;
}


/**
 * called from flow-control to trigger retransmission of chunks that have previously
 * been sent to the address that timed out.
 * It is only called from flowcontrol, so association should be set correctly here
 * @param  assoc_id     pointer to the id value of the association, where timeout occurred
 * @param  address      address that timed out
 * @param   mtu         current path mtu (this needs to be fixed, too !)
 * @param   chunks      pointer to an array, that will contain pointers to chunks that need to
                        be retransmitted after this function returns. Provide space !
 * @return  -1 on error, 0 for empty list, else number of chunks that can be retransmitted
 */
int rtx_t3_timeout(void *assoc_id, unsigned int address, unsigned int mtu, chunk_data ** chunks)
{
    DLL_Return ExitCode;
    rtx_buffer *rtx;
    /* assume a SACK with 5 fragments and 5 duplicates :-) */
    /* it's size == 20+5*4+5*4 == 60        */
    unsigned int size = 60;
    int chunks_to_rtx = 0;
    void *tmp;

    event_logi(INTERNAL_EVENT_0, "rtx_t3_timeout (address==%u)", address);

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (DLL_IsListEmpty(rtx->chunk_list) == DLL_TRUE)
        return 0;
    if ((ExitCode = DLL_CurrentPointerToHead(rtx->chunk_list)) != DLL_NORMAL) {
        DLL_error_log(ERROR_MAJOR, ExitCode);
        return -1;
    }
    tmp = DLL_GetCurrentPointer(rtx->chunk_list);
    do {
        if (((chunk_data *) tmp)->num_of_transmissions < 1) {
            error_log(ERROR_MAJOR,
                      "Somehow (chunk_data*)tmp)->num_of_transmissions is less than 1 !");
            break;
        }
        /* only take chunks that were transmitted to *address* */
        if (((chunk_data *) tmp)->used_destinations
            [((chunk_data *) tmp)->num_of_transmissions - 1] == address) {
            chunks[chunks_to_rtx] = tmp;
            size += chunks[chunks_to_rtx]->chunk_len;
            event_logii(VVERBOSE,
                        "Scheduling chunk (tsn==%u), len==%u for rtx",
                        chunks[chunks_to_rtx]->chunk_tsn, chunks[chunks_to_rtx]->chunk_len);
            chunks_to_rtx++;
        }
        ExitCode = DLL_IncrementCurrentPointer(rtx->chunk_list);
        if (ExitCode == DLL_NORMAL)
            tmp = DLL_GetCurrentPointer(rtx->chunk_list);
        else
            tmp = NULL;

    }
    while (size <= mtu && tmp != NULL);
    event_logi(VVERBOSE, "Scheduled %d chunks for rtx", chunks_to_rtx);
    if (address > 0x8FFF)
        error_logi(ERROR_FATAL, "False Address %u", address);
    if (chunks_to_rtx == 0) {
        error_logi(ERROR_MAJOR, "No chunks to retransmit to address %u", address);
    }
    return chunks_to_rtx;
}

/**
 * a function called by FlowCtrl, when chunk has been given to the bundling
 * instance, but is already contained in the reliable transfer list.
 * some data in that chunks must be updated.
 * @param  data_chunk   pointer to tha chunk that is to be updated
 * @param   dest        index of the destination address where the chunk will go next
 * @return  -1 on error, 0 if okay
 */
int rtx_update_retrans_chunks(void *data_chunk, unsigned int dest)
{
    chunk_data *match;
    chunk_data upd_chunk;
    rtx_buffer *rtx;
    DLL_Return ExitCode;
    int (*pfun) () = NULL;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return (-1);
    }
    pfun = rtx_sort_tsn;
    match = (chunk_data *) data_chunk;
    event_logii(INTERNAL_EVENT_0,
                "rtx_update_retrans_chunks (chunk-tsn==%u, destination=%u)",
                match->chunk_tsn, dest);

    ExitCode = DLL_FindRecord(rtx->chunk_list, &upd_chunk, match, pfun);
    if (ExitCode != DLL_NORMAL) {
        error_logi(ERROR_MINOR,
                   "rtx_update_retrans_chunks: chunk (tsn=%u) not found", match->chunk_tsn);
        return -1;
    }
    upd_chunk.num_of_transmissions++;
    if (upd_chunk.num_of_transmissions >= MAX_DEST)
        upd_chunk.num_of_transmissions = MAX_DEST;
    upd_chunk.used_destinations[upd_chunk.num_of_transmissions - 1] = dest;
    memcpy(&(upd_chunk.transmission_time),
           &(((chunk_data *) data_chunk)->transmission_time), sizeof(struct timeval));
    ExitCode = DLL_UpdateCurrentRecord(rtx->chunk_list, &upd_chunk);
    if (ExitCode != DLL_NORMAL) {
        error_log(ERROR_MINOR, "rtx_update_retrans_chunks: list problem");
        return -1;
    }
    chunk_list_debug(VVERBOSE, rtx->chunk_list);
    return 0;
}

/**
 * a function called by FlowCtrl, when chunks have been given to the bundling
 * instance, but need to be kept in the buffer until acknowledged
 * @return 0 if OK, -1 if there is an error (list error)
 */
int rtx_save_retrans_chunks(void *data_chunk)
{
    chunk_data *dat;
    rtx_buffer *rtx;
    DLL_Return ExitCode;
    int (*pfun) () = NULL;

    event_log(INTERNAL_EVENT_0, "rtx_save_retrans_chunks");

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return (-1);
    }

    /*      chunk_list_debug(rtx->chunk_list); */

    dat = (chunk_data *) data_chunk;

    /* TODO : check, if all values are set correctly */
    dat->gap_reports = 0L;
    pfun = rtx_sort_tsn;
    if ((ExitCode = DLL_AddRecord(rtx->chunk_list, dat, pfun)) != DLL_NORMAL) {
        DLL_error_log(ERROR_FATAL, ExitCode);
        return -1;
    }
    if (after(dat->chunk_tsn, rtx->highest_tsn))
        rtx->highest_tsn = dat->chunk_tsn;
    else
        error_log(ERROR_MINOR, "Data Chunk has TSN that was already assigned (i.e. is too small)");

    chunk_list_debug(VVERBOSE, rtx->chunk_list);

    rtx->num_of_chunks = DLL_GetNumberOfRecords(rtx->chunk_list);
    return 0;
}

/**
 * output debug messages for the list of saved chunks
 * @param   event_log_level  INTERNAL_EVENT_0 INTERNAL_EVENT_1 EXTERNAL_EVENT_X EXTERNAL_EVENT
 * @param   chunk_list  the list about which we print information
 */
void chunk_list_debug(short event_log_level, List * chunk_list)
{
    DLL_Return ExitCode;
    chunk_data *dat;
    unsigned int size, count;

    if (event_log_level <= Current_event_log_) {
        event_log(event_log_level, "------------- Chunk List Debug ------------------------\n");
        DLL_StoreCurrentPointer(chunk_list);
        if ((size = DLL_GetNumberOfRecords(chunk_list)) == 0) {
            event_log(event_log_level, " Size of List == 0 ! ");
        } else if (size <= 10) {
            event_logi(event_log_level, " Size of List == %u ! ", size);
            if ((ExitCode = DLL_CurrentPointerToHead(chunk_list)) != DLL_NORMAL) {
                DLL_error_log(ERROR_MAJOR, ExitCode);
                DLL_RestoreCurrentPointer(chunk_list);
                return;
            }
            do {
                dat = DLL_GetCurrentPointer(chunk_list);
                event_logii(event_log_level,
                            "________________ Chunk _________________\nChunk Size %u  -- TSN : %u  ",
                            dat->chunk_len, dat->chunk_tsn);
                event_logiii(event_log_level,
                             "Gap repts=%u -- initial dest=%d  Transmissions = %u",
                             dat->gap_reports, dat->initial_destination, dat->num_of_transmissions);
                event_logii(event_log_level,
                            "Transmission Time : %lu secs, %06lu usecs",
                            dat->transmission_time.tv_sec, dat->transmission_time.tv_usec);
                for (count = 0; count < dat->num_of_transmissions; count++)
                    event_logii(event_log_level, "Destination[%u] == %u", count,
                                dat->used_destinations[count]);
            }
            while ((ExitCode = DLL_IncrementCurrentPointer(chunk_list)) != DLL_NOT_FOUND);
            event_log(event_log_level, "------------- Debug : DONE  ------------------------");
        } else {
            event_logi(event_log_level, " Size of List == %u ! ", size);
        }
        DLL_RestoreCurrentPointer(chunk_list);
    }
}


/**
 * function that returns the consecutive tsn number that has been acked by the peer.
 * @return the ctsna value
 */
unsigned int rtx_readLocalTSNacked()
{
    rtx_buffer *rtx;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        event_log(INTERNAL_EVENT_0, "rtx_buffer instance not set !");
        return (0);
    }
    return rtx->lowest_tsn;
}

/**
 * called, when a Cookie, that indicates the peer's restart, is received in the ESTABLISHED state
    -> we need to restart too
 */
void rtx_restart_reliable_transfer()
{
    chunk_data *chunk_ptr;
    rtx_buffer *rtx;
    DLL_Return ExitCode;
    int count;
    /* ******************************************************************* */
    /* FIXME :
       IMPLEMENTATION NOTE: It is an implementation decision on how
       to handle any pending datagrams. The implementation may elect
       to either A) send all messages back to its upper layer with the
       restart report, or B) automatically re-queue any datagrams
       pending by marking all of them as never-sent and assigning
       new TSN's at the time of their initial transmissions based upon
       the updated starting TSN (as defined in section 5).

       Version 13 says : SCTP data chunks MAY be retained !
       (this is implementation specific)
       ******************************************************************** */
    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return;
    }
    if (DLL_IsListEmpty(rtx->chunk_list) == DLL_TRUE)
        return;

    if ((ExitCode = DLL_CurrentPointerToHead(rtx->chunk_list)) != DLL_NORMAL) {
        DLL_error_log(ERROR_MAJOR, ExitCode);
        return;
    }
    chunk_ptr = DLL_GetCurrentPointer(rtx->chunk_list);
    do {
        chunk_ptr->gap_reports = 0;
        timerclear(&(chunk_ptr->transmission_time));
        chunk_ptr->ack_time = 0;
        chunk_ptr->initial_destination = (-1);
        for (count = 0; count < chunk_ptr->num_of_transmissions; count++)
            chunk_ptr->used_destinations[count] = 0xFFFFFFFF;
        chunk_ptr->num_of_transmissions = 0;

        ExitCode = DLL_IncrementCurrentPointer(rtx->chunk_list);
        if (ExitCode == DLL_NORMAL)
            chunk_ptr = DLL_GetCurrentPointer(rtx->chunk_list);
        else
            chunk_ptr = NULL;

    }
    while (chunk_ptr != NULL);

    event_log(INTERNAL_EVENT_0, "Retransmission Queue has been reset...");
    /* FIXME : Update data belonging to the RTX-Instance ??? */
    chunk_list_debug(VVERBOSE, rtx->chunk_list);
    return;
}



/**
 * Function returns the number of chunks that are waiting in the queue to be acked
 * @return size of the retransmission queue
 */
unsigned int rtx_readNumberOfUnackedChunks()
{
    unsigned int queue_len;
    rtx_buffer *rtx;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return 0;
    }
    queue_len = DLL_GetNumberOfRecords(rtx->chunk_list);
    event_logi(VERBOSE, "rtx_readNuumberOfUnackedChunks() returns %u", queue_len);
    return queue_len;
}


/**
 * function to return the last a_rwnd value we got from our peer
 * @return  peers advertised receiver window
 */
unsigned int rtx_read_remote_receiver_window()
{
    rtx_buffer *rtx;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return 0;
    }
    event_logi(VERBOSE, "rtx_read_remote_receiver_window returns %u", rtx->peer_arwnd);
    return rtx->peer_arwnd;
}


/**
 * function to set the a_rwnd value when we got it from our peer
 * @param  new_arwnd      peers newly advertised receiver window
 * @return  0 for success, -1 for error
 */
int rtx_set_remote_receiver_window(unsigned int new_arwnd)
{
    rtx_buffer *rtx;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return -1;
    }
    event_logi(VERBOSE, "rtx_set_his_receiver_window(%u)", new_arwnd);
    rtx->peer_arwnd = new_arwnd;
    return 0;
}

/**
 * function that is called by SCTP-Control, when ULP requests
 * shutdown in an established association
 * @return  0 for success, -1 for error
 */
int rtx_shutdown()
{
    rtx_buffer *rtx;

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return -1;
    }
    event_log(VERBOSE, "rtx_shutdown() activated");
    rtx->shutdown_received = TRUE;
    event_log(VERBOSE, "calling fc_shutdown()");
    fc_shutdown();
    return 0;
}


/*
   CHECKME : Check retransmission procedures case when SHUTDOWN is initiated.
 */


/**
 * function that is called by SCTP-Control, when peer indicates
 * shutdown and sends us his last ctsna...this function dequeues
 * all chunks, and returns the number of chunks left in the queue
 * @param  ctsna    up to this tsn we can dequeue all chunks here
 * @return  number of chunks that are still queued
 */
unsigned int rtx_rcv_shutdown_ctsna(unsigned int ctsna)
{
    rtx_buffer *rtx;
    int result;

    event_logi(INTERNAL_EVENT_0, "rtx_rcv_shutdown_ctsna(ctsna==%u)", ctsna);

    rtx = (rtx_buffer *) mdi_readReliableTransfer();
    if (!rtx) {
        error_log(ERROR_MAJOR, "rtx_buffer instance not set !");
        return (0);
    }
    if (after(ctsna, rtx->lowest_tsn) || (ctsna == rtx->lowest_tsn)) {
        event_logiii(VVERBOSE,
                     "after(%u, %u) == true, call rtx_dequeue_up_to(%u)",
                     ctsna, rtx->lowest_tsn, ctsna);
        result = rtx_dequeue_up_to(ctsna);
        if (result < 0) {
            error_log(ERROR_MAJOR, "Bad ctsna arrived in shutdown  or no chunks in queue     ....");
        }
        rtx->lowest_tsn = ctsna;
        event_logi(VVERBOSE, "Updated rtx->lowest_tsn==ctsna==%u", ctsna);
    }
    if (rtx->shutdown_received == TRUE) {
        if (fc_readNumberOfQueuedChunks() == 0) {
            sci_allChunksAcked();
        }
    }
    return (DLL_GetNumberOfRecords(rtx->chunk_list));
}
