/*
 *  $Id: streamengine.c,v 1.9 2001/03/13 17:02:09 ajung Exp $
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 *
 * Purpose: This modules implements the interface defined in streamengine.h and 
 *          holds the private list of streams to process them for reasons of
 *          sending and receiving datachunks.
 *
 * TODO (and this is a big one !) :
 *   - use a new structure for chunks, that may be passed down to flowcontrol, bundling, reltransfer etc.
 *     without being copied over and over again ! This will require modification of these modules, too !
 *   - replace the ubiquitious  sizeof(SCTP_chunk_header) + sizeof(SCTP_data_chunk_header) by MACRO
 *   - use meanigful names in this module, and add some documentation. This is miserable.....
 *   - the superfluous use of memcpy() and malloc() should be cleaned up as well.
 *   - this module makes extensive use of lists, replace these by using glib-functions()
 */

#include "globals.h"
#include "dll_main.h"           /* doubly linked list library */
#include "errno.h"
#include "flowcontrol.h"
#include "streamengine.h"
#include "distribution.h"
#include "bundling.h"


/******************** Structure Definitions ****************************************/

/* Structure for lists used by each stream */
typedef struct
{
    List *nrDL;                 /* Doubly Linked List for numbered Chunks */
    List *uDL;                  /* Doubly Linked List for unnumbered Chunks */
    List *RDL;                  /* Doubly Linked List for receive Datagrams */
    unsigned int nextRsn;
    unsigned int nextSsn;
    unsigned int maxNrRStream;
    unsigned int maxNrSStream;
} Streaminfo;


/* TODO : also store protocol ID here, so that it can be delivered to the upper layer, too ! */
typedef struct
{
    SCTP_data_chunk_se *data;
    unsigned short length;
} Chunkinfo;



typedef struct
{
    unsigned char *data;
    unsigned int length;
} Datagraminfo;

/******************** Prototypes ****************************************/

void check_if_complete_nrDLiLi(unsigned short stream_id);
void check_if_complete_uDLiLi(unsigned short stream_id);
void unorderedChunk(SCTP_data_chunk_se * chunk, unsigned int byte_count, unsigned int protId);
int sort_nrchunk(void *arg1, void *arg2);
int sort_uchunk(void *arg1, void *arg2);
int equal_tsn_nrchunk(void *arg1, void *arg2);
int equal_tsn_uchunk(void *arg1, void *arg2);

/******************** Declarations *************************************************/

Streaminfo *streaminfolst;

  /* look def.dll_main.h with typedef SCTP_data_chunk_se */
List *nrDLiLi;                  /* Doubly Linked List for numbered Chunks */
List *uDLiLi;                   /* Doubly Linked List for unnumbered Chunks */
List *ulp_RDLiLi;               /* Doubly Linked List for receive Datagrams */



DLL_Return retvalue;

/******************** Function Definitions *****************************************/

/* This function is called to instanciate one Stream Engine for an association.
   It creates and initializes the Lists for Sending and Receiving Data. 
   It is called by Message Distribution.
   returns: the pointer to the Stream Engine 
*/

void *se_new_stream_engine(unsigned int nrRStr, /* max of streams to receive */
                           unsigned int nrSStr)
{                               /* max of streams to send */
    int i;

    event_logii(EXTERNAL_EVENT, "new_stream_engine: #inStreams=%d, #outStreams=%d", nrRStr, nrSStr);

    streaminfolst = (Streaminfo *) malloc(nrRStr * sizeof(Streaminfo));
    if (streaminfolst == NULL) {
        error_log_sys(ERROR_FATAL, errno);
        exit(1);
    }

/* initialize StreamInfo List for received Chunks */
/* and initiate for all streams with pointers to Chunk Lists */
/* initialize sequence numbers = 0 for send and receive chunks */
    for (i = 0; i < nrRStr; i++) {
        DLL_CreateList(&(streaminfolst[i].nrDL));
        DLL_CreateList(&(streaminfolst[i].uDL));
        DLL_CreateList(&(streaminfolst[i].RDL));

        DLL_InitializeList(streaminfolst[i].nrDL, sizeof(Chunkinfo)); /* Initialize for all streams
                                                                         the Listentry with NULL pointers */
        DLL_InitializeList(streaminfolst[i].uDL, sizeof(Chunkinfo));
        DLL_InitializeList(streaminfolst[i].RDL, sizeof(Datagraminfo));
        streaminfolst[i].nextRsn = 0;
        streaminfolst[i].nextSsn = 0;
        streaminfolst[i].maxNrRStream = nrRStr;
        streaminfolst[i].maxNrSStream = nrSStr;
    }

    return (&streaminfolst[0]);
}




/* Deletes the instance pointed to by streamengine.
*/
void se_delete_stream_engine(void *streaminfolst)
{
    event_log(EXTERNAL_EVENT, "delete streamengine");
    /* free complete  contents of all lists */
    free(streaminfolst);
}



void se_readNumberOfStreams(unsigned short *inStreams, unsigned short *outStreams)
{
    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log_sys(ERROR_MAJOR, errno);
        *inStreams = 0;
        *outStreams = 0;
        return;
    }
    *inStreams = streaminfolst[0].maxNrRStream;
    *outStreams = streaminfolst[0].maxNrSStream;
}



/******************** Functions for Sending *****************************************/

/**
 * This function is called to send a chunk.
 *  called from MessageDistribution
 * @return 0 for success, -1 for error (e.g. data sent in shutdown state etc.)
*/
int
se_ulpsend(unsigned short streamId,
           unsigned char *buffer,
           unsigned int byteCount,
           unsigned int protocolId, short destAddressIndex, void*  context,
           unsigned int lifetime, gboolean unorderedDelivery, /* optional (=FALSE if none) */
           gboolean  dontBundle)
{                               /* optional (=FALSE if none) */
    gint32 result = 0;
    guint32 state;
    SCTP_data_chunk_se s_chunk; /* Chunk to send */

    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log(ERROR_MAJOR, "StreamEngine Instance does not exist....Returning !");
        return -1;
    }
    if (streamId > streaminfolst[streamId].maxNrSStream) {
        error_log(ERROR_MINOR, "STREAM ID OVERFLOW in se_ulpsend()");
        mdi_sendFailureNotif(buffer, byteCount, context);
        return -1;
    }
    state = sci_getState();
    if (sci_shutdown_procedure_started() == TRUE) {
        event_logi(EXTERNAL_EVENT,
                   "Cannot send Chunk, state(==%u) already in SHUTDOWN-phase", state);
        return -1;
    }

    s_chunk.chunk_id = CHUNK_DATA;
    s_chunk.chunk_flags = SCTP_DATA_BEGIN_SEGMENT + SCTP_DATA_END_SEGMENT;
    s_chunk.chunk_length =
        htons(byteCount + sizeof(SCTP_data_chunk_header) + sizeof(SCTP_chunk_header));
    s_chunk.tsn = htonl(0);
    s_chunk.stream_id = htons(streamId);
    s_chunk.stream_sn = htons(streaminfolst[streamId].nextSsn);
    s_chunk.protocolId = htonl(protocolId);
    if (unorderedDelivery) {
        /* FIXME : should not StreamSequence Number be Zero in this case too ????  */
        s_chunk.chunk_flags = s_chunk.chunk_flags + SCTP_DATA_UNORDERED;
    } else {                    /* unordered flag not put */
        streaminfolst[streamId].nextSsn++; /* incr. for next send message */
        if (streaminfolst[streamId].nextSsn > 0xFFFF)
            streaminfolst[streamId].nextSsn = 0;
    }
    memcpy(s_chunk.data, buffer, byteCount);

    event_logi(EXTERNAL_EVENT, "=============> ulp sends a chunk (SSN=%u) to StreamEngine <=============",
               ntohs(s_chunk.stream_sn));

    result =
        fc_send_data_chunk((SCTP_data_chunk *) & s_chunk, destAddressIndex, lifetime, dontBundle, context);
    if (result != 0)
        error_logi(ERROR_MINOR, "se_ulpsend() failed with result %d", result);

    return result;
}


/******************** Functions for Receiving **************************************/

/**
 * This function is called from distribution layer to receive a chunk.
 */
short se_ulpreceive(unsigned char *buffer, unsigned int *byteCount, unsigned short streamId)
{
    Datagraminfo datagram;

    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log_sys(ERROR_MAJOR, errno);
        exit(1);
    }
    if (streamId > streaminfolst[streamId].maxNrRStream) {
        error_log(ERROR_MINOR, "STREAM ID OVERFLOW");
        return (STREAM_ID_OVERFLOW);
    } else {
        if (DLL_IsListEmpty(streaminfolst[streamId].RDL)) { /* =TRUE */
            event_log(EXTERNAL_EVENT, "NO DATA AVAILABLE");
            return (NO_DATA_AVAILABLE);
        } else {
            DLL_CurrentPointerToHead(streaminfolst[streamId].RDL);
            DLL_GetCurrentRecord(streaminfolst[streamId].RDL, &datagram);
            memcpy(buffer, datagram.data, datagram.length);
            *byteCount = datagram.length;
            event_logii(VVERBOSE, "se_ulpreceive(%u) : Freeing Address %x\n",
                        streamId, datagram.data);
            free(datagram.data);
            DLL_DeleteCurrentRecord(streaminfolst[streamId].RDL);
        }
    }
    event_log(EXTERNAL_EVENT, "ulp receives a datagram from se");
    return (RECEIVE_DATA);
}




/*
 * This function is called from Bundling to forward received chunks to Stream Engine.
 * TODO : cleanups, do error handling in proper errorhandler-module. This is messy !!! :-(
*/
void se_recvDataChunk(SCTP_data_chunk_se * dataChunk, unsigned int byteCount)
{
    struct error_chunk          /* Error Chunk, Invalid Stream Id */
    {
        unsigned char chunk_id;
        unsigned char chunk_flags;
        unsigned short chunk_length;
        unsigned short cause_code;
        unsigned short cause_length;
        unsigned short stream_id;
        unsigned short reserve;
    } e_chunk;
    /* Error Chunk invalid Stream Id */
    Chunkinfo ChunkInf;
    Chunkinfo ChunkInfFound;
    Datagraminfo Datagram;
    SCTP_data_chunk_se d_chunk;
    unsigned int nextRsn;
    unsigned int byte_count;

    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log_sys(ERROR_MAJOR, errno);
        exit(1);
    }

    byte_count = byteCount;
    memcpy(&d_chunk, dataChunk, byte_count); /* chunk in host BO */
    d_chunk.chunk_length = ntohs(dataChunk->chunk_length);
    d_chunk.tsn = ntohl(dataChunk->tsn);
    d_chunk.stream_id = ntohs(dataChunk->stream_id);
    d_chunk.stream_sn = ntohs(dataChunk->stream_sn);
    d_chunk.protocolId = ntohl(dataChunk->protocolId);

    if (d_chunk.stream_id > streaminfolst[d_chunk.stream_id].maxNrRStream) {
        /* Error Chunk -- CLEAN THIS UP !!!!!!!!!!!!!!!!!! ARGHHH */
        e_chunk.chunk_id = CHUNK_ERROR;
        e_chunk.chunk_flags = 0;
        e_chunk.chunk_length = htons(12);
        e_chunk.cause_code = htons(STREAM_ID_OVERFLOW);
        e_chunk.cause_length = htons(8);
        e_chunk.stream_id = htons(d_chunk.stream_id);
        e_chunk.reserve = htons(0);
        error_log(ERROR_MAJOR, "STREAM ID OVERFLOW");
        bu_put_Ctrl_Chunk((SCTP_simple_chunk *) & e_chunk);
        bu_sendAllChunks(NULL);

    } else {

        if (d_chunk.chunk_flags & SCTP_DATA_UNORDERED) {

            if (d_chunk.chunk_length > (sizeof(SCTP_chunk_header) + sizeof(SCTP_data_chunk_header))) {

                uDLiLi = streaminfolst[d_chunk.stream_id].uDL;
                ulp_RDLiLi = streaminfolst[d_chunk.stream_id].RDL;

                event_logiiii(EXTERNAL_EVENT,
                             "se got unnumbered chunk (tsn=%u, sid=%u, ssn=%u, prot ID=%u) from peer",
                             d_chunk.tsn, d_chunk.stream_id, d_chunk.stream_sn, d_chunk.protocolId);

                unorderedChunk(&d_chunk, byte_count, d_chunk.protocolId);
            } else {
                error_log(ERROR_MAJOR,
                          "Zero length or faulty unordered chunk - handler not yet implemented !");
                /* TODO : Zero length Chunk handling, or faulty chunk treatment */
            }
        } else {
            event_logiiii(EXTERNAL_EVENT,
                         "se got numbered chunk (tsn=%u, sid=%u, ssn=%u, protID=%u) from peer",
                         d_chunk.tsn, d_chunk.stream_id, d_chunk.stream_sn, d_chunk.protocolId);

            nrDLiLi = streaminfolst[d_chunk.stream_id].nrDL;
            ulp_RDLiLi = streaminfolst[d_chunk.stream_id].RDL;
            nextRsn = streaminfolst[d_chunk.stream_id].nextRsn;

            if (d_chunk.stream_sn == nextRsn) {
                if ((d_chunk.chunk_flags & SCTP_DATA_BEGIN_SEGMENT)
                    && (d_chunk.chunk_flags & SCTP_DATA_END_SEGMENT)) {
                    if (d_chunk.chunk_length >
                        (sizeof(SCTP_chunk_header) + sizeof(SCTP_data_chunk_header))) {
                        Datagram.length =
                            byte_count - sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
                        Datagram.data = (unsigned char *) malloc(Datagram.length);
                        event_logi(VVERBOSE,
                                   "se_recvDataChunk(1): Allocated Datagram address %x\n",
                                   Datagram.data);

                        if (Datagram.data == NULL)
                            error_log_sys(ERROR_FATAL, errno);
                        memcpy(Datagram.data, &(d_chunk.data), Datagram.length);

                        DLL_CurrentPointerToTail(ulp_RDLiLi);
                        retvalue = DLL_InsertRecord(ulp_RDLiLi, &Datagram, DLL_BELOW); /*add Datagram on tail */

                        if (retvalue == DLL_NORMAL)
                            mdi_dataArriveNotif(d_chunk.stream_id, Datagram.length,d_chunk.protocolId, FALSE);
                        else {
                            if (retvalue == DLL_MEM_ERROR)
                                DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
                            else {
                                if (retvalue == DLL_NOT_MODIFIED)
                                    DLL_error_log(ERROR_MAJOR, DLL_NOT_MODIFIED);
                            }
                        }
                    } else {
                        error_log(ERROR_MAJOR,
                                  "Zero length or faulty ordered chunk - handler not yet implemented !");
                        /* TODO : Zero length Chunk handling, or faulty chunk treatment */
                    }
                    streaminfolst[d_chunk.stream_id].nextRsn = (nextRsn + 1) % 0x10000;
                    check_if_complete_nrDLiLi(d_chunk.stream_id);

                } else {
                    /* stream_sn = nextRsn,  BEGIN or END or no Flag */
                    ChunkInf.data = (SCTP_data_chunk_se *) malloc(byte_count);
                    event_logi(VVERBOSE,
                               "se_recvDataChunk(2): Allocated ChunkInf address %x\n",
                               ChunkInf.data);

                    if (ChunkInf.data == NULL)
                        error_log_sys(ERROR_FATAL, errno);
                    memcpy(ChunkInf.data, &d_chunk, byte_count);

                    ChunkInf.length =
                        byte_count - sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);

                    retvalue =
                        DLL_FindRecord(nrDLiLi, &ChunkInfFound, &ChunkInf, equal_tsn_nrchunk);

                    if (retvalue != DLL_NORMAL) {
                        /* enter a Chunk in the numbered Chunklist, sorted via sn and tsn */

                        retvalue = DLL_CurrentPointerToHead(nrDLiLi);
                        retvalue = DLL_AddRecord(nrDLiLi, &ChunkInf, sort_nrchunk);
                        if (retvalue == DLL_NORMAL)
                            check_if_complete_nrDLiLi(d_chunk.stream_id);
                        else
                            DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
                    }
                    /* else {
                       memcpy(ChunkInfFound.data, ChunkInf.data, byte_count);  * new Chunk with equal tsn *
                       ChunkInfFound.length = ChunkInf.length;            * not inserted         *
                       }
                     */
                }

            } else {
                /* stream_sn != nextRsn */
                ChunkInf.data = (SCTP_data_chunk_se *) malloc(byte_count);
                event_logi(VVERBOSE,
                           "se_recvDataChunk(3): Allocated ChunkInf address %x\n", ChunkInf.data);

                if (ChunkInf.data == NULL)
                    error_log_sys(ERROR_FATAL, errno);

                memcpy(ChunkInf.data, &d_chunk, byte_count);
                ChunkInf.length =
                    byte_count - sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
                /* enter a Chunk in the numbered Chunklist, sorted via sn and tsn */
                retvalue = DLL_CurrentPointerToHead(nrDLiLi);
                retvalue = DLL_AddRecord(nrDLiLi, &ChunkInf, sort_nrchunk);

                if (retvalue == DLL_MEM_ERROR)
                    DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
            }

        }                       /* if !(d_chunk.chunk_flags & SCTP_DATA_UNORDERED) */

    }                           /* if !(d_chunk.stream_id > streaminfolst[d_chunk.stream_id].maxNrRStream)  */


}



/* Procedure to treat unordered Chunks
*/
void unorderedChunk(SCTP_data_chunk_se * chunk, unsigned int byte_count, unsigned int protId)
{
    Datagraminfo Datagram;
    Chunkinfo ChunkInf;
    Chunkinfo ChunkInfFound;

    if ((chunk->chunk_flags & SCTP_DATA_BEGIN_SEGMENT) &&
        (chunk->chunk_flags & SCTP_DATA_END_SEGMENT)) {
        Datagram.length = byte_count - sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
        Datagram.data = (unsigned char *) malloc(Datagram.length);
        event_logi(VVERBOSE, "unorderedChunk(1): Allocated Datagram address %x\n", Datagram.data);

        if (Datagram.data == NULL)
            error_log_sys(ERROR_FATAL, errno);
        memcpy(Datagram.data, chunk->data, Datagram.length); /* copy Chunk Data */
        DLL_CurrentPointerToHead(ulp_RDLiLi);
        retvalue = DLL_InsertRecord(ulp_RDLiLi, &Datagram, DLL_ABOVE); /* add Datagram on head */
        if (retvalue == DLL_NORMAL)
            mdi_dataArriveNotif(chunk->stream_id, Datagram.length, protId, TRUE);
        else {
            if (retvalue == DLL_MEM_ERROR)
                DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
            else {
                if (retvalue == DLL_NOT_MODIFIED)
                    DLL_error_log(ERROR_MAJOR, DLL_NOT_MODIFIED);
            }
        }
    } else {                    /* BEGIN or END or no Flag */
        ChunkInf.data = (SCTP_data_chunk_se *) malloc(byte_count);
        event_logi(VVERBOSE, "unorderedChunk(2): Allocated ChunkInf address %x\n", ChunkInf.data);

        if (ChunkInf.data == NULL)
            error_log_sys(ERROR_FATAL, errno);
        memcpy(ChunkInf.data, chunk, byte_count); /* copy Chunk data */
        ChunkInf.length = byte_count;

        retvalue = DLL_FindRecord(uDLiLi, &ChunkInfFound, &ChunkInf, equal_tsn_uchunk);
        if (retvalue != DLL_NORMAL) {
            DLL_CurrentPointerToHead(uDLiLi); /* tsn not equal, insert chunk */
            retvalue = DLL_AddRecord(uDLiLi, &ChunkInf, sort_uchunk);
            if (retvalue == DLL_NORMAL)
                check_if_complete_uDLiLi(chunk->stream_id);
            else
                DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
        }
        /* else
           {  memcpy(ChunkInfFound.data, ChunkInf.data, byte_count);    * new Chunk with equal tsn *
           ChunkInfFound.length = ChunkInf.length;          * not inserted         *
           }
         */
    }
}




/* sort function for DLL_AddRecord */
int sort_nrchunk(void *arg1, void *arg2)
{
    Chunkinfo *nrDLiLi;
    Chunkinfo *ChunkInf;

    nrDLiLi = (Chunkinfo *) arg1;
    ChunkInf = (Chunkinfo *) arg2;

    if (nrDLiLi->data->stream_sn > ChunkInf->data->stream_sn)
        return 1;
    else {
        if (nrDLiLi->data->stream_sn < ChunkInf->data->stream_sn)
            return -1;
        else {
            if (nrDLiLi->data->tsn > ChunkInf->data->tsn)
                return 1;
            else {
                if (nrDLiLi->data->tsn < ChunkInf->data->tsn)
                    return -1;
                else
                    return 0;
            }
        }
    }
}



/* sort function for DLL_AddRecord */
int sort_uchunk(void *arg1, void *arg2)
{
    Chunkinfo *uDLiLi;
    Chunkinfo *ChunkInf;

    uDLiLi = (Chunkinfo *) arg1;
    ChunkInf = (Chunkinfo *) arg2;

    if (uDLiLi->data->tsn > ChunkInf->data->tsn)
        return 1;
    else {
        if (uDLiLi->data->tsn < ChunkInf->data->tsn)
            return -1;
        else
            return 0;
    }
}



/* find function for DLL_FindRecord */
int equal_tsn_uchunk(void *arg1, void *arg2)
{
    Chunkinfo *uDLiLi;
    Chunkinfo *ChunkInf;

    uDLiLi = (Chunkinfo *) arg1;
    ChunkInf = (Chunkinfo *) arg2;

    if (uDLiLi->data->tsn > ChunkInf->data->tsn)
        return 1;
    else {
        if (uDLiLi->data->tsn < ChunkInf->data->tsn)
            return -1;
        else
            return 0;
    }
}




/* find function for DLL_FindRecord */
int equal_tsn_nrchunk(void *arg1, void *arg2)
{
    Chunkinfo *nrDLiLi;
    Chunkinfo *ChunkInf;

    nrDLiLi = (Chunkinfo *) arg1;
    ChunkInf = (Chunkinfo *) arg2;

    if (nrDLiLi->data->tsn > ChunkInf->data->tsn)
        return 1;
    else {
        if (nrDLiLi->data->tsn < ChunkInf->data->tsn)
            return -1;
        else
            return 0;
    }
}





/* Procedure to check the element with stream_sn = nextRsn in nrDLiLi if complete
   and if so, unqueue and enter in ulp_RDLiLi
*/
void check_if_complete_nrDLiLi(unsigned short stream_id)
{
    boolean endflag;
    boolean beginflag;
    unsigned char flags;
    unsigned int nr_ofCtoDatagr;
    unsigned int length_ofDatagr;
    unsigned int last_tsn;
    Chunkinfo nrChunk;
    Chunkinfo *debug_chunkptr;
    Datagraminfo Datagram;
    unsigned int nextRsn;
    int i;
    int l;

    nr_ofCtoDatagr = 0;
    length_ofDatagr = 0;
    event_log(INTERNAL_EVENT_0, "check if complete numbered Chunk List");
    nextRsn = streaminfolst[stream_id].nextRsn;
    retvalue = DLL_CurrentPointerToHead(nrDLiLi);
    if (retvalue == DLL_NULL_LIST)
        return;
    DLL_GetCurrentRecord(nrDLiLi, &nrChunk);
    if (nrChunk.data->stream_sn == nextRsn) {
        flags = nrChunk.data->chunk_flags;
        if ((flags & SCTP_DATA_BEGIN_SEGMENT)
            && (flags & SCTP_DATA_END_SEGMENT)) {
            if (nrChunk.data->chunk_length >
                (sizeof(SCTP_chunk_header) + sizeof(SCTP_data_chunk_header))) {
                beginflag = TRUE;
                endflag = TRUE;
                Datagram.length =
                    length_ofDatagr + nrChunk.data->chunk_length -
                    sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
                Datagram.data = (unsigned char *) malloc(Datagram.length);
                event_logii(VVERBOSE,
                            "check_if_complete_nrDLiLi (%u)- 1: Alloced Datagram address %x\n",
                            stream_id, Datagram.data);

                if (Datagram.data == NULL)
                    error_log_sys(ERROR_FATAL, errno);

                memcpy(Datagram.data, nrChunk.data->data, nrChunk.length);
                DLL_CurrentPointerToTail(ulp_RDLiLi);
                retvalue = DLL_InsertRecord(ulp_RDLiLi, &Datagram, DLL_BELOW); /*add Datagram on tail */
                if (retvalue == DLL_NORMAL) {
                    DLL_CurrentPointerToHead(nrDLiLi);
                    debug_chunkptr = DLL_GetCurrentPointer(nrDLiLi);
                    event_logii(VVERBOSE,
                                "check_if_complete_nrDLiLi(%u) - 3 : DELETING Address %x\n",
                                stream_id, debug_chunkptr->data);

                    DLL_DeleteCurrentRecord(nrDLiLi);
                    /* TODO : check if free() is needed */
                    streaminfolst[stream_id].nextRsn = (nextRsn + 1) % 0x10000;
                    /* not entirely sure about the length here */
                    mdi_dataArriveNotif(stream_id, nrChunk.length, nrChunk.data->protocolId, FALSE);
                } else {
                    if (retvalue == DLL_MEM_ERROR)
                        DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
                    else {
                        if (retvalue == DLL_NOT_MODIFIED)
                            DLL_error_log(ERROR_MAJOR, DLL_NOT_MODIFIED);
                    }
                }
            }
            streaminfolst[stream_id].nextRsn = (nextRsn + 1) % 0x10000;
            check_if_complete_nrDLiLi(nrChunk.data->stream_id);
        } else {
            if (flags & SCTP_DATA_BEGIN_SEGMENT) {
                beginflag = TRUE;
                endflag = FALSE;
                last_tsn = nrChunk.data->tsn; /* first element */
                nr_ofCtoDatagr = 1;
                length_ofDatagr =
                    nrChunk.data->chunk_length - sizeof(SCTP_chunk_header) -
                    sizeof(SCTP_data_chunk_header);
                while ((retvalue == DLL_NORMAL) && (endflag == FALSE)) {
                    /* end of the loop:  last element of the list reached, !=tsn+1, !=nextRsn */
                    retvalue = DLL_GetNextRecord(nrDLiLi, &nrChunk); /* next element */
                    flags = nrChunk.data->chunk_flags;
                    if (nrChunk.data->stream_sn == nextRsn) {
                        if (nrChunk.data->tsn == last_tsn + 1) {
                            last_tsn++;
                            nr_ofCtoDatagr++;
                            length_ofDatagr =
                                length_ofDatagr +
                                nrChunk.data->chunk_length -
                                sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
                            if (flags & SCTP_DATA_END_SEGMENT) {
                                endflag = TRUE;
                                Datagram.length = length_ofDatagr;
                                if (Datagram.length > 0) {
                                    Datagram.data = (unsigned char *)
                                        malloc(Datagram.length);
                                    event_logiii(VVERBOSE,
                                                 "check_if_complete_nrDLiLi (%u)-2 : Alloced Datagram address %x (%u Bytes)\n",
                                                 stream_id, Datagram.data, Datagram.length);

                                    if (Datagram.data == NULL)
                                        error_log_sys(ERROR_FATAL, errno);

                                    DLL_CurrentPointerToHead(nrDLiLi);
                                    DLL_GetCurrentRecord(nrDLiLi, &nrChunk);
                                    l = 0;
                                    for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of chunks */
                                        memcpy(Datagram.data
                                               + l, nrChunk.data->data, nrChunk.length);
                                        l = l + nrChunk.length;
                                        DLL_GetNextRecord(nrDLiLi, &nrChunk);
                                    }
                                    /* Datagram complete,  copy numbered chunks to tail of the list */
                                    event_log(INTERNAL_EVENT_0, "Datagram complete");
                                    DLL_CurrentPointerToTail(ulp_RDLiLi);
                                    retvalue = DLL_InsertRecord(ulp_RDLiLi, &Datagram, DLL_BELOW); /* add Datagram on tail */
                                    if (retvalue == DLL_NORMAL) {
                                        DLL_CurrentPointerToHead(nrDLiLi);
                                        for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of chunks transferred */
                                            debug_chunkptr = DLL_GetCurrentPointer(nrDLiLi);
                                            event_logii
                                                (VVERBOSE,
                                                 "check_if_complete_nrDLiLi(%u) - 4 : DELETING Address %x\n",
                                                 stream_id, debug_chunkptr->data);

                                            DLL_DeleteCurrentRecord(nrDLiLi);
                                            /* TODO : check if free() is needed */
                                        }
                                        /* HOW very original, to call this variale "l" !!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
                                        mdi_dataArriveNotif(stream_id, l,nrChunk.data->protocolId, FALSE);
                                    } else { /* break if DLL error */
                                        if (retvalue == DLL_MEM_ERROR)
                                            DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
                                        else {
                                            if (retvalue == DLL_NOT_MODIFIED)
                                                DLL_error_log(ERROR_MAJOR, DLL_NOT_MODIFIED);
                                        }
                                        break;
                                    }
                                } else { /* length of Datagram = 0 */
                                    DLL_CurrentPointerToHead(nrDLiLi);
                                    for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of chunks transferred */
                                        debug_chunkptr = DLL_GetCurrentPointer(nrDLiLi);
                                        event_logii
                                            (VVERBOSE,
                                             "check_if_complete_nrDLiLi(%u) - 5 : DELETING Address %x\n",
                                             stream_id, debug_chunkptr->data);
                                        DLL_DeleteCurrentRecord(nrDLiLi);
                                        /* TODO : check if free() is needed */
                                    }
                                }
                                streaminfolst[stream_id].nextRsn = (nextRsn + 1)% 0x10000;

                            }
                            /* else: not END flag */

/* CLEAN UP THIS MESS -- This is UGLY -- I don't want this. I want *break* only in switch statements */

                        } else  /* not tsn+1 */
                            break;
                    } else      /* not next_Rsn */
                        break;
                }               /* end of while !endflag */
                endflag = FALSE;

            }
            /* else: not BEGIN flag */

        }                       /*not BEGIN- + END-Flag */

    }
    /* else: not next_Rsn */
}







/* Procedure to check the new element in uDLiLi if complete
   and if so, unqueue (BEGIN to END) and enter in ulp_RDLiLi
*/
void check_if_complete_uDLiLi(unsigned short stream_id)
{
    boolean endflag;
    boolean beginflag;
    unsigned char flags;
    unsigned int nr_ofCtoDatagr;
    unsigned int length_ofDatagr;
    unsigned int last_tsn;
    Chunkinfo uChunk;
    Chunkinfo *debug_chunkptr;
    Datagraminfo Datagram;
    int i;
    int l;


    nr_ofCtoDatagr = 0;
    length_ofDatagr = 0;

    event_log(INTERNAL_EVENT_0, "check if complete unnumbered Chunk List");
    retvalue = DLL_CurrentPointerToHead(uDLiLi);
    DLL_GetCurrentRecord(uDLiLi, &uChunk);
    flags = uChunk.data->chunk_flags;
    while (retvalue == DLL_NORMAL) { /* so long the list is not empty */

        if (flags & SCTP_DATA_BEGIN_SEGMENT) {
            beginflag = TRUE;
            endflag = FALSE;
            DLL_StoreCurrentPointer(uDLiLi); /* pointer and index of BEGIN chunk */

            last_tsn = uChunk.data->tsn; /* first element */
            nr_ofCtoDatagr = 1; /* Nr of Chunks to Datagram */
            length_ofDatagr =
                uChunk.data->chunk_length - sizeof(SCTP_chunk_header) -
                sizeof(SCTP_data_chunk_header);
            while ((retvalue == DLL_NORMAL) & !endflag)
                /* end of the loop:  last element of the list reached or !=tsn+1 or 
                   a datagram complete */
            {
                retvalue = DLL_GetNextRecord(uDLiLi, &uChunk); /* next element */
                flags = uChunk.data->chunk_flags;
                if (uChunk.data->tsn == last_tsn + 1) {
                    last_tsn++;
                    nr_ofCtoDatagr++;
                    length_ofDatagr =
                        length_ofDatagr + uChunk.data->chunk_length -
                        sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header);
                    if (flags & SCTP_DATA_END_SEGMENT) {
                        endflag = TRUE;
                        Datagram.length = length_ofDatagr;
                        if (Datagram.length > 0) {
                            Datagram.data = (unsigned char *)
                                malloc(Datagram.length);
                            event_logii(VVERBOSE,
                                        "check_if_complete_uDLiLi(%u)-1 : Alloced Datagram address %x\n",
                                        stream_id, Datagram.data);

                            if (Datagram.data == NULL)
                                error_log_sys(ERROR_FATAL, errno);
                            /* go to the BEGIN chunk to fetch it */
                            retvalue = DLL_RestoreCurrentPointer(uDLiLi);

                            DLL_GetCurrentRecord(uDLiLi, &uChunk);
                            l = 0;
                            for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of chunks */
                                memcpy(Datagram.data + l, uChunk.data->data, uChunk.length);
                                l = l + uChunk.length;
                                DLL_GetNextRecord(uDLiLi, &uChunk);
                            }
                            /* Datagram complete,   copy numbered chunks to head of the list */
                            event_log(INTERNAL_EVENT_0, "Datagram complete");
                            DLL_CurrentPointerToHead(ulp_RDLiLi);
                            retvalue = DLL_InsertRecord(ulp_RDLiLi, &Datagram, DLL_ABOVE); /* add Datagram on head */
                            if (retvalue == DLL_NORMAL) {
                                retvalue = DLL_RestoreCurrentPointer(uDLiLi);
                                for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of transferred chunks */
                                    /* TODO : check if free() is needed */
                                    debug_chunkptr = DLL_GetCurrentPointer(uDLiLi);
                                    event_logii(VVERBOSE,
                                                "check_if_complete_uDLiLi(%u) - 2 : DELETING Address %x\n",
                                                stream_id, debug_chunkptr->data);
                                    DLL_DeleteCurrentRecord(uDLiLi); /* delete the transferred chunks */
                                }

                                /* How *very* original, to call this variale "l" !!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */
                                mdi_dataArriveNotif(stream_id, l, uChunk.data->protocolId, TRUE);
                            } else {
                                if (retvalue == DLL_MEM_ERROR)
                                    DLL_error_log(ERROR_MAJOR, DLL_MEM_ERROR);
                                else {
                                    if (retvalue == DLL_NOT_MODIFIED)
                                        DLL_error_log(ERROR_MAJOR, DLL_NOT_MODIFIED);
                                }
                                break;
                            }

                        } else { /* length of Datagram = 0 */
                            retvalue = DLL_RestoreCurrentPointer(uDLiLi);
                            for (i = 0; i < nr_ofCtoDatagr; i++) { /* i = nr of transferred chunks */
                                /* TODO : check if free() is needed */
                                DLL_DeleteCurrentRecord(uDLiLi); /* delete the transferred chunks */
                            }
                        }
                    }
                    /* else:  not END flag */

                } else          /* else:  tsn != last_tsn + 1 */
                    break;

            }                   /* end of while !endflag */

        } else {                /* not BEGIN flag */
            retvalue = DLL_GetNextRecord(uDLiLi, &uChunk);
            flags = uChunk.data->chunk_flags;
        }

    }                           /* end of while list is not empty */

}

/**
 * function to return the number of chunks that can be retrieved
 * by the ULP - this function may need to be refined !!!!!!
 */
guint32 se_numOfQueuedChunks()
{
    guint32 i, number_of_recv_streams, num_of_chunks;
    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log(ERROR_MAJOR, "Could not read StreamEngine Instance !");
        streaminfolst = NULL;
        return 0xFFFFFFFF;
    }

    number_of_recv_streams = streaminfolst[0].maxNrRStream;
    num_of_chunks = 0;

    for (i = 0; i < number_of_recv_streams; i++) {
        /* Add number of all chunks (i.e. lengths of all RDL lists of all streams */
        num_of_chunks += DLL_GetNumberOfRecords(streaminfolst[i].RDL);
    }
    streaminfolst = NULL;
    return num_of_chunks;
}



/**
 * function to return the number of streams that we may
 * send on
 */
guint16 se_numOfSendStreams()
{
    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log(ERROR_MAJOR, "Could not read StreamEngine Instance !");
        streaminfolst = NULL;
        return 0;
    }
    return (guint16) (streaminfolst[0].maxNrSStream);

}

/**
 * function to return the number of streams that we are allowed to
 * receive data on
 */
guint16 se_numOfRecvStreams()
{
    streaminfolst = (Streaminfo *) mdi_readStreamEngine();
    if (streaminfolst == NULL) {
        error_log(ERROR_MAJOR, "Could not read StreamEngine Instance !");
        streaminfolst = NULL;
        return 0;
    }

    return (guint16) (streaminfolst[0].maxNrRStream);

}
