/*
 *  $Id: timer_list.c,v 1.2 2001/02/23 18:21:53 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * module : timer_list implements a linked list of timer events
 *
 */

#include "timer_list.h"

#include <stdio.h>
#include <stdlib.h>

static unsigned int tid = 1;
static List *timer_list;


/**
 *	function to initialize a list. Creates a timer_list structure
 *	@param	new_list 	pointer to newly alloc'ed list structure
 */
void init_timer_list()
{
    DLL_Return ExitCode;
    if (DLL_CreateList(&timer_list) == NULL)
        error_log(ERROR_FATAL, "List Create Failed \n");

    if ((ExitCode = DLL_InitializeList(timer_list, sizeof(AlarmTimer)))
        != DLL_NORMAL) {
        if (ExitCode == DLL_ZERO_INFO)
            error_log(ERROR_MINOR, "Size of AlarmTimer record is zero.\n\n");
        if (ExitCode == DLL_NULL_LIST)
            error_log(ERROR_MINOR, "timer_list points to a NULL.\n\n");
        exit(EXIT_FAILURE);
    }
}



int timercompare(AlarmTimer * one, AlarmTimer * two)
{
    if (timercmp(&(one->action_time), &(two->action_time), ==))
        return 0;
    else if timercmp
        (&(one->action_time), &(two->action_time), <)return -1;
    else if timercmp
        (&(one->action_time), &(two->action_time), >)return 1;
    error_log(ERROR_FATAL, "No valid result in  timercompare !\n");
    return 0;
}

int idcompare(AlarmTimer * one, AlarmTimer * two)
{
    if (one->timer_id == two->timer_id)
        return 0;
    else if (one->timer_id < two->timer_id)
        return -1;
    else if (one->timer_id > two->timer_id)
        return 1;
    error_log(ERROR_FATAL, "No valid result in  idcompare !\n");
    return 0;
}




/**
 *	function to delete a list. Walks through the list and deallocates
 *	all timer_item structs. Finally destroys the timer_list struct
 *	@param	del_list	pointer to the timer_list struct to be deleted
 *	@return	0 on success, -1 if pointer was NULL or other error
 */
void del_timer_list(void)
{
    DLL_DestroyList(&timer_list);
}

/**
 *	this function inserts a timer_item into the list. Keeps it ordered,
 *	and updates length, and possibly one timeval entry. Checks whether
 *	we insert at beginning/end first. timer_item must have been alloc'ed
 *	first by the application, this is not done by this function !
 *	@param	tlist		pointer to the timer_list instance
 *	@param	item	pointer to the event item that is to be added
 *	@return	timer_id on success, 0 if a pointer was NULL or other error
 */
unsigned int insert_item(AlarmTimer * item)
{
    int (*pfun) () = NULL;
    DLL_Return result;
    DLL_SetSearchModes(timer_list, DLL_HEAD, DLL_DOWN);
    item->timer_id = tid++;
    if (item->timer_id == 0) {
        tid++;
        item->timer_id = 1;
    }
    pfun = timercompare;
    result = DLL_AddRecord(timer_list, item, pfun);
    if (result == DLL_NORMAL)
        return item->timer_id;
    error_log(ERROR_FATAL, "Memory allocation error in insert_item\n");
    return 0;
}

/**
 *	a function to remove a certain action item,
 *	then traverses the list from the start, updates length etc.
 *	@param	tlist		pointer to the timer_list instance
 *	@param	tid	id of the timer to be removed
 *	@param	item	pointer to where deleted data is to be copied !
 *	@return	0 on success, -1 if a pointer was NULL or other error, -1 if not found
 */
int remove_item(unsigned int id, AlarmTimer * item)
{
    int (*pfun) () = NULL;
    DLL_Return result;
    AlarmTimer tmp;
    tmp.timer_id = id;
    pfun = idcompare;
    result = DLL_FindRecord(timer_list, item, &tmp, pfun);
    switch (result) {
    case DLL_NOT_FOUND:
        return -1;
        break;
    case DLL_NULL_LIST:
        return -1;
        break;
    case DLL_NORMAL:
        result = DLL_DeleteCurrentRecord(timer_list);
        break;
    default:
        break;
    }
    if (result == DLL_NORMAL)
        return 0;
    return -1;
}



/**
 *	a function to get the pointer to a certain action item, traverses the list
 *    copies the item into the provided pointer (reserve enough space !!)
 *	@param	tlist		pointer to the timer_list instance
 *	@param	timer_id	id of the timer to be found
 *	@param	item	pointer to where found data is to be copied !
 *	@return	0 on success, -1 if a pointer was NULL or other error
 */
int get_item(unsigned int id, AlarmTimer * item)
{
    int (*pfun) () = NULL;
    DLL_Return result;
    AlarmTimer tmp;
    tmp.timer_id = id;
    pfun = idcompare;
    result = DLL_FindRecord(timer_list, item, &tmp, pfun);
    switch (result) {
    case DLL_NULL_LIST:
        return -1;
        break;
    case DLL_NORMAL:
        return 0;
        break;
    default:
        break;
    }
    error_log(ERROR_FATAL, "No valid result in  get_item !\n");
    return -1;

}

/**
 *      function to be called, when a timer is reset. Basically calls get_item(),
 *    saves the function pointer, updates the execution time (msecs milliseconds
 *      from now) and removes the item from the list. Then calls insert_item
 *      with the updated timer_item struct.
 *      @param  tlist           pointer to the timer_list instance
 *      @param  id                      id of the timer to be updated
 *      @param  msecs           action to be executed msecs ms from _now_
 *      @return new timer_id, 0 if a pointer was NULL or other error
 */
unsigned int update_item(unsigned int id, unsigned int msecs)
{
    AlarmTimer tmp_item;
    int result;
    if (timer_list == NULL)
        return 0;
    result = remove_item(id, &tmp_item);
    if (result != 0)
        return 0;               /* i.e. timer not in list anymore */
    /* update action time, and  write back to the list */
#ifdef HAVE_GETTIMEOFDAY
    gettimeofday(&(tmp_item.action_time), NULL);
#endif
    adl_add_msecs_totime(&(tmp_item.action_time), msecs);
    return (insert_item(&tmp_item));
}



void print_item_info(AlarmTimer * item)
{
    event_logii(INTERNAL_EVENT_0, "TimerID: %d, Type : %d", item->timer_id, item->timer_type);
    event_logii(INTERNAL_EVENT_0, "action_time: %ld sec, %ld usec\n",
                item->action_time.tv_sec, item->action_time.tv_usec);
}

void print_debug_list(short event_log_level)
{
    AlarmTimer tmp;
    if (event_log_level <= Current_event_log_) {
        printf("-------------Entering print_debug_list() ------------------------\n");
        if (timer_list == NULL) {
            printf("tlist pointer == NULL\n");
            return;
        }

        if (DLL_CurrentPointerToHead(timer_list) != DLL_NORMAL) {
            printf("Timer-List is empty !\n");
            return;
        }
        print_time();
        printf("List Length : %ld \n", DLL_GetNumberOfRecords(timer_list));

        do {
            DLL_GetCurrentRecord(timer_list, &tmp);
            print_item_info(&tmp);
        }
        while (DLL_IncrementCurrentPointer(timer_list) == DLL_NORMAL);

        printf("-------------Leaving print_debug_list() ------------------------\n");
    }
    return;
}

 /**
 * the semantics of this function :
 * @return -1 if no timer in list, 0 if timeout and action must be taken, else time to
            next eventin milliseconds....
 */
int get_msecs_to_nexttimer()
{
    long secs, usecs;
    DLL_Return result;
    int msecs;
    AlarmTimer next;
    struct timeval now;

    if (DLL_IsListEmpty(timer_list) == DLL_TRUE)
        return -1;
#ifdef HAVE_GETTIMEOFDAY
    gettimeofday(&now, NULL);
#endif
    result = DLL_CurrentPointerToHead(timer_list);
    result = DLL_GetCurrentRecord(timer_list, &next);
    secs = next.action_time.tv_sec - now.tv_sec;
    usecs = next.action_time.tv_usec - now.tv_usec;
    if (secs < 0)
        return 0;
    if (usecs < 0) {
        secs--;
        usecs += 1000000;
    }
    if (secs < 0)
        return 0;
    /* here we will be cutting of the rest..... */
    msecs = (int) (1000 * secs + usecs / 1000);
    return (msecs);
}

int get_next_event(AlarmTimer * dest)
{
    DLL_Return result;
    if (DLL_IsListEmpty(timer_list) == DLL_TRUE)
        return -1;
    result = DLL_CurrentPointerToHead(timer_list);
    result = DLL_GetCurrentRecord(timer_list, dest);
    return (result == DLL_NORMAL);
}

int timer_list_empty()
{
    if (DLL_IsListEmpty(timer_list) == DLL_TRUE)
        return 1;
    else
        return 0;
}
