/*
 *  $Id: terminal.c,v 1.2 2001/03/16 17:49:44 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * terminal.c  - main program module
 *
 */

#include "sctp.h"

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>         /* for atoi() under Linux */

#define DISCARD_PORT                          9
#define MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES    10
#define MAXIMUM_NUMBER_OF_IN_STREAMS          1
#define MAXIMUM_NUMBER_OF_OUT_STREAMS         1 
#define SCTP_GENERIC_PAYLOAD_PROTOCOL_ID      0
#define min(x,y)            (x)<(y)?(x):(y)

static unsigned char localAddressList[MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES][SCTP_MAX_IP_LEN];
static unsigned char destinationAddress[SCTP_MAX_IP_LEN];

static unsigned short noOfLocalAddresses = 0;

static unsigned short remotePort = DISCARD_PORT;
static unsigned short localPort  = 1000;
static unsigned char  tosByte    = 0x10;  /* IPTOS_LOWDELAY */
static unsigned int associationID;

static int verbose  = 0;
static int vverbose = 0;

void getArgs(int argc, char **argv)
{
    int c;
    extern char *optarg;
    extern int optind;

    while ((c = getopt(argc, argv, "d:r:s:t:vV")) != -1)
    {
        switch (c) {
        case 'd':
            if (strlen(optarg) < SCTP_MAX_IP_LEN) {
                strcpy(destinationAddress, optarg);
            }
            break;
        case 'l':
            localPort = atoi(optarg);
            break;
        case 'r':
            remotePort = atoi(optarg);
            break;
        case 's':
            if ((noOfLocalAddresses < MAXIMUM_NUMBER_OF_LOCAL_ADDRESSES) &&
                (strlen(optarg) < SCTP_MAX_IP_LEN  )) {
                strcpy(localAddressList[noOfLocalAddresses], optarg);
                noOfLocalAddresses++;
            }
            break;
        case 't':
            tosByte = (unsigned char) atoi(optarg);
            break;
        case 'v':
            verbose = 1;
            break;
        case 'V':
            verbose = 1;
            vverbose = 1;
            break;
        }
    }
}

void dataArriveNotif(unsigned int assocID, unsigned int streamID, unsigned int len,
                     unsigned int protoID, unsigned int unordered, void* ulpDataPtr)
{
    unsigned char chunk[SCTP_MAXIMUM_DATA_LENGTH];
    int length;
 
    if (vverbose) {  
      fprintf(stdout, "%-8x: Data arrived (%u bytes on stream %u, %s)\n",
                      assocID, len, streamID, (unordered==SCTP_ORDERED_DELIVERY)?"ordered":"unordered");
      fflush(stdout);
    }
    /* read it */
    length = sizeof(chunk);
    sctp_receive(assocID, streamID, chunk, &length);
}

void sendFailureNotif(unsigned int assocID,
                      unsigned char *unsent_data, unsigned int dataLength, unsigned int *context, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Send failure\n", assocID);
    fflush(stdout);
  }
}

void networkStatusChangeNotif(unsigned int assocID, short destAddrIndex, unsigned short newState, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    SCTP_PathStatus pathStatus;
    unsigned short pathID;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Network status change: path %u is now %s\n", 
        assocID, destAddrIndex, ((newState == SCTP_PATH_OK) ? "ACTIVE" : "INACTIVE"));
        fflush(stdout);
    }
    
    /* if the primary path has become inactive */
    if ((newState == SCTP_PATH_UNREACHABLE) &&
        (destAddrIndex == sctp_getPrimary(assocID))) {
        
        /* select a new one */ 
        sctp_getAssocStatus(assocID, &assocStatus);
        for (pathID=0; pathID < assocStatus.numberOfAddresses; pathID++){
            sctp_getPathStatus(assocID, pathID, &pathStatus);
            if (pathStatus.state == SCTP_PATH_OK)
                break;
        }
        
        /* and use it */
        if (pathID < assocStatus.numberOfAddresses) {
            sctp_setPrimary(assocID, pathID);
        }
    }
}

void* communicationUpNotif(unsigned int assocID, unsigned short status,
                           int noOfDestinations,
                           unsigned short noOfInStreams, unsigned short noOfOutStreams, void* dummy)
{	
    if (verbose) {  
        fprintf(stdout, "%-8x: Communication up (%u paths)\n", assocID, noOfDestinations);
        fflush(stdout);
    }
    return NULL;
}

void communicationLostNotif(unsigned int assocID, unsigned short status, void* ulpDataPtr)
{	
    unsigned char buffer[SCTP_MAXIMUM_DATA_LENGTH];
    unsigned int bufferLength;
    unsigned short streamID, streamSN;
    unsigned int protoID;
    
    if (verbose) {
        fprintf(stdout, "%-8x: Communication lost (status %u)\n", assocID, status);
        fflush(stdout);
    }
    return;
    
    /* retrieve data */
    bufferLength = sizeof(buffer);
    while (sctp_receiveUnsent(assocID, buffer, &bufferLength,           
                              &streamID, &streamSN, &protoID) > 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
    
    bufferLength = sizeof(buffer);
    while (sctp_receiveUnacked(assocID, buffer, &bufferLength,
			                   &streamID, &streamSN, &protoID) > 0){
        /* do something with the retrieved data */
        /* after that, reset bufferLength */
        bufferLength = sizeof(buffer);
    }
                      
    /* delete the association */
    sctp_deleteAssociation(assocID);
}

void communicationErrorNotif(unsigned int assocID, unsigned short status, void* dummy)
{
  if (verbose) {  
    fprintf(stdout, "%-8x: Communication error (status %u)\n", assocID, status);
    fflush(stdout);
  }
}

void restartNotif(unsigned int assocID, void* ulpDataPtr)
{
    SCTP_AssociationStatus assocStatus;
    
    if (verbose) {  
        fprintf(stdout, "%-8x: Restart\n", assocID);
        fflush(stdout);
    }
    sctp_getAssocStatus(assocID, &assocStatus);
}

void shutdownCompleteNotif(unsigned int assocID, void* ulpDataPtr)
{
    if (verbose) {  
        fprintf(stdout, "%-8x: Shutdown complete\n", assocID);
        fflush(stdout);
    }
    
    /* delete the association */
    sctp_deleteAssociation(assocID);
}

void stdinCallback()
{
    unsigned char readBuffer[256];
    
    if (fgets(readBuffer, sizeof(readBuffer), stdin)==NULL) {
        sctp_shutdown(associationID);
    } else {
        sctp_send(associationID,
                  0,
                  readBuffer, strlen(readBuffer),
                  SCTP_GENERIC_PAYLOAD_PROTOCOL_ID,
                  SCTP_USE_PRIMARY, SCTP_NO_CONTEXT, 
                  SCTP_INFINITE_LIFETIME, SCTP_ORDERED_DELIVERY, SCTP_BUNDLING_DISABLED);
    }
}

int main(int argc, char **argv)
{
    unsigned short sctpInstance;
    SCTP_ulpCallbacks terminalUlp;
    SCTP_InstanceParameters instanceParameters;

    /* initialize the terminal_ulp variable */
    terminalUlp.dataArriveNotif          = &dataArriveNotif;
    terminalUlp.sendFailureNotif         = &sendFailureNotif;
    terminalUlp.networkStatusChangeNotif = &networkStatusChangeNotif;
    terminalUlp.communicationUpNotif     = &communicationUpNotif;
    terminalUlp.communicationLostNotif   = &communicationLostNotif;
    terminalUlp.communicationErrorNotif  = &communicationErrorNotif;
    terminalUlp.restartNotif             = &restartNotif;
    terminalUlp.shutdownCompleteNotif    = &shutdownCompleteNotif;

    /* handle all command line options */
    getArgs(argc, argv);

    /* set up the "server" */
    sctpInstance = sctp_registerInstance(localPort,
					  MAXIMUM_NUMBER_OF_IN_STREAMS,
					  MAXIMUM_NUMBER_OF_OUT_STREAMS,
					  noOfLocalAddresses, localAddressList,
					  terminalUlp);
    
    /* set the TOS byte */
    sctp_getAssocDefaults(sctpInstance, &instanceParameters);
    instanceParameters.ipTos=tosByte;
    sctp_setAssocDefaults(sctpInstance, &instanceParameters);
    
    associationID = sctp_associate(sctpInstance, MAXIMUM_NUMBER_OF_OUT_STREAMS, destinationAddress, remotePort, NULL);
    
    sctp_registerStdinCallback(&stdinCallback);
    
    /* run the event handler forever */
    while (sctp_eventLoop() >= 0);

    /* this will never be reached */
    return 0;
}




